//
// Binder.cpp
//
// $Id: //poco/Main/Data/SQLite/src/Binder.cpp#5 $
//
// Library: SQLite
// Package: SQLite
// Module:  Binder
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/Data/SQLite/Binder.h"
#include "Poco/Data/SQLite/Utility.h"
#include "Poco/Data/Date.h"
#include "Poco/Data/Time.h"
#include "Poco/Exception.h"
#include "Poco/DateTimeFormatter.h"
#include "Poco/DateTimeFormat.h"
#include <cstdlib>


using Poco::DateTimeFormatter;
using Poco::DateTimeFormat;


namespace Poco {
namespace Data {
namespace SQLite {


Binder::Binder(sqlite3_stmt* pStmt):
	_pStmt(pStmt)
{
}


Binder::~Binder()
{
}


void Binder::bind(std::size_t pos, const Poco::Int32 &val, Direction dir)
{
	int rc = sqlite3_bind_int(_pStmt, (int) pos, val);
	checkReturn(rc);
}


void Binder::bind(std::size_t pos, const Poco::Int64 &val, Direction dir)
{
	int rc = sqlite3_bind_int64(_pStmt, (int) pos, val);
	checkReturn(rc);
}


#ifndef POCO_LONG_IS_64_BIT
void Binder::bind(std::size_t pos, const long &val, Direction dir)
{
	long tmp = static_cast<long>(val);
	int rc = sqlite3_bind_int(_pStmt, (int) pos, tmp);
	checkReturn(rc);
}
#endif


void Binder::bind(std::size_t pos, const double &val, Direction dir)
{
	int rc = sqlite3_bind_double(_pStmt, (int) pos, val);
	checkReturn(rc);
}


void Binder::bind(std::size_t pos, const std::string& val, Direction dir)
{
	int rc = sqlite3_bind_text(_pStmt, (int) pos, val.c_str(), (int) val.size()*sizeof(char), SQLITE_TRANSIENT);
	checkReturn(rc);
}


void Binder::bind(std::size_t pos, const Date& val, Direction dir)
{
	DateTime dt(val.year(), val.month(), val.day());
	std::string str(DateTimeFormatter::format(dt, Utility::SQLITE_DATE_FORMAT));
	bind(pos, str, dir);
}


void Binder::bind(std::size_t pos, const Time& val, Direction dir)
{
	DateTime dt;
	dt.assign(dt.year(), dt.month(), dt.day(), val.hour(), val.minute(), val.second());
	std::string str(DateTimeFormatter::format(dt, Utility::SQLITE_TIME_FORMAT));
	bind(pos, str, dir);
}


void Binder::bind(std::size_t pos, const DateTime& val, Direction dir)
{
	std::string dt(DateTimeFormatter::format(val, DateTimeFormat::ISO8601_FORMAT));
	bind(pos, dt, dir);
}


void Binder::bind(std::size_t pos, const NullData&, Direction)
{
	sqlite3_bind_null(_pStmt, pos);
}


void Binder::checkReturn(int rc)
{
	if (rc != SQLITE_OK)
		Utility::throwException(rc);
}


} } } // namespace Poco::Data::SQLite
