//----------------------------------------------------------------------------
//	Copyright (C) 2002 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file XmlTrace.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/07/15

	\brief Declaration of XmlTrace

	\sa XmlTrace.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/

#ifdef _MSC_VER
#pragma warning(disable : 4786)
#endif

#ifndef ODEMX_XMLTRACE_INCLUDED
#define ODEMX_XMLTRACE_INCLUDED

#include <odemx/util/Trace.h>
#include <odemx/util/HtmlGenerator.h>
#include <odemx/base/Simulation.h>
#include <odemx/base/TraceFilter.h>

#include <iostream>
#include <string>
//#include <set>


namespace odemx {


	/** \class FileInfo
		
		\ingroup util
		
		\author Ronald Kluth
	
		\brief store information about the marks contained in xml files 
	
		\since 2.0
	*/
	struct FileInfo {
		std::string name;
		SimTime startTime;
		SimTime endTime;
	};


	/** \var typedef std::vector< std::string > FileList;
	 
		\brief vector of strings representing file names
	*/
	typedef std::vector< FileInfo > FileList;


	/** \class XmlTrace
		
		\ingroup util

		\author Ronald Kluth

		\brief Declaration of odemx::XmlTrace

		\sa Trace, TraceProducer, TraceConsumer

		<!-- [\note {notes}]* -->

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class XmlTrace : public virtual TraceConsumer,
					 public TraceFilter,
					 public HtmlGenerator {
		// Interface
	public:
		/// construction
		XmlTrace(Simulation* s, std::string fileName = "", unsigned int limit = 2500);
		/// \overload
		XmlTrace(Simulation* s, std::ostream& os);
		/// destruction
		virtual ~XmlTrace();
		
		// Implementation
	private:
		
		/**
			\name Simple trace marks

			Simple trace marks are handled by the following methods.

			@{
			
		*/
		virtual void mark(const TraceProducer* sender, const MarkType* m, const char* comment = 0);

		// Marks that include information about producer associations:
		virtual void mark(const TraceProducer* sender, const MarkType* m, const TraceProducer* partner, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, const TraceProducer* firstPartner, const TraceProducer* secondPartner, const char* comment = 0);

		// Marks that include information about producer attributes:
		virtual void mark(const TraceProducer* sender, const MarkType* m, bool newValue, bool oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, char newValue, char oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, short newValue, short oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, int newValue, int oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, long newValue, long oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, float newValue, float oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, double newValue, double oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, const char* newValue, const char* oldValue, const char* comment = 0);
		
		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned char newValue, unsigned char oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned short newValue, unsigned short oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned int newValue, unsigned int oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned long newValue, unsigned long oldValue, const char* comment = 0);
		//@}

		/**
			\name Complex trace marks

			Complex trace marks are transmitted with the folowing methodes.
			A complex trace mark can be interpreted as a tree of tag-value pairs.

			@{
		*/
		/// beginMark indicates the construction of a new mark
		virtual void beginMark(const TraceProducer* sender, const MarkType* m);
		/// endMarks signals the end of construction
		virtual void endMark(const char* comment = 0);

		// These methods are called to construct complex xml marks.
		virtual void beginTag(Tag t);
		virtual void endTag(Tag t);

		virtual void addTag(Tag t, bool value);
		virtual void addTag(Tag t, char value);
		virtual void addTag(Tag t, short value);
		virtual void addTag(Tag t, int value);
		virtual void addTag(Tag t, long value);
		virtual void addTag(Tag t, float value);
		virtual void addTag(Tag t, double value);
		virtual void addTag(Tag t, const char* value);
		virtual void addTag(Tag t, const TraceProducer* value);

		virtual void addTag(Tag t, unsigned char value);
		virtual void addTag(Tag t, unsigned short value);
		virtual void addTag(Tag t, unsigned int value);
		virtual void addTag(Tag t, unsigned long value);
		//@}
		
		/**
			\name Trace Control
			@{
		*/
		virtual void startTrace(); ///< XmlTrace initializes a new trace
		virtual void stopTrace(); ///< XmlTrace cleans up
		virtual void pauseTrace(); ///< XmlTrace starts a break
		virtual void continueTrace(); ///< XmlTrace ends a break
		//@}

	private:
		Simulation* sim; ///< pointer to simulation
		bool manageOut; ///< manage the outstream
		bool pause; ///< in pause mode
		bool active; ///< is active
		bool filterOut; ///< is filtering out
		std::ostream* out; ///< outstream for XML text
		std::string xmlFileName; ///< file name base for XML files
		FileInfo currentFile; ///< store name, start and end time
		FileList xmlFiles; ///< stores all xml file names
		
		unsigned int xmlMarkCount; ///< count marks to estimate file size
		unsigned int xmlMarkLimit; ///< maximum mark count per xml file
		SimTime limitReachTime; ///< stores the SimTime when mark limit was reached
		
		
		/**
		 \name Help Procedures
		 @{
		*/

		void openXmlFile();
		void closeXmlFile();
		FileList& getXmlFileList();
		void resetCurrentFile();
		void outputXmlFileList();
		
		void checkMarkCount();
		bool limitReached();
		
		void beginTraceTree();
		void endTraceTree();
		void openXmlTag(const char* n); ///< open XML tag
		void closeXmlTag(const char* n); ///< close XML tag
		void openTag(Tag t); ///< open a trace tag
		void closeTag(); ///< close a trace tag

		/// write value \p v inside xml tag \p n
		template <class T>
		void writeValue(T v, const char* n) {
			if (skipOutput())
				return;

			openXmlTag(n);
			// write value
			*out << v;
			closeXmlTag(n);
		}

		/// write value \p v inside trace tag \p t
		template <class T>
		void writeValue(T v, Tag t) {
			if (skipOutput())
				return;

			openTag(t);
			// write value
			*out << v;
			closeTag();
		}

		/// write bool value \p v inside xml tag \p n
		void writeValue(bool v, const char* n);
		/// write TraceProducer \p tp inside xml tag \p n
		void writeValue(const TraceProducer* tp, const char* n);

		/// write bool value \p v inside trace tag \p t
		void writeValue(bool v, Tag t);
		/// write TraceProducer \p tp inside trace tag \p t
		void writeValue(const TraceProducer* tp, Tag t);
		/// print process state as string
		const char* printState( int val );
		
		/// write new value old value pair
		template <class T>
		void writeNvOv(const TraceProducer* sender, const MarkType* m, T newValue, T oldValue, const char* comment/*= 0*/) {
			if (skipOutput())
				return;

			beginMarkTree(sender, m);
		
			*out << "<tag name=\"new value\">";
			if ( m -> getId() == 1030 )
				*out << printState( (int)newValue );
			else
				*out << newValue;
			//*out << newValue;
			*out << "</tag>" << std::endl;
			
			*out << "<tag name=\"old value\">";
			if ( m -> getId() == 1030 )
				*out << printState( (int)oldValue );
			else
				*out << oldValue;
			//*out << oldValue;
			*out << "</tag>" << std::endl;
			
			endMarkTree(comment);
		}

		/// begin complex mark
		void beginMarkTree(const TraceProducer* sender, const MarkType* m);
		/// end complex mark
		void endMarkTree(const char* comment = 0);

		/// is current mark filtered out
		bool skipOutput() {return pause || !hasPassed();}

		/// \overload
		const char* translate(const char* s);
		/// translate string to xml string
		const char* translate(const std::string& s);
		//@}
	};
}

#endif
