//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Trace.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/11

	\brief Trace mechanism implementation

	\sa Trace.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifdef _MSC_VER
#pragma warning(disable : 4530)
#endif

#include "Trace.h"
#include "ErrorHandling.h"

#include <algorithm>
#include <cstring>

using namespace std;
using namespace odemx;

Trace::Trace() : state(CREATED), openMark(false) {
}

Trace::~Trace() {
	if (state==RUNNING || state==PAUSED)
		stopTrace();
}

//
// TraceConsumer management
//
void Trace::addConsumer(TraceConsumer* c) {
	assert(c!=0);

	cons.push_back(c);

	if (state==RUNNING || state==PAUSED)
		c->startTrace();
	
	if (state==PAUSED)
		c->pauseTrace();
}

void Trace::removeConsumer(TraceConsumer* c) {
	assert(c!=0);
	assert(find(cons.begin(), cons.end(), c)!=cons.end());

	if (state==RUNNING || state==PAUSED)
		c->stopTrace();

	cons.remove(c);
}

//
// Trace control
//
void Trace::startTrace() {
	list<TraceConsumer*>::iterator i;

	if (state!=CREATED)
		return;

	state = RUNNING;

	for (i=cons.begin(); i!=cons.end(); ++i)
		(*i)->startTrace();
}

void Trace::stopTrace() {
	list<TraceConsumer*>::iterator i;

	if (state==PAUSED)
		continueTrace();
		
	if (state!=RUNNING)
		return;

	if (openMark)
		endMark();

	state = FINISHED;

	for (i=cons.begin(); i!=cons.end(); ++i)
		(*i)->stopTrace();
}

void Trace::pauseTrace() {
	list<TraceConsumer*>::iterator i;

	if (state!=RUNNING)
		return;

	if (openMark)
		endMark();

	state = PAUSED;

	for (i=cons.begin(); i!=cons.end(); ++i)
		(*i)->pauseTrace();
}

void Trace::continueTrace() {
	list<TraceConsumer*>::iterator i;

	if (state!=PAUSED)
		return;

	state = RUNNING;

	for (i=cons.begin(); i!=cons.end(); ++i)
		(*i)->continueTrace();
}

//
// Simple trace marks are reported by the following methods.
//
void Trace::mark(const TraceProducer* sender, MarkType m, const char* comment) {
	if (!sendMarks())
		return;

	list<TraceConsumer*>::iterator i;

	for (i=cons.begin(); i!=cons.end(); ++i)
		(*i)->mark(sender, &m, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, const TraceProducer* partner, const char* comment) {
	if (!sendMarks())
		return;

	list<TraceConsumer*>::iterator i;

	for (i=cons.begin(); i!=cons.end(); ++i)
		(*i)->mark(sender, &m, partner, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, const TraceProducer* firstPartner, const TraceProducer* secondPartner, const char* comment) {
	if (!sendMarks())
		return;

	list<TraceConsumer*>::iterator i;

	for (i=cons.begin(); i!=cons.end(); ++i)
		(*i)->mark(sender, &m, firstPartner, secondPartner, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, bool newValue, bool oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, char newValue, char oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, short newValue, short oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, int newValue, int oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, long newValue, long oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, float newValue, float oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, double newValue, double oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, const char* newValue, const char* oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, unsigned char newValue, unsigned char oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, unsigned short newValue, unsigned short oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, unsigned int newValue, unsigned int oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

void Trace::mark(const TraceProducer* sender, MarkType m, unsigned long newValue, unsigned long oldValue, const char* comment) {
	broadcastMark(sender, m, newValue, oldValue, comment);
}

// 
// Composed trace marks are built with the following methods.
//
void Trace::beginMark(const TraceProducer* sender, MarkType m) {
	if (!sendMarks())
		return;

	if (openMark)
		endMark();

	list<TraceConsumer*>::iterator i;
	for (i=cons.begin(); i!=cons.end(); ++i) \
		(*i)->beginMark(sender, &m);

	openMark=true;
}

void Trace::endMark(const char* comment /*= 0*/) {
	if (!sendMarks())
		return;

	if (!openMark) {
		error("Trace::endMark() is sent without a corresponding beginMark().");
		return;
	}

	list<TraceConsumer*>::iterator i;
	for (i=cons.begin(); i!=cons.end(); ++i) \
		(*i)->endMark(comment);

	openMark=false;
}

void Trace::beginTag(Tag t) {
	if (!sendMarks())
		return;

	if (!openMark) {
		error("Trace::beginTag() is sent outside an open composed mark.");
		return;
	}

	list<TraceConsumer*>::iterator i;
	for (i=cons.begin(); i!=cons.end(); ++i)
		(*i)->beginTag(t);
}

void Trace::endTag(Tag t) {
	if (!sendMarks())
		return;

	if (!openMark) {
		error("Trace::endTag() is sent outside an open composed mark.");
		return;
	}

	list<TraceConsumer*>::iterator i;
	for (i=cons.begin(); i!=cons.end(); ++i)
		(*i)->endTag(t);
}

void Trace::addTag(Tag t, bool value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, char value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, short value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, int value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, long value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, float value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, double value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, const char* value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, const TraceProducer* value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, unsigned char value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, unsigned short value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, unsigned int value) {
	boradcastTagValue(t, value);
}

void Trace::addTag(Tag t, unsigned long value) {
	boradcastTagValue(t, value);
}

// help procedures
template <class T>
inline void Trace::broadcastMark(const TraceProducer* sender, MarkType m, T newValue, T oldValue, const char* comment) {
	if (!sendMarks())
		return;

	list<TraceConsumer*>::iterator i;
	for (i=cons.begin(); i!=cons.end(); ++i)
		(*i)->mark(sender, &m, newValue, oldValue, comment);
}

template <class T>
inline void Trace::boradcastTagValue(Tag t, T value) {
	if (!sendMarks())
		return;

	if (!openMark) {
		error("In Trace a tag is sent outside an open composed mark.");
		return;
	}

	list<TraceConsumer*>::iterator i;
	for (i=cons.begin(); i!=cons.end(); ++i)
		(*i)->addTag(t, value);
}

/**
	\brief operator ==
*/
bool operator ==(const Tag& t1, const Tag& t2) {
	return t1.getId()==t2.getId() && 
			(strcmp(t1.getName(), t2.getName())==0); 
}

const MarkTypeId MarkType::NOID = 0; 
const char* MarkType::NONAME = 0;
const TagId Tag::NOID = 0;
const char* Tag::NONAME = 0;

