//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file StatisticObject.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/05/24

	\brief Interfaces for statistic objects

	\sa StatisticObject.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_STATISTICOBJECT_INCLUDED
#define ODEMX_STATISTICOBJECT_INCLUDED

#include <odemx/util/SimTime.h>
#include <list>

namespace odemx {
	class StatisticManager;

	/** \class StatisticObject

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Abstract base class for statistic generating classes

		<!-- [\note {notes}]* -->

		\sa StatisticManager

		StatisticObject declares a method for resetting statistics.
		This is required to skip misleading start and initialisation
		phases of simulations. It also handles registration of
		statistic generating objects.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class StatisticObject {
	public:
		/**
			\brief Construction
			
			Add statistic object to StatisticManager.
		*/
		StatisticObject(StatisticManager* manager);

		/**
			\brief Destruction
			
			Remove statistic object from StatisticManager.
		*/
		virtual ~StatisticObject();

		/// reset statistics
		virtual void reset();

		/// reset time
		virtual SimTime getResetTime() const {return reset_at;}

		/// get statistic manager
		StatisticManager* getStatisticManager() {return manager;}

		/// change statistic manager
		StatisticManager* setStatisticManager(StatisticManager* nm);

	private:
		StatisticManager* manager;
	
	protected:
		SimTime reset_at;
	};

	/** \class StatisticManager

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Management of StatisticObjects

		<!-- [\note {notes}]* -->

		\sa StatisticObject

		StatisticManager provides a reset method for resetting all 
		registered StatisticObjects. It also provides methods for
		registration.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class StatisticManager {
	public:
		/// Contruction
		StatisticManager();

		/// Destruction
		virtual ~StatisticManager();

		/// reset statistics
		virtual void reset();

		/// register StatisticObject
		virtual void add(StatisticObject* o);

		/// unregister StatisticObject
		virtual void remove(StatisticObject* o);

		/// get SimTime for reset time
		virtual SimTime getTime() {return 0.0;}

		// Implementation
	protected:
		std::list<StatisticObject*> so;
	};
}

#endif

