//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Observable.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/02/11

	\brief Observation scheme

	<!-- [\sa <pair-file>] -->

	Observation is meant as an association scheme between two
	objects, one observed, one observing, that share an individual (observation)
	interface. The observed object reports events and attribute changes
	of some meaning through the interface to the observer, which implements this
	xxxObserver interface.
	The xxxObserver interface is defined along with the class of the observed 
	object, where xxx is replaced with the class name. An Observer is 
	a client to its observed object and managed by the template class Observable.
	A class that support the observation scheme as a type for observed objects
	should use Observable as a base class.

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/
#if defined(_MSC_VER)
#pragma warning(disable : 4250 4786)
#endif

#ifndef ODEMX_OBSERVABLE_INCLUDED
#define ODEMX_OBSERVABLE_INCLUDED

#include <list>
#include <algorithm>

namespace odemx {
	/** \class Observable

		\ingroup util

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Observable provides management of Observer.

		\copydoc Observable.h

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	template <class Observer> class Observable {
		public:
			/**
				\brief Construction

				\param o
					pointer to Observer
			*/
			Observable(Observer* o = 0) {if(o!=0) observer.push_back(o);}
			
			/**
				\brief get Observer
			*/
			const std::list<Observer*>& getObserver() {return observer;}

			/**
				\brief add new Observer
			*/
			void addObserver(Observer* newObserver) {
				typename std::list<Observer*>::iterator i
					=std::find(observer.begin(), observer.end(), newObserver);

				if (i!=observer.end())
					return;

				observer.push_back(newObserver);
			}

			/**
				\brief remove Observer
			*/
			void removeObserver(Observer* ob) {observer.remove(ob);}
			
		private:
			std::list<Observer*> observer; ///< list of pointer to Observer
	};

/**
	\brief Broadcast event

	\ingroup util

	\param ObserverType
		Type of observer (xxxObserver)

	\param Event
		Event handler function without on ( Create(this) -> onCreate(this) )
*/
#define _obsForEach(ObserverType, Event) \
{\
	for(std::list<ObserverType* >::const_iterator i=Observable<ObserverType >::getObserver().begin();\
		i!=Observable<ObserverType >::getObserver().end(); ++i) (*i)->on##Event;\
}

/**
	\brief Broadcast attribute change

	\ingroup util

	\param ObserverType
		Type of observer (xxxObserver)

	\param Attribute
		Changed attribute name

	\param oldValue
		Old value of attribute

	\param newValue
		New value of attribute
*/
#define _obsAForEach(ObserverType, Attribute, oldValue, newValue) \
{\
	for(std::list<ObserverType* >::const_iterator i=Observable<ObserverType >::getObserver().begin();\
		i!=Observable<ObserverType >::getObserver().end(); ++i) (*i)->onChange##Attribute(this, oldValue, newValue);\
}


/**
	\brief Broadcast event with observer template

	\ingroup util

	\param ObserverType
		Type of observer (xxxObserver)

	\param Event
		Event handler function without on ( Create(this) -> onCreate(this) )
*/
#define _obsForEachT(ObserverType, Event) \
{\
	for(typename std::list<ObserverType* >::const_iterator i=Observable<ObserverType >::getObserver().begin();\
		i!=Observable<ObserverType >::getObserver().end(); ++i) (*i)->on##Event;\
}

/**
	\brief Broadcast attribute change with observer template

	\ingroup util

	\param ObserverType
		Type of observer (xxxObserver)

	\param Attribute
		Changed attribute name

	\param oldValue
		Old value of attribute

	\param newValue
		New value of attribute
*/
#define _obsAForEachT(ObserverType, Attribute, oldValue, newValue) \
{\
	for(typename std::list<ObserverType* >::const_iterator i=Observable<ObserverType >::getObserver().begin();\
		i!=Observable<ObserverType >::getObserver().end(); ++i) (*i)->onChange##Attribute(this, oldValue, newValue);\
}



}

#endif

