//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file PortT.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Declaration of odemx::PortT, odemx::PortHeadT and odemx::PortTailT

	\sa PortT.cpp, PortHeadT.cpp, PortTailT.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/



#include <list>
#include <string>
#include <typeinfo>
#include <odemx/base/Simulation.h>
#include <odemx/base/Process.h>
#include <odemx/synchronization/Memo.h>
#include <odemx/util/ErrorHandling.h>
#include <odemx/util/Report.h>
#include <odemx/statistics/Statistics.h>

#include <odemx/Debug.h>

namespace odemx {


	// forward declaration
	template < typename ElementType >
	class PortHeadT;
	template < typename ElementType >
	class PortTailT;
	
	template < typename ElementType >
	class PortHeadObserverT;
	template < typename ElementType >
	class PortTailObserverT;


	/**
		\brief Port modes

		Since ports implement a limited buffer, a behavior needs to be 
		defined when a port is full, but an element is to be inserted,
		or when a port is empty and some process is trying to read from it.
		Modes for port head and corresponding port tail can differ.
	*/
	enum PortMode {
		ERROR_MODE,		///< error when full/empty
		WAITING_MODE,	///< context switch when full/empty
		ZERO_MODE		///< ignore objects when full/empty, return 0
	};



	/** \class PortT

		\ingroup synch

		\author Ronald Kluth
		
		<!-- [\author <author>]* -->

		\brief %Port is an internal class that implements the internal buffer.

		\sa PortHeadT, PortTailT

		This internal class manages the limited buffer that is the base
		for a communications port. It maintains references to its port tail,
		which describes the input interface, and its port head, which is the
		output interface of a port.
		
		\since 2.0
	*/


	template < typename ElementType >
	class PortT: public DefLabeledObject,
				 public StatisticObject,
				 public virtual ReportProducer {
	public:

	// interface
	public:
	
		friend class PortHeadT< ElementType >;
		friend class PortTailT< ElementType >;
	
		/// Construction
		PortT( Simulation* sim, Label l, unsigned int max, 
			   PortHeadT< ElementType >* ph = 0, 
			   PortTailT< ElementType >* pt = 0 );
		/// Destruction	
		virtual ~PortT();


		/**
			\brief report port statistics
			\param r
				the report object which receives the data

			This function is used to send information collected from a port
			to a report object which displays the data.
		*/
		void report( Report* r );

		/// reset statistics
		virtual void reset();
		/// min buffer length
		unsigned int getMinCount() const { return (unsigned int)portAccum.getMin(); }
		/// max buffer length
		unsigned int getMaxCount() const { return (unsigned int)portAccum.getMax(); }
		/// current buffer length
		unsigned int getNowCount() const { return elementList.size(); }
		/// average buffer length
		double getAvgCount() const { return portAccum.getMean();}

	// implementation
	private:

		std::list< ElementType > elementList;	///< internal buffer for PortElements
		unsigned int maxCapacity;				///< Port capacity
		PortHeadT< ElementType >* portHead;		///< corresponding Head interface
		PortTailT< ElementType >* portTail;		///< corresponding Tail interface
		Simulation*	env;						///< Simulation context
		char* elementTypeName;					///< string representation of element type
		
		// statistics
		unsigned int putCalls;
		unsigned int getCalls;
		unsigned int ungetCalls;
		unsigned int headAlerts;
		unsigned int tailAlerts;
		Accum portAccum;

	};


	/** \class PortHeadT

		\ingroup synch

		\author Ronald Kluth
		
		<!-- [\author <author>]* -->

		\brief %PortHeadT provides read access to a port 

		\sa PortTailT, PortT

		This class manages read access to a limited internal buffer.
		It can be run in three different error modes according to
		PortMode. Get on an empty port will either be handled by
		making the reader wait (and alerting him upon availability), 
		reporting an error, or returning zero.  

		\since 2.0
	*/	

	template < typename ElementType >
	class PortHeadT: public Memo,	// is a trace producer
					 public DefLabeledObject,
					 public virtual ReportProducer,
					 public Observable< PortHeadObserverT< ElementType > > {

	// public interface
	public:
	
		/**
			\brief Construction
			\param sim
				pointer to the simulation object
			\param portName
				label of the internal port, this object's label will be "portName_head"
			\param m
				port head's mode, default is WAITING_MODE
			\param max
				maximum number of elements
			\param pho
				initial observer
		*/
		PortHeadT( Simulation* sim, Label portName, 
				  PortMode m = WAITING_MODE, 
				  unsigned int max = 10000,
				  PortHeadObserverT< ElementType >* pho = 0 );

		/// Destruction
		virtual ~PortHeadT ();
	
	
		/**
			\brief Port read access
			\return
				the next port element from the internal buffer

			This function is used to read information from a port. It also
			responds according to the object's PortMode.

			\sa unget()
		*/
		ElementType get();        // provides the first PortElement

		/**
			\brief Port access, put element back at front
			\param newElement
				the element to put back into the buffer
			
			\return
				\p true if the port element was successfully added at the front

			This function is used to put an element back at the front 
			of the internal port.
			
			\sa get()
		*/
		bool unget( ElementType newElement );
		
		/**
			\brief Get corresponding PortTail 
			\return
				corresponding PortTail to this PortHead 

			This function provides access to the tail of a port head.
			If it does not exist yet, it is created.
		*/		
		PortTailT< ElementType >* getTail();	


		/**
			\brief Get maximum capacity of the internal buffer
			\return
				capacity of the port

			This function is used to determine the maximum number of
			elements that can be buffered by the internal port.
			
			\sa setMaxCapacity()
		*/
		unsigned int getMaxCapacity();


		/**
			\brief Change maximum capacity of the port
			\param newMaximum
				the new maximum capacity of the internal port

			This function allows to change the internal buffer limit.
			
			\sa getMaxCapacity()
		*/		
		void setMaxCapacity( unsigned int newMaximum );


		/**
			\brief Get the port head's mode
			\return
				the currently set mode

			This function is used to determine the port head's mode. 
			
			\sa setMode()
		*/		
		PortMode getMode();


		/**
			\brief Change the port head's mode
			\param newMode
				new mode to be set

			This function is used to change the port head's mode. 
			
			\sa getMode()
		*/		
		void setMode( PortMode newMode );


		/**
			\brief Get current number of port elements
			\return
				current number of elements

			This function returns the current number of buffered elements.
		*/
		unsigned int count();


		/**
			\brief Check for availability of an element
			\return
				\p true if the port is not empty

			This function is used to determine whether an element is 
			available from the internal port.
		*/		
		virtual bool available ();


		/**
			\brief Cut this PortHead from its tail
			\return
				new port head

			This function will create a	new internal port, which
			is attached to this head. A new tail for that port must then be
			requested via getTail(). Furthermore, a new port head will 
			be attached to the old internal port. That new head will be 
			returned.
			
			This provides the ability to insert a process into
			the middle of a port connection, for example as a filter.
			
			\sa splice()
		*/
		PortHeadT< ElementType >* cut();


		/**
			\brief Splice two separate ports together
			\param oldTail
				the tail of the port to which this head's buffer will be attached

			This function is used to join two ports together. This head must
			be upstream of, i.e. before, oldTail's port, which this buffer 
			will be joined with. The contents of oldTail's Port is added to this 
			Port. This head, oldTail, and oldTail's Port are deleted. If a
			if the new head's queue becomes non-empty, it is alerted. 

			\sa cut()
		*/
		void splice( PortTailT< ElementType >* oldTail );


		/**
			\brief Report producing capability
			\param r
				Report object that receives accumulated data

			This function is called by a Report object during report 
			generation to provide its data.
		*/
		void report( Report* r ) { internalPort -> report( r ); }

	private:
	
		friend class PortTailT< ElementType >;


		/**
			\brief Alert the first stored process

			This function is designed to wake up suspended processes, 
			i.e. reschedule them at simulation time \p now when they
			are waiting for an empty port to receive an element. Only 
			the first remembered process is alerted by the %PortHead. 
		*/
		virtual void alert();
		
		PortT< ElementType >* internalPort;	///< internal buffer for port elements
		PortMode mode;						///< mode for read requests 
		Simulation* env;					///< simulation context
		
		// ========================= Trace Marks ===============================

		/**
			\brief Get currently running Sched object or simulation
			\return
				pointer to simulation or current Sched object
		*/
		TraceProducer* getPartner();

	public:
		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace PortHead calls.
			A TraceConsumer can use these constants to identify trace
			events sent	by PortHead.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markCreate;
		static const MarkType markDestroy;

		static const MarkType markGet;
		static const MarkType markUnget;
		static const MarkType markCut;
		static const MarkType markSplice;
		static const MarkType markAlert;

		static const MarkType markChangeMode;
		static const MarkType markChangeMaxCapacity;
		//@}

	};



	/** \interface PortHeadObserver

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief Observer for PortHead-specific calls.

		<!-- [\note {notes}]* -->

		\sa PortHead

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	
	template < typename ElementType >
	class PortHeadObserverT: public MemoObserver {
	public:
		virtual ~PortHeadObserverT() {}

		virtual void onCreate(PortHeadT< ElementType >* sender) {} ///< Construction
		virtual void onDestroy(PortHeadT< ElementType >* sender) {} ///< Destruction

		virtual void onGet(PortHeadT< ElementType >* sender) {}
		virtual void onUnget(PortHeadT< ElementType >* sender, ElementType newElement) {}
		virtual void onCut(PortHeadT< ElementType >* sender) {}
		virtual void onSplice(PortHeadT< ElementType >* sender, PortTailT< ElementType >* oldTail) {}
		virtual void onAlert(PortHeadT< ElementType >* sender, Sched* alerted) {}

		/// PortHead mode change
		virtual void onChangeMode(PortHeadT< ElementType >* sender, PortMode oldMode, PortMode newMode) {}
		/// PortHead capacity change
		virtual void onChangeMaxCapacity(PortHeadT< ElementType >* sender, unsigned int oldMax, unsigned int newMax) {}
	};



	/** \class PortTail

		\ingroup synch

		\author Ronald Kluth
		
		<!-- [\author <author>]* -->

		\brief %PortTail provides write access to a port 

		\sa PortHead, Port

		This class manages write access to a limited internal buffer.
		It can be run in three different error modes according to
		PortMode. Put on a full port will either be handled by
		making the sender wait (and alerting him upon availability), 
		reporting an error, or returning zero.  

		\since 2.0
	*/
	template < typename ElementType >
	class PortTailT: public Memo,	// is a trace producer
					 public DefLabeledObject,
					 public virtual ReportProducer,
					 public Observable< PortTailObserverT< ElementType > > {

	// public interface
	public:
	
		/**
			\brief Construction
			\param sim
				pointer to the simulation object
			\param portName
				label of the internal port, this object's label will be "portName_tail"
			\param m
				port tail's mode, default is WAITING_MODE
			\param max
				maximum number of elements, default is 10000
			\param pto
				initial observer
		*/
		PortTailT( Simulation* sim, Label portName, 
				  PortMode m = WAITING_MODE, 
				  unsigned int max = 10000,
				  PortTailObserverT< ElementType >* pto = 0 );
		
		/// Destruction
		virtual ~PortTailT ();
		
		
		/**
			\brief Port write access
			\param newElement
				the element to insert into the port
			\return
				\p 1 if insertion successful, \p 0 in ZERO_MODE, else \p -1

			This function is used to insert an element into a port. It also
			responds according to the object's PortMode.
		*/
		int put( ElementType newElement );
		
		
		/**
			\brief Get corresponding PortHead
			\return
				corresponding PortHead to this PortTail 

			This function provides access to the head of this port tail.
			If it does not exist yet, it is created.
		*/
		PortHeadT< ElementType >* getHead();
		
		
		/**
			\brief Cut this PortTail from its head
			\return
				new port tail

			This function will create a	new internal port, which
			is attached to this tail. A new head for that port must then be
			requested via getHead(). Furthermore, a new port tail will 
			be attached to the old internal port. That new tail will be 
			returned.
			
			This provides the ability to insert a process into
			the middle of a port connection, for example as a filter.
			
			\sa splice()
		*/
		PortTailT< ElementType >* cut();
		
		
		/**
			\brief Splice two separate ports together
			\param oldHead
				the tail of the port to which this head's buffer will be attached

			This function is used to join two ports together. This tail must
			be downstream of, i.e. positioned after, oldHead's port, which 
			this buffer will be joined with. The contents of this Port 
			is added to oldHead's Port. This tail, oldHead, and this Port are 
			deleted. 

			\sa get()
		*/
		void splice( PortHeadT< ElementType >* oldHead );


		/**
			\brief Get maximum capacity of the internal buffer
			\return
				capacity of the port

			This function is used to determine the maximum number of
			elements that can be buffered by the internal port.
			
			\sa setMaxCapacity()
		*/
	 	unsigned int getMaxCapacity();
	 	
	 	
	 	/**
			\brief Change maximum capacity of the port
			\param newMaximum
				the new maximum capacity of the internal port

			This function allows to change the internal buffer limit.
			
			\sa getMaxCapacity()
		*/
		void setMaxCapacity( unsigned int newMaximum );


		/**
			\brief Determine free space of the port
			\return
				number of elements that the port can still take in 

			This function returns the remaining available space
			of the internal buffer.
		*/
		unsigned int getAvailableSpace();


		/**
			\brief Get the port tail's mode
			\return
				the currently set mode

			This function is used to determine the port tail's mode. 
			
			\sa setMode()
		*/		
		PortMode getMode();


		/**
			\brief Change the port tail's mode
			\param newMode
				new mode to be set

			This function is used to change the port tail's mode. 
			
			\sa getMode()
		*/		
		void setMode( PortMode newMode );


		/**
			\brief Check for availability of the buffer
			\return
				\p true if the port is not full

			This function is used to determine whether a port 
			still has room to contain new elements.
		*/
		virtual bool available ();


		/**
			\brief Report producing capability
			\param r
				Report object that receives accumulated data

			This function is called by a Report object during report 
			generation to provide its data.
		*/
		void report( Report* r ) { internalPort -> report( r ); }

	private:

		friend class PortHeadT< ElementType >;
		
		
		/**
			\brief Alert the first stored process

			This function is designed to wake up suspended processes, 
			i.e. reschedule them at simulation time \p now when they
			are waiting for a full port to become available. Only 
			the first remembered process is alerted by the %PortTail. 
		*/
		virtual void alert();
		
		PortT< ElementType >* internalPort;	///< internal buffer of PortElements
		PortMode	mode;	///< mode for read requests 
		Simulation* env;	///< simulation context
	
			// ========================= Trace Marks ===============================

		/**
			\brief Get currently running Sched object or simulation
			\return
				pointer to simulation or current Sched object
		*/
		TraceProducer* getPartner();

	public:
		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace PortHead calls.
			A TraceConsumer can use these constants to identify trace
			events sent	by PortHead.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markCreate;
		static const MarkType markDestroy;

		static const MarkType markPut;
		static const MarkType markCut;
		static const MarkType markSplice;
		static const MarkType markAlert;

		static const MarkType markChangeMode;
		static const MarkType markChangeMaxCapacity;
		//@}

	};



	/** \interface PortTailObserver

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief Observer for PortTail-specific calls.

		<!-- [\note {notes}]* -->

		\sa PortTail

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/

	template < typename ElementType >	
	class PortTailObserverT: public MemoObserver {
	public:
		virtual ~PortTailObserverT() {}

		virtual void onCreate(PortTailT< ElementType >* sender) {} ///< Construction
		virtual void onDestroy(PortTailT< ElementType >* sender) {} ///< Destruction

		virtual void onPut(PortTailT< ElementType >* sender, ElementType newElement) {}
		virtual void onCut(PortTailT< ElementType >* sender) {}
		virtual void onSplice(PortTailT< ElementType >* sender, PortHeadT< ElementType >* oldHead) {}
		virtual void onAlert(PortTailT< ElementType >* sender, Sched* alerted) {}

		/// PortTail mode change
		virtual void onChangeMode(PortTailT< ElementType >* sender, PortMode oldMode, PortMode newMode) {}
		/// PortTail capacity change
		virtual void onChangeMaxCapacity(PortTailT< ElementType >* sender, unsigned int oldMax, unsigned int newMax) {}
	};

	
	/// template class to determine whether \p T is a pointer type	
	template <typename T>
	struct PointerList {
		/// check for pointer type
		static const bool value = false;
		
		/// nothing to delete when elements were not dynamically created 
		static void deleteList( std::list< T >& toDelete ) {}
	};		
	
	/// specialization with \p value true when \p T is a pointer type 
	template <typename T>
	struct PointerList<T*> {
		
		/// check for pointer type
		static const bool value = true;
		
		/// delete port elements because they were dynamically allocated
		static void deleteList( std::list< T* >& toDelete ) {

			warning( "Deleting unused port elements" ); 
	
			typename std::list< T* >::iterator iter;
			
			for ( iter = toDelete.begin(); iter != toDelete.end(); ++iter )
				delete *iter;
		}
	};
	
}

