//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Wait.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/06/06

	\brief Declaration of odemx::Wait

	\sa Wait.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_WAIT_INCLUDED
#define ODEMX_WAIT_INCLUDED

#include <odemx/base/Process.h>
#include <odemx/base/Simulation.h>

#include <list>

namespace odemx {
	/** \class Wait

		\ingroup synch

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Wait for termination of a number of process objects

		\note Wait supports Trace

		<!-- [\sa {references to other classes}]* -->

		%Wait is used to synchronise a process with the termination of
		one ore more partner processes.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Wait : public ProcessObserver,
				 public DefLabeledObject,
				 public virtual TraceProducer
	{
	public:
		/**
			\brief Construction
			\param s
				pointer to Simulation object
			\param l
				label of this object
		*/
		Wait(Simulation* s, Label l);

		/**
			\brief Construction and wait

			This constructor creates a Wait object and waits for
			Process \p oP1 to finish. If the blocked Process is interrupted,
			it is reactivated and the constructor returns. To check whether
			the process was interrupted, use the isInterrupted() method of
			Process.

			\param s
				pointer to Simulation object
			\param l
				label of this object
			\param oP1
				process to wait for
		*/
		Wait(Simulation* s, Label l, Process* oP1);

		/**
			\brief Construction

  			This constructor creates a Wait object and waits for the Process
			objects \p oP1 and (or) \p oP2 to finish. If the blocked Process 
			is interrupted, it is reactivated and the constructor returns.
			To check whether the process was interrupted, use the isInterrupted()
			method of Process.

			\param s
				pointer to Simulation object
			\param l
				label of this object
			\param oP1
				first process to wait for
			\param oP2
				second process to wait for
			\param all
				wait for \p all processes
		*/
		Wait(Simulation* s, Label l, Process* oP1, Process* oP2, bool all=true);

		/**
			\brief Construction

    		This constructor creates a Wait object and waits for the Process
			objects \p oP1, \p oP2 and(or) \p oP3 to finish. If the blocked Process 
			is interrupted, it is reactivated and the constructor returns.
			To check whether the process was interrupted, use the isInterrupted()
			method of Process.

			\param s
				pointer to Simulation object
			\param l
				label of this object
			\param oP1
				first process to wait for
			\param oP2
				second process to wait for
			\param oP3
				third process to wait for
			\param all
				wait for \p all processes
		*/
		Wait(Simulation* s, Label l, Process* oP1, Process* oP2, Process* oP3, bool all=true);

		/**
			\brief Construction

    		This constructor creates a Wait object and waits for the Process
			objects stored in \p oP to finish. If the blocked Process 
			is interrupted, it is reactivated and the constructor returns.
			To check whether the process was interrupted, use the isInterrupted()
			method of Process.

			\param s
				pointer to Simulation object
			\param l
				abel of this object
			\param size
				number of processes in \p oP
			\param oP
				processes to wait for
			\param all
				wait for \p all processes
		*/
		Wait(Simulation* s, Label l, int size, Process* oP[], bool all=true);

		/// Destructor
		virtual ~Wait();

		/// Add process \p p to observed list
		void addProcess(Process* p);

		/// Remove process \p p from observed list
		void removeProcess(Process* p);

		/// Get Condition (wait for one or all observed Process objects to finish)
		bool getCondition() const {return waitForAll;}

		/// Set Condition (wait for one or all observed Process objects to finish)
		void setCondition(bool all=true);

		/**
			\brief Wait for (one) all observed process to finish

			Wait for one or all observed processes to finish. If the blocked
			Process is interrupted int is reactivated and wait() returns false.
			Otherwise wait() returns true.
		*/
		bool wait();

		/// Pointer to trace
		virtual Trace* getTrace() const {return env;}

		/**
			\name Wait specific marks
			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markCreate;
		static const MarkType markDestroy;

		static const MarkType markWait;
		static const MarkType markContinue;

		static const TagId baseTagId;
		//@}

	public:
		// ProcessObserver functions
		virtual void onDestroy(Process* sender);
		virtual void onChangeState(Process* sender, Process::State oldState, Process::State newState);

	private:
		// Implementation		
		Simulation* env; ///< simulation
		Process* waitP; ///< blocked process
		std::list<Process*> observed; ///< observed processes
		bool waitForAll; ///< condition

		void initObservedList(int size, Process* oP[]); ///< init observed list
		bool checkObserved(); ///< check condition
	};
}

#endif

