//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Res.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/21

	\brief Implementation of classes in Res.h

	\sa Res.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <odemx/synchronization/Res.h>
#include <odemx/base/InSort.h>
#include <odemx/util/ErrorHandling.h>

#include <string>
#include <cassert>
#include <algorithm>

using namespace std;
using namespace odemx;

Res::Res(Simulation* s, Label l, unsigned int startTokenNumber,
		 unsigned int maxTokenNumber, ResObserver* o/* = 0*/)
 : 
   	Observable<ResObserver>(o), 
   	StatisticObject(s), 
	env(s), 
	tokenNumber(startTokenNumber), 
	maxToken(maxTokenNumber),
   	initToken(startTokenNumber),
	tokenStatistics(s, (string(l) + "_tokenStatistic").c_str()),
	acquireWait(s, (string(l) + "_queue").c_str(), &priOrder)
{
	assert(s!=0);

	// set label
	DefLabeledObject::setLabel(s, l);

	// trace
	getTrace()->mark(this, markCreate);

	// observer
	_obsForEach(ResObserver, Create(this));

	// Init statistics
	reset();
}

Res::~Res() {
	// trace
	getTrace()->mark(this, markDestroy);

	// observer
	_obsForEach(ResObserver, Destroy(this));
}

// Token
unsigned int Res::acquire(unsigned int n) {
	// compute order of service
	acquireWait.inSort(getCurrentProcess());

	if (n>tokenNumber ||
		acquireWait.getTop()!=getCurrentProcess()) {
		// not enough token or not the 
		// first process to serve

		// trace
		traceARSF(markAcquireFail, n);

		// observer
		_obsForEach(ResObserver, AcquireFail(this, n));

		// statistics
		SimTime waitTime=env->getTime();

		// block execution
		while (n>tokenNumber || acquireWait.getTop()!=getCurrentProcess()) {
			getCurrentProcess()->sleep();

			if (getCurrentProcess()->isInterrupted()) {
				acquireWait.remove(getCurrentProcess());
				return 0;
			}			
		}

		// block released
		// statistics		
		sumWaitTime+=env->getTime()-waitTime;
	}
	
	// trace
	getTrace()->mark(this, markChangeTokenNumber, tokenNumber-n, tokenNumber);
	
	// observer
	_obsAForEach(ResObserver, TokenNumber, tokenNumber, tokenNumber-n);

	// acquire token
	tokenNumber -= n;

	// statistics
	users++;
	tokenStatistics.update(tokenNumber);

	// trace
	traceARSF(markAcquireSucceed, n);

	// observer
	_obsForEach(ResObserver, AcquireSucceed(this, n));

	// remove from list
	acquireWait.remove(getCurrentProcess());
	
	// awake next process
	awakeFirst(&acquireWait);

	return n;
}

void Res::release(unsigned int n) {
	if ((tokenNumber+n)>maxToken) {
		// Not enough room left.
		// trace
		traceARSF(markReleaseFail, n);

		// observer
		_obsForEach(ResObserver, ReleaseFail(this, n));

		// Error:
		error("Res: could not release all token.");

		// change token number
		n=maxToken-tokenNumber;
	}

	// trace
	getTrace()->mark(this, markChangeTokenNumber, tokenNumber+n, tokenNumber);
	
	// observer
	_obsAForEach(ResObserver, TokenNumber, tokenNumber, tokenNumber+n);

	// release token
	tokenNumber += n;

	// statistics
	providers++;
	tokenStatistics.update(tokenNumber);

	// trace
	traceARSF(markReleaseSucceed, n);

	// observer
	_obsForEach(ResObserver, ReleaseSucceed(this, n));

	// awake process waiting for release
	awakeFirst(&acquireWait);
}

void Res::control(unsigned int n) {
	// change max token limit	
	maxToken+=n;

	// add token
	release(n);
}

unsigned int Res::unControl(unsigned int n) {
	unsigned int tmp;
	// secure enough token
	tmp=acquire(n);

	// remove token
	maxToken-=tmp;

	return tmp;
}

void Res::reset() {
	// reset time
	StatisticObject::reset();

	// reset queue statistics
	acquireWait.reset();

	// Accum reset and re-init
	tokenStatistics.reset();
	tokenStatistics.update(getTokenNumber());

	users=providers=0;
	sumWaitTime=0;
}

double Res::getAVWaitTime() const {
	return sumWaitTime/getAcquires();
}

void Res::report(Report* r) {
	assert(r);
	acquireWait.report(r);

	static const char* labels[] = {"Name", "Reset at", "Queue", "Acquires", "Releases",
								   "Init number of token", "Limit number of token",
								   "Min number of token", "Max number of token", "Now number of token",
								   "Avg number of token", "Avg waiting time"};
	static const ColumnType types[] = {STRING, REAL, STRING, INTEGER, INTEGER,
									   INTEGER, INTEGER,
									   INTEGER, INTEGER, INTEGER,
									   REAL, REAL};
	static utTableDef def(sizeof(labels)/sizeof(char*), labels, types);

	Table* t=r->createTable("Res Statistics", &def);
	assert(t);

	*t << getLabel() << getResetTime() << acquireWait.getLabel() << getAcquires() << getReleases(); 
	*t << getInitial() << getTokenLimit() << getMin() << getMax() << getTokenNumber();
	*t << getAVFreeToken() << getAVWaitTime();
}

void Res::traceARSF(MarkType m, unsigned int n) {
	getTrace()->beginMark(this, m);
	getTrace()->addTag(tagCurrent, getCurrentProcess());
	getTrace()->addTag(tagTokenNumber, n);
	getTrace()->endMark();
}

const MarkTypeId Res::baseMarkId = 1000;

const MarkType Res::markCreate = MarkType("create", baseMarkId+1, typeid(Res));
const MarkType Res::markDestroy = MarkType("destroy", baseMarkId+2, typeid(Res));

const MarkType Res::markAcquireFail = MarkType("acquireFail", baseMarkId+10, typeid(Res));
const MarkType Res::markAcquireSucceed = MarkType("acquireSucceed", baseMarkId+11, typeid(Res));
const MarkType Res::markReleaseFail = MarkType("releaseFail", baseMarkId+12, typeid(Res));
const MarkType Res::markReleaseSucceed = MarkType("releaseSucceed", baseMarkId+13, typeid(Res));

const MarkType Res::markChangeTokenNumber = MarkType("changeTokenNumber", baseMarkId+20, typeid(Res));

const TagId Res::baseTagId = 1000;

const Tag Res::tagCurrent = Tag(baseTagId+1);
const Tag Res::tagTokenNumber = Tag(baseTagId+2);

