//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Memo.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Declaration of odemx::Memo

	\sa Memo.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/


#ifndef ODEMX_MEMO_INCLUDED
#define ODEMX_MEMO_INCLUDED

#include <vector>
#include <list>
#include <odemx/base/Sched.h>

namespace odemx {

	// forward declaration
	class MemoObserver;

	/** \class Memo

		\ingroup synch

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %Memo is the base class for all structures that need to remember processes.

		\sa Process, Timer, PortHead, PortTail

		%Memo is an abstract base class for all user defined structures in a model 
		that need to store a list of suspended processes or other schedulable 
		objects in order to wake them when a certain event occurs. 
		This can be the case for timers, but also
		for structures like buffered communication ports. %Memo offers a means
		to notify suspended processes of the availability of a resource and 
		reschedule them at that point in simulation time when a resource is 
		available or a timeout occurs. The recommended usage with processes is
		via their \p wait() function.
		
		(par Example:)
		(include Base1.cpp)

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/

	class Memo: public Observable<MemoObserver>, 
				public virtual TraceProducer {

	// types
	public:
	
	/**
		\brief Memo types

		%Memo-derivations can be different types of structures. When 
		a pointer to a %Memo object is returned, it can be useful to have 
		a means of distinguishing them simply by asking for their
		type using \p getMemoType(). Currently only timers and ports
		are supported by the library.
	*/
	enum MemoType {
		TIMER,
		PORTHEAD,
		PORTTAIL
	};

	// public interface
	protected:

		friend class Process;

		/**
			\brief Construction
			\param sim
				pointer to the simulation object
			\param t
				type of this object
			\param mo
				initial observer of this object
		*/
		Memo( Simulation* sim, MemoType t, MemoObserver* mo = 0 );

		/// Destruction
		virtual ~Memo();
		
		/**
			\brief Process interaction
		
			These functions can be used by processes to interact with
			%Memo objects.
			
			@{
		*/
		/**
			\brief Remember a Sched object
			\param newObject
				Sched object to be saved
			\return
				\p true when successfully stored, \p false if \p newObject already stored

			This function is used to add processes or, more generally, 
			schedulable objects to the \p memoryList.

			\sa forget()
		*/
		bool remember( Sched* newObject );	// add Process to object's Memory
	

		/**
			\brief Remove a Sched object
			\param rememberedObject
				Sched object to be removed
			\return
				\p true when successfully removed, \p false if \p rememberedObject not stored

			This function is used to remove processes or, more generally, 
			schedulable objects from the \p memoryList.

			\sa remember()
		*/
		bool forget( Sched* rememberedObject );     // remove Process from object's Memory

		/**
			\brief Remove all Sched objects

			This function is used to remove all processes or, more generally, 
			schedulable objects from the \p memoryList.

			\sa remember(), forget()
		*/
		void eraseMemory();     // remove Process from object's Memory
		//@}
	
		/**
			\name Status information

			These functions return information about the Memo object

			@{
		*/
		/**
			\brief Check availability of this Memo object

			This function is required to be implemented by all %Memo-derived
			classes. It is used to	check the availability of a 
			resource before suspending the requesting process.  

			\sa Process::wait(), PortHead, PortTail
		*/
		virtual bool available() = 0;


		/**
			\brief Check type of this Memo object

			The MemoType is useful when a Memo pointer is returned
			but, depending on the type of this object, different 
			reactions are necessary.

			\sa Process::wait()
		*/
		MemoType getMemoType();
		//@}

		/**
			\brief Alert, i.e. reschedule, all stored processes

			By default, this function is designed to wake up suspended 
			or newly created processes, i.e. reschedule them at simulation 
			time \p now. The function can be redefined in derived classes 
			to suit different needs.
		*/
		virtual void alert();

		/**
			\brief Get pointer to Trace
			\return
				pointer to Trace object
		*/
		virtual Trace* getTrace() const { return env; }

	public:
		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace Memo events.
			A TraceConsumer can use these constants to identify trace
			events sent	by Memo.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markRemember;
		static const MarkType markForget;
		static const MarkType markAlert;

		static const TagId baseTagId;

		static const Tag tagPartner;
		//@}
		
	protected:
		std::list<Sched*> memoryList; ///< Sched object storage 

	private:
		Simulation* env; ///< simulation context
		MemoType type;   ///< this object's memo type
		
	private:
		void traceAlert(); ///< send alert mark with memory list
	};

	/// Provided for convenience
	typedef std::vector<Memo*> MemoVector;


	/** \interface MemoObserver

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief Observer for Memo-specific calls.

		<!-- [\note {notes}]* -->

		\sa Timer, PortHead, PortTail

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	
	class MemoObserver {
	public:
		virtual ~MemoObserver() {}

		virtual void onCreate(Memo* sender) {} ///< Construction
		virtual void onDestroy(Memo* sender) {} ///< Destruction

		virtual void onRemember(Memo* sender, Sched* s) {} ///< Remember \p s 
		virtual void onForget(Memo* sender, Sched* s) {} ///< Forget \p s
		virtual void onAlert(Memo* sender) {} ///< Alert all processes
	};
}

#endif /*ODEMX_MEMO_INCLUDED*/

