//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Statistics.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/05/20

	\brief Declaration of statistic-classes

	\sa Statistics.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_STATISTICS_INCLUDED
#define ODEMX_STATISTICS_INCLUDED

#include <vector>

#include <odemx/util/Report.h>
#include <odemx/util/LabeledObject.h>
#include <odemx/util/StatisticObject.h>
#include <odemx/base/Simulation.h>

namespace odemx {
	/**
		\defgroup statistics Statistic

		A group of classes which provide tools for statistical analysis of
		simulations.
	*/

	/** \class Tab

		\ingroup statistics

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Base-class for statistics

		\note !Not! Tab from ODEM

		<!-- [\sa {references to other classes}]* -->

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Tab : public DefLabeledObject,
				public StatisticObject,
				public virtual ReportProducer
	{
	public:
		Tab (Simulation* s, Label label="");
		virtual ~Tab();

	protected:
		void update() {obs++;}
		virtual void reset() {obs=0; StatisticObject::reset();}

		unsigned int getObs() const {return obs;}

	protected:
		Simulation* env;
		unsigned int obs;
		SimTime resetAt;

	};


	/** \class Count

		\ingroup statistics

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Counter

		\note Count from ODEM

		\note Count supports Report

		<!-- [\sa {references to other classes}]* -->

		Count is counting integers. It can be used also to
		calculate an integer sum of all provided numbers.
		To increase the counter use update().

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Count : public Tab {
	public:
		/// Construction
		Count (Simulation* s, Label title="");
		/// Destruction
		virtual ~Count();

		/// Update counter
		void update (int v=1) {
			Tab::update();
			value += v;
		};

		/// Reset counter and statistics
		virtual void reset() {
			Tab::reset();
			value = 0;
		}

		/// Get current counter value
		int getValue() {return value;}

		/// Report generation
		virtual void report(Report* r);

	private:
		int value;
	};


	/** \class Sum

		\ingroup statistics

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Sum

		\note Sum from ODEM

		\note Sum supports Report

		<!-- [\sa {references to other classes}]* -->

		Sum is used to compute 'double' sums. Use update()
		to change the sum.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class  Sum : public Tab {
	public:
		/// Construction
		Sum (Simulation* s, Label title="");
		/// Destruction
		virtual ~Sum();

		/// Update sum
		void update (double v=1.0) {
			Tab::update();
			value += v;
		};

		/// Reset sum and statistics
		virtual void reset() {
			Tab::reset();
			value = 0.0;
		}

		/// Get current value
		double getValue() {return value;}

		/// Report generation
		virtual void report(Report* r);

	private:
		double value;
	};


	/** \class Tally

		\ingroup statistics

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Compute statistics about provided data

		\note Tally from ODEM

		\note Tally supports Report

		<!-- [\sa {references to other classes}]* -->

		Tally computes minimum, maximum, mean-value,
		divergence and number of samples (size)
		independent from simulation time (see Accum).


		A value of 1 for 4 time units together with a value
		of 0 for 1 time unit will result in a mean value of
		(1+0)/2 = 0.5.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Tally : public Tab {
	public:
		/// Construction
		Tally (Simulation* s, Label title="");
		/// Destruction
		virtual ~Tally();

		/// Update statistics
		virtual void update (double v);
		/// Reset statistics
		virtual void reset();

		/// Get number of samples
		unsigned int getSize() {return getObs();}
		/// Get minimum
		double getMin() {return min;}
		/// Get maximum
		double getMax() {return max;}
		/// Get average
		double getMean() {return getObs() ? sum/getObs() : 0.0;}
		/// Get divergence
		double getDivergence();

		/// Report generation
		virtual void report(Report* r);

	protected:
		double sum, sumsq, min, max;
	};


	/** \class Accum

		\ingroup statistics

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Compute statistics about provided data

		\note Accum from ODEM
		\note Accum supports Report

		<!-- [\sa {references to other classes}]* -->

		Accum computes minimum, maximum, mean-value,
		divergence and number of samples (size) according to
		the simulation time (see Tally).

		Time weighted averages are computed according two
		different approximation stategies:

		step: constant values between samples

			A value of 1 hold for 4 time units together with a
			value of 0 hold for 1 time unit will result in a mean
			value of (1*4/2 + 0*1)/5 = 0.1 .


		linear: linear growing/decreasing values between samples

			A value of 1 hold for 4 time units together with a
			value of 0 hold for 1 time unit will result in a mean
			value of (1*4 + 0*1)/5 = 0.2 .

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Accum : public Tab {
	public:
		enum Approx {stepwise, linear};
		/// Construction
		Accum (Simulation* s, Label title="", Approx app = stepwise);
		/// Destruction
		virtual ~Accum();

		/// Update statistics
		virtual void update (double v);
		/// Reset statistics
		virtual void reset();

		/// Get number of samples
		unsigned int getSize() const {return getObs();}
		/// Get minimum
		double getMin() const {return min;}
		/// Get maximum
		double getMax() const {return max;}
		/// Get average (over time)
		double getMean() const;
		/// Get divergence
		double getDivergence() const;

		/// Report generation
		virtual void report(Report* r);

	protected:
		double sumt, sumsqt, min, max, lasttime, lastv;
		Approx approx;
	};

	/** \class Histo

		\ingroup statistics

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Statistical analysis plus histogram

		\note Histo from ODEM
		\note Histo supports Report

		<!-- [\sa {references to other classes}]* -->

		Histo computes a statistical analysis like Tally
		and also generates a histogram.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Histo : public Tally {
	public:
		/**
			\brief Construction
			\param s
				pointer to Simulation object
			\param title
				label of this object
			\param low
				lowest border
			\param up
				highest border
			\param n
				number of categories
		*/
		Histo(Simulation* s, Label title, double low, double up, int n);
		/// Destruction
		virtual ~Histo();

		/// Update statistics
		virtual void update(double v);
		/// Reset statistics
		virtual void reset();

		/// Get Tally-statistics
		const Tally* getTally() {return reinterpret_cast<const Tally*>(this);}
		/// Get histogram data
		const std::vector<int>& getData() {return data;}

		/// Get highest value
		int maximumelem();

		/// Report generation
		virtual void report(Report* r);

	protected:
		double lower, upper;
		int ncells;
		std::vector<int> data;
		int limit;
		double width;
	};


	/** \class Regress

		\ingroup statistics

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Regress analysis

		\note Regress from ODEM
		\note Regress supports Report

		<!-- [\sa {references to other classes}]* -->

		Regress computes the linear correlation between
		provided data X an Y.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Regress : public Tab {
	public:
		/// Construction
		Regress(Simulation* s, Label title);
		/// Destruction
		virtual ~Regress();

		/// Update statistics
		void update(double vx, double vy);
		/// Reset statistics
		virtual void reset();

		/// Has sufficient data for a regress analysis?
		bool hasSufficientData() {return getObs()>5;}

		double getXBar();
		double getYBar();

		double getDx();
		double getDy();

		double getV1(); // RES.ST.DEV
		double getV2(); // EST.REG.COEFF
		double getV3(); // INTERCEPT
		double getV4(); // ST.DEV.REG.COEFF
		double getV5(); // CORR.COEFF

		/// Report generation
		virtual void report(Report* r);

	protected:
		double x,y,xx,xy,yy;
	};
}

#endif

