//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Dist.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/05/09

	\brief Declaration of random number generator base class

	\sa Dist.cpp

	Implementation of a random number generator taken from ODEM.

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_DIST_INCLUDED
#define ODEMX_DIST_INCLUDED

#include <odemx/util/Report.h>
#include <odemx/util/LabeledObject.h>
#include <odemx/util/StatisticObject.h>

namespace odemx {
	extern const unsigned long zyqmodulo;

	/** \class DistContext

		\ingroup random

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %DistContext stores global data for Dist.

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		The random number generator (RNG) Dist does need some 'global'
		data for generating independent random numbers. These data are stored
		in %DistContext.

		\warning The RNG's in one DistContext are independent
		from one another (within the limits of the RNG in Dist). Those RNG's
		in different DistContexts are NOT independent. Without changing the seed manually
		one will get the same sequence of random numbers in every DistContext!

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class DistContext : public virtual LabelScope,
						public virtual StatisticManager
	{
	public:
		/**
			\brief Construction
		*/
		DistContext ();
		/// Destruction
		virtual ~DistContext();

		/**
			\brief Set start seed for new RNG's
			\param n
				new seed

			The new seed will be used for the initialisation
			of new RNG's in this context. Choosing a good seed
			is not trivial. The one used as default is known
			to be sufficient for simulations.

			\warning
				The quality of the random numbers
				depends on the selected seed!
		*/
		void setSeed(int n = 0);

	protected:
		friend class Dist;

		/**
			\brief Get current start seed

			This function returns the current seed without
			generating a new seed value. The returned value
			was used for the last created RNG in this context.

			\sa getNextSeed()
		*/
		unsigned long getSeed();

		/**
			\brief Get start seed for a new RNG

			This function returns a new seed. The
			seed is generated by a random number generator
			different from the one used for ordinary random
			numbers. That is why the created random number
			generators in one %DistContext are independent from
			one another.
		*/
		unsigned long getNextSeed();

	private:
		unsigned long zyqseed;

		/**
			\brief generate next start seed
		*/
		void zyq_next_seed();
	};

	/** \class Dist

		\ingroup random

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Linear Random Number Generator

		\note LRNG from ODEM

		<!-- [\sa {references to other classes}]* -->

		%Dist is the base class for all random number generators of
		ODEMx. It provides an implementation of an integer random
		number generator using the linear congruencial algorithm.
		This algorithm is known to produce sequences of numbers which
		are sufficiently random for simulations. \n
		This class is not used directly. Instead one of the derived
		classes is used, which provide transformations into sequences
		of random numbers with different distributions.

		\sa Idist and Rdist

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Dist : public DefLabeledObject,
				 public StatisticObject,
				 public virtual ReportProducer
	{
	protected:
		/**
			\brief Construction

			\param label
				Label of the object
			\param c
				pointer to DistContext object
		*/
		Dist(DistContext* c=0, Label label="");
		/// Destruction
		virtual ~Dist();

		/**
			\brief Get report table definition

			All random number generators support Report. Their
			data are reported in one table where each line represents
			one generator. This function is used ODEMx-internall to
			realise the report.
		*/
		TableDefinition* getTableDefinition();

		/**
			\brief Get report table
			\copydoc getTableDefinition()
		*/
		Table* getTable(Report* r);

	public:
		/**
			\brief Set seed for the random number generators
			\param n
				new seed

			The new seed will be used for generating
			new random numbers.

			\warning The quality of the generated random numbers
			depends on the selected seed!
		*/
		virtual void setSeed(int n = 0);

		/**
			\brief Get start seed

			This function returns the start seed of
			the random number generator.
		*/
		unsigned long getSeed();

	protected:
		/**
			\brief Get new random number

			This function returns a new random number. It
			is used by derived classes.
		*/
		double getSample();

	private:
		DistContext* context;
		unsigned long u,ustart;
		unsigned int antithetics;
	};

}

#endif

