//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Dist.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/05/09

	\brief Implementation of random number generator base class

	\sa Dist.h

	Implementation of a random number generator taken from ODEM.

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <odemx/random/Dist.h>
#include <odemx/util/ErrorHandling.h>

#include <cassert>

using namespace odemx;

const unsigned long odemx::zyqmodulo = 67099547;

DistContext::DistContext() :
	zyqseed(907)
{
}

DistContext::~DistContext() {
}

void DistContext::setSeed(int n/*=0*/) {
	if (n<0) n= -n;
	if (n>=(int)zyqmodulo) n=n - ((int)(n/zyqmodulo))*zyqmodulo;
	if (n==0) n=(int)(zyqmodulo/2);
	zyqseed=n;
}

unsigned long DistContext::getSeed() {
	return zyqseed;
}

unsigned long DistContext::getNextSeed() {
	zyq_next_seed();
	return zyqseed;
}

void DistContext::zyq_next_seed() {
	int k;

	k=7; zyqseed= zyqseed*k;
	if (zyqseed >= zyqmodulo)
	    zyqseed = zyqseed - ((int)(zyqseed/zyqmodulo))*zyqmodulo;
	k=13; zyqseed= zyqseed*k;
	if (zyqseed >= zyqmodulo)
	    zyqseed = zyqseed - ((int)(zyqseed/zyqmodulo))*zyqmodulo;
	k=15; zyqseed= zyqseed*k;
	if (zyqseed >= zyqmodulo)
	    zyqseed = zyqseed - ((int)(zyqseed/zyqmodulo))*zyqmodulo;
	k=27; zyqseed= zyqseed*k;
	if (zyqseed >= zyqmodulo)
	    zyqseed = zyqseed - ((int)(zyqseed/zyqmodulo))*zyqmodulo;
}

Dist::Dist(DistContext* c/*=0*/, Label label/*=""*/)
 : StatisticObject(c), context(c)
{
	if (context==0) {
		// error: Dist was created without reference to DistContext.
		fatalError("Dist(); invalid argument: DistContext* c", -1);
	}

	// set label
	DefLabeledObject::setLabel(c, label);

	antithetics = 0;
	u=ustart=c->getNextSeed();
}

Dist::~Dist() {
}

void Dist::setSeed(int n/*=0*/) {
	if (n<0) n= -n;
	if (n>=(int)zyqmodulo) n=n - ((int)(n/zyqmodulo))*zyqmodulo;
	if (n==0) n=(int)(zyqmodulo/2);
	u=ustart=n;
}

unsigned long Dist::getSeed() {
	return ustart;
}

double Dist::getSample() {
	int k;
	double zyqsample, help;

	k=32; u=k*u;
	if( u>=zyqmodulo)  u =  u- ((int)(u/zyqmodulo))*zyqmodulo;
	k=32; u=k*u;
	if( u>=zyqmodulo)  u =  u- ((int)(u/zyqmodulo))*zyqmodulo;
	k=8; u=k*u;
	if( u>=zyqmodulo)  u =  u- ((int)(u/zyqmodulo))*zyqmodulo;

	help=(double)u/(double)zyqmodulo;
	zyqsample= antithetics ? 1.0 -  help : help;
	return zyqsample;
}

class RNGTable : public TableDefinition {
public:
	virtual unsigned long getNumberOfColumns() const {return 8;}
	virtual const char* getLabelOfColumn(unsigned long i) const {
		static const char* labels[8]={"Name", "Reset at", "Type", "Uses", "Seed", "Parameter 1", "Parameter 2", "Parameter 3"};

		if (i>=8)
			return 0;

		return labels[i];
	}

	virtual ColumnType getTypeOfColumn(unsigned long i) const {
		static const ColumnType types[8]={STRING, REAL, STRING, INTEGER, INTEGER, REAL, REAL, REAL};

		if (i>=8)
			return INVALID;

		return types[i];
	}
};

TableDefinition* Dist::getTableDefinition() {
	static RNGTable td;
	return &td;
}

Table* Dist::getTable(Report* r) {
	assert(r!=0);

	Table* t=r->createTable("Random Number Generators", getTableDefinition());
	assert(t!=0);

	*t << getLabel() << getResetTime();

	return t;
}

