#ifndef ODEMX_PROTOCOL_ENTITY_TEMPLATE_INCLUDED
#define ODEMX_PROTOCOL_ENTITY_TEMPLATE_INCLUDED

#include <odemx/base/Process.h>
#include <odemx/synchronization/PortTemplate.h>
#include <odemx/protocol/ProtocolSimulation.h>

namespace odemx {

	class ProtocolLayer;
	class NetNode;
	
	
	// interface that allows for polymorphy
	class ProtocolEntityInterface {
	protected:
		ProtocolEntityInterface() {}
		virtual ~ProtocolEntityInterface() {}
	
	public:
		// call PortTail::put()  
		virtual void putTop( PortElement* e ) = 0;
		virtual void putBottom( PortElement* e ) = 0;		
	};
	
	
	
	template < typename TopPortElement, typename BottomPortElement >
	class ProtocolEntityT : public ProtocolEntityInterface, public Process {
		
	public:
	
		ProtocolEntityT( ProtocolSimulation* sim, ProtocolLayer* layer, NetNode* owner ):  
			Process( sim, "Entity" ),
			node( owner ),
			top( 0 ),
			bottom( 0 )
			{
				// entity labels automatically get their owner and layer added in front
				DefLabeledObject::setLabel( sim, 
											(std::string(owner->getLabel())+" "+\
											 std::string(layer->getLabel())+\
											 " Entity").c_str()\
										  );
				
				topPort = new PortHeadT< TopPortElement >( sim, (string(getLabel())+" Top Port").c_str() );
				
				bottomPort = new PortHeadT< BottomPortElement >( sim, (string(getLabel())+" Bottom Port").c_str() );
			}

		
		virtual ~ProtocolEntityT()
			{
				// port destructor will not remove the port 
				// if there is another process connected to head/tail
				delete topPort;
				delete bottomPort;
			}

		
		virtual void handleUpperLayerInput( TopPortElement topInput ) = 0;
		
		virtual void handleLowerLayerInput( BottomPortElement bottomInput ) = 0;
		
		virtual int main() {

			Memo* theAlerter = 0;
			
			OUT( "\n" << getLabel() << " active" );
				
			while (true) {
				
				// check input ports and wait for message arrival
				theAlerter = wait( getTopPortHead(), getBottomPortHead() );

				OUT( "\n" << getLabel() << " active" );

				if ( theAlerter == getTopPortHead() ) {

					OUT( "## " << getLabel() << " received message from UpperEntity" );
					handleUpperLayerInput( getTop() );
				}
				
				else if ( theAlerter == getBottomPortHead() ) {

					OUT( "## " << getLabel() << " received message from TransmissionMedium" );
					handleLowerLayerInput( getBottom() );
				}

				else if ( theAlerter == 0 ) {
						
					if ( isInterrupted() ) {
						OUT( "## " << getLabel() << " interrupted by " << getInterrupter() -> getLabel() );
					}
					else {
						OUT( "## " << getLabel() << " theAlerter == 0" );
					}
				}
				
				else {
					error( "ProtocolEntityT::main(): cannot identify alerter" );
					break;
				}
				
				theAlerter = 0;
			}
			
			return 0;
		}
		
		// the port elements of a port and the elements for the
		// corresponding next layer must be the same 
		void transmit( BottomPortElement m ) {
			
			if ( (void*)this == (void*)(getNode() -> getBottomLayerEntity()) ) {
				getNode() -> getMedium() -> transmit( (ProtocolMessage*)m );
			}
			else {
				getBottomEntity() -> putTop( (PortElement*)m );
			}
		}
		
		void passUpward( TopPortElement m ) {
			
			if ( (void*)this == (void*)(getNode() -> getTopLayerEntity()) ) {
				getNode() -> output( m );
			}
			else {
				getTopEntity() -> putBottom( (PortElement*)m );
			}
		}

		/************** Port methods **************/

		PortHeadT< TopPortElement >* getTopPortHead() { return topPort; }
		PortTailT< TopPortElement >* getTopPortTail() { return topPort -> getTail(); }
		
		PortHeadT< BottomPortElement >* getBottomPortHead() { return bottomPort; }
		PortTailT< BottomPortElement >* getBottomPortTail() { return bottomPort -> getTail(); }

		// read the ProtocolEntity's ports
		TopPortElement getTop() { return topPort -> get(); }
		BottomPortElement getBottom() { return bottomPort -> get(); }

		// call PortTail::put()  
		void putTop( PortElement* e ) {
			getTopPortTail() -> put( dynamic_cast<TopPortElement>(e) );
		}		

		void putBottom( PortElement* e ) {
			getBottomPortTail() -> put( dynamic_cast<BottomPortElement>(e) );
		}

						
		/************ Setters and getters ****************/
		
		void setTopEntity( ProtocolEntityInterface* e ) { top = e; }
		
		void setBottomEntity( ProtocolEntityInterface* e ) { bottom = e; }
		
		ProtocolEntityInterface* getTopEntity() { return top; }
		
		ProtocolEntityInterface* getBottomEntity() { return bottom; }

		NetNode* getNode() { return node; }
		
	private:
		PortHeadT< TopPortElement >* topPort;
		PortHeadT< BottomPortElement >* bottomPort;
		
		NetNode* node;
		ProtocolEntityInterface* top;
		ProtocolEntityInterface* bottom;
	};
}


#endif /* ODEMX_PROTOCOL_ENTITY_TEMPLATE_INCLUDED */
