//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file TransmissionEvents.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Declaration of odemx::UnicastTransmission, odemx::BroadcastTransmission

	\sa TransmissionEvents.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/



#ifndef ODEMX_TRANSMISSION_EVENT_INCLUDED
#define ODEMX_TRANSMISSION_EVENT_INCLUDED

#include <odemx/base/Event.h>

#include <odemx/Debug.h>

namespace odemx {

	// forward declaration
	class ProtocolSimulation;

	class UnicastTransmissionObserver;
	class BroadcastTransmissionObserver;
	

	/** \class UnicastTransmission

		\ingroup protocol

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %UnicastTransmission is an Event class for directed protocol message transmission 
		
		\sa ProtocolMessage, TransmissionMedium

		UnicastTransmission objects are simulation events scheduled by the
		TransmissionMedium at the transmission time of a given message.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class UnicastTransmission: public Event,
							   public virtual TraceProducer,
							   public Observable<UnicastTransmissionObserver> {	
	public:	
		
		
		/**
			\brief Construction
			\param sim
				pointer to the Simulation object
			\param l
				label of this object
			\param m
				message to be transmitted over the medium
		*/
		UnicastTransmission( ProtocolSimulation* sim, Label l, 
							 ProtocolMessage* m, UnicastTransmissionObserver* o = 0 );

		
		/**
			\brief transmission of the given message

			This function implements the action represented by the Event.
			In this case, the next hop is given with the message and
			and it is simply forwarded to that node at the specified
			transmission time. Transmission failure for this case is 
			dealt with by TransmissionMedium.
		*/
		virtual void eventAction();

		/// report failed transmission attempt for statistics
		void reportFailure( ProtocolMessage* m, NetNode* sender, NetNode* receiver );
		
		/// report successful transmission attempt for statistics
		void reportSuccess( ProtocolMessage* m, NetNode* sender, NetNode* receiver );

	private:
		
		/// message to be forwarded by this event
		ProtocolMessage* message;

	public:
		
		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace NetTopology calls.
			A TraceConsumer can use these constants to identify  
			trace events sent by NetTopology.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markUnicastSuccess;
		static const MarkType markUnicastFailure;
		
		static const TagId baseTagId;

		static const Tag tagFrom;
		static const Tag tagTo;
		//@}
	};
	
	
	
	/** \interface UnicastTransmissionObserver

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief Observer for UnicastTransmission specific simulation events.

		<!-- [\note {notes}]* -->

		\sa UnicastTransmission, TransmissionMedium

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class UnicastTransmissionObserver: public EventObserver {
		
	public:
		/// Destructor
		virtual ~UnicastTransmissionObserver() {}

		/// Successful unicast transmission
		virtual void onSuccess( UnicastTransmission* sender, ProtocolMessage* m, 
										 NetNode* n1, NetNode* n2 ) {}
		
		/// Bad connection for unicast transmission
		virtual void onFailure( UnicastTransmission* sender, ProtocolMessage* m, 
										 NetNode* n1, NetNode* n2 ) {}
	};


	
	
	/** \class BroadcastTransmission

		\ingroup protocol

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %BroadcastTransmission is an Event class for protocol message transmission 
		
		\sa ProtocolMessage, TransmissionMedium

		BroadcastTransmission objects are simulation events scheduled by the
		TransmissionMedium at the transmission time of a given message, which
		is then forwarded to all neighbors of the sending NetNode.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class BroadcastTransmission: public Event,
								 public virtual TraceProducer,
								 public Observable<BroadcastTransmissionObserver> {
	public:
		
		
		/**
			\brief Construction
			\param sim
				pointer to the Simulation object
			\param l
				label of this object
			\param m
				message to be transmitted over the medium
		*/
		BroadcastTransmission( ProtocolSimulation* sim, Label l, 
							   ProtocolMessage* m, BroadcastTransmissionObserver* o = 0 );

		
		/**
			\brief broadcast transmission of the given message

			This function implements the action represented by the Event.
			In this case, the sender is given with the message and
			and it is forwarded to all nodes connected to the sender.
			This information is gathered from the NetTopology defined for
			a ProtocolSimulation.
		*/
		virtual void eventAction();

		
		/// report failed transmission attempt for statistics
		void reportFailure( ProtocolMessage* m, NetNode* sender, NetNode* receiver );

		/// report successful transmission attempt for statistics
		void reportSuccess( ProtocolMessage* m, NetNode* sender, NetNode* receiver );

		
	private:
		
		/// message to be forwarded by this event
		ProtocolMessage* message;


	public:
		
		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace NetTopology calls.
			A TraceConsumer can use these constants to identify  
			trace events sent by NetTopology.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markBroadcastSuccess;
		static const MarkType markBroadcastFailure;
		
		static const TagId baseTagId;

		static const Tag tagFrom;
		static const Tag tagTo;
		//@}
	};


	/** \interface BroadcastTransmissionObserver

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief Observer for BroadcastTransmission specific simulation events.

		<!-- [\note {notes}]* -->

		\sa BroadcastTransmission, TransmissionMedium

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class BroadcastTransmissionObserver: public EventObserver {
		
	public:
		/// Destructor
		virtual ~BroadcastTransmissionObserver() {}

		/// Successful broadcast transmission
		virtual void onSuccess( BroadcastTransmission* sender, ProtocolMessage* m, 
										 NetNode* n1, NetNode* n2 ) {}
		
		/// Bad connection for broadcast transmission
		virtual void onFailure( BroadcastTransmission* sender, ProtocolMessage* m, 
										 NetNode* n1, NetNode* n2 ) {}
	};

	
}

#endif /* ODEMX_TRANSMISSION_EVENT_INCLUDED */

