//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file ProtocolEntity.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Declaration of odemx::ProtocolEntity

	\sa ProtocolEntity.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/


#ifndef ODEMX_PROTOCOL_ENTITY_INCLUDED
#define ODEMX_PROTOCOL_ENTITY_INCLUDED

#include <odemx/base/Process.h>
#include <odemx/synchronization/PortTemplate.h>
#include <odemx/protocol/ProtocolSimulation.h>

namespace odemx {

	// forward declaration
	class ProtocolLayer;
	class NetNode;
	
	class ProtocolEntityObserver;
	

	/** \class ProtocolEntity

		\ingroup protocol

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %ProtocolEntity is the base class for active layer components
		
		\sa NetNode, ProtocolLayer

		The OSI model describes protocol stacks in terms of layers with 
		different tasks. The active components that execute these tasks 
		are called protocol entities. Each NetNode has exactly one
		ProtocolEntity per layer. All ProtocolEntities belonging to a 
		NetNode are automatically created and linked according 
		to their layers upon construction of that node. That means each 
		node has a pointer to the entity one layer above and one layer below.
		Protocol entities communicate via buffered ports. Usually, an entity
		is in waiting state. Whenever a message arrives at its top or bottom 
		input port, it is alerted to handle the message.  

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class ProtocolEntity : public Process,
						   public Observable<ProtocolEntityObserver> {
		
	public:
	
		/**
			\brief Construction
			\param sim
				pointer to the ProtocolSimulation object
			\param layer
				pointer to the entity's layer
			\param owner
				pointer to the owning NetNode
		*/
		ProtocolEntity( ProtocolSimulation* sim, ProtocolLayer* layer, 
						NetNode* owner, ProtocolEntityObserver* o = 0 );
		
		/// Destruction
		virtual ~ProtocolEntity();

		
		/** \name Entity behavior
		
			These functions are used to implement the behavior of
			a ProtocolEntity.
			
			@{
		*/
	protected:
		
		/**
			\brief definition of an entity's life cycle
				
			This function must be defined because ProtocolEntity is 
			derived from Process. It has a default implementation and 
			defines the life cycle of an entity. Per default, protocol 
			entities simply call \p Process::wait() to wait for input
			from the layer above or below. Upon alert, the corresponding
			handling function is called.
		*/
		virtual int main();
		
	
	public:
		/**
			\brief deal with message from upper layer entity
			\param topInput
				first element from the top port
				
			This function must be defined by all non-abstract derivations of
			ProtocolEntity. It is called whenever the Entity is alerted
			by its top input port and should define a behavior according
			to the message type received.
		*/
		virtual void handleUpperLayerInput( PortElement* topInput ) = 0;
		
		
		/**
			\brief deal with message from lower layer entity
			\param bottomInput
				first element from the bottom port
				
			This function must be defined by all non-abstract derivations of
			ProtocolEntity. It is called whenever the Entity is alerted
			by its bottom input port and should define a behavior according
			to the message type received.
		*/
		virtual void handleLowerLayerInput( PortElement* bottomInput ) = 0;
			
		
		/**
			\brief transmit a message via lower layer
			\param e
				message to be passed to the layer below
				
			This function is used by an entity to pass a message to the 
			entity one layer below. Usually, the element types of an
			entity's bottom port and the lower entity's top port are
			the same since there should only be one type of messages 
			passed between two layers.
			
			In case the ProtocolEntity is on the lowest layer, the
			message is sent via the transmission medium. This way,
			entities in all layers can transmit messages transparently,
			without knowledge of lower layer functions.
		*/
		void transmit( PortElement* e );
		
		
		/**
			\brief pass a message to the layer above
			\param e
				message to be passed to the layer above
				
			This function is the opposite of \p transmit(). Messages
			are passed to the layer above. Usually, the element types of an
			entity's bottom port and the lower entity's top port are
			the same since there should only be one type of messages 
			passed between two layers.
			
			In case the ProtocolEntity is on the top layer, the
			message is handed to the output port of the NetNode.
		*/
		void passUpward( PortElement* e );
		//@}
		
		/**
			\brief get this entity's owner node
			\return
				pointer to the entity's owning NetNode
		*/
		NetNode* getNode();
		
		
		/**
			\brief get the TransmissionMedium
			\return
				pointer to the simulation's TransmissionMedium
		*/
		TransmissionMedium* getTransmissionMedium();
		

	// implementation
	protected:
		
		friend class NetNode;

		/** \name Port access
			
			These functions are provided to get access to the port heads,
			for example as alerter in waiting state, and to read or write 
			elements from and to ports.
			
			@{
		*/
		
		/// get a pointer the entity's top port head
		PortHead* topPortHead();
		
		/// get a pointer the entity's bottom port head
		PortHead* bottomPortHead();
		
		/// read the ProtocolEntity's top port
		PortElement* readTop();
		
		/// read the ProtocolEntity's bottom port
		PortElement* readBottom();

		/// put a message into the top port  
		void putTop( PortElement* e );
		
		/// put a message into the bottom port
		void putBottom( PortElement* e );
		
	private:
		
		/// get a pointer the entity's top port tail
		PortTail* getTopPortTail();
		
		/// get a pointer the entity's bottom port tail
		PortTail* getBottomPortTail();
		//@}
		

		/**
			\brief set the top entity
			\param e
				pointer to the entity of the layer above
		*/
		void setTopEntity( ProtocolEntity* e );
		
		
		/**
			\brief set the bottom entity
			\param e
				pointer to the entity of the layer below
		*/
		void setBottomEntity( ProtocolEntity* e );
		
		/// get a pointer to the entity in the layer above
		ProtocolEntity* getTopEntity();
		
		/// get a pointer to the entity in the layer above
		ProtocolEntity* getBottomEntity();

	private:
		/// top input port
		PortHead* topPort;
		/// bottom input port
		PortHead* bottomPort;
		
		/// owner node
		NetNode* node;
		/// entity above
		ProtocolEntity* top;
		/// entity below
		ProtocolEntity* bottom;
		
	public:
		

		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace NetTopology calls.
			A TraceConsumer can use these constants to identify  
			trace events sent by NetTopology.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markReadTop;
		static const MarkType markReadBottom;
		static const MarkType markTransmit;
		static const MarkType markPassUpward;
		
		/*static const TagId baseTagId;

		static const Tag tagFrom;
		static const Tag tagTo;
		static const Tag tagBoth;
*/
	};
	
	
	
	/** \interface ProtocolEntityObserver

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief Observer for ProtocolEntity specific simulation events.

		<!-- [\note {notes}]* -->

		\sa ProtocolEntity

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	class ProtocolEntityObserver: public ProcessObserver {
		
	public:
		/// Destructor
		virtual ~ProtocolEntityObserver() {}
				
		/// Read the top port
		virtual void onReadTop( ProtocolEntity* sender, PortElement* e ) {}
		/// Read the bottom port
		virtual void onReadBottom( ProtocolEntity* sender, PortElement* e ) {}
		/// Transmit message
		virtual void onTransmit( ProtocolEntity* sender, PortElement* e ) {}
		/// Pass on a message
		virtual void onPassUpward( ProtocolEntity* sender, PortElement* e ) {}	
	};

}


#endif /* ODEMX_PROTOCOL_ENTITY_INCLUDED */
