//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file ProtocolEntity.cpp

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Implementation of classes in ProtocolEntity.h

	\sa ProtocolEntity.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/



#include <odemx/protocol/ProtocolEntity.h>
#include <odemx/protocol/ProtocolLayer.h>
#include <odemx/protocol/NetNode.h>
#include <odemx/protocol/TransmissionMedium.h>

#include <string>

using namespace odemx;


ProtocolEntity::ProtocolEntity( ProtocolSimulation* sim, ProtocolLayer* layer, 
								NetNode* owner, ProtocolEntityObserver* o ):  
	// entity labels automatically get their owner and layer added in front
	Process( sim, (std::string(owner->getLabel())+" "+\
			 	   std::string(layer->getLabel())+" Entity").c_str() ),
	Observable<ProtocolEntityObserver>( o ),
	node( owner ),
	top( 0 ),
	bottom( 0 )
	{	
		topPort = new PortHead( sim, (string(getLabel())+" Top Port").c_str() );	
		bottomPort = new PortHead( sim, (string(getLabel())+" Bottom Port").c_str() );
	}


ProtocolEntity::~ProtocolEntity() 
	{
		// port destructor will not remove the port 
		// if there is another process connected to head/tail
		if ( topPort )
			delete topPort;
		if ( bottomPort )
			delete bottomPort;
	}



int ProtocolEntity::main() {

	Memo* theAlerter = 0;
	
	while (true) {
		
		// check input ports and wait for message arrival
		theAlerter = wait( topPortHead(), bottomPortHead() );

		if ( theAlerter == topPortHead() ) {
			handleUpperLayerInput( readTop() );
		}
		
		else if ( theAlerter == bottomPortHead() ) {
			handleLowerLayerInput( readBottom() );
		}

		else if ( theAlerter == 0 ) {
			
			if ( isInterrupted() ) {
				warning( "ProtocolEntity::main(): wait state interrupted by another process" );
			}
			else {
				error( "ProtocolEntity::main(): wait state alerter is 0" );
			}
		}
		
		else {
			error( "ProtocolEntity::main(): cannot identify alerter" );
			break;
		}
		
		theAlerter = 0;
	}
	
	return 0;
}


void ProtocolEntity::transmit( PortElement* e ) {
	
	// trace
	getTrace() -> mark( this, markTransmit );
	
	//observer
	_obsForEach( ProtocolEntityObserver, Transmit( this, e ) );
	
	if ( this == getNode() -> getBottomLayerEntity() ) {
		getTransmissionMedium() -> transmit( (ProtocolMessage*)e );
	}
	else {
		getBottomEntity() -> putTop( e );
	}
}


void ProtocolEntity::passUpward( PortElement* e ) {

	// trace
	getTrace() -> mark( this, markPassUpward );
	
	//observer
	_obsForEach( ProtocolEntityObserver, PassUpward( this, e ) );
	
	if ( this == getNode() -> getTopLayerEntity() ) {
		getNode() -> output( e );
	}
	else {
		getTopEntity() -> putBottom( e );
	}
}



/********** port access **************************************/


PortHead* ProtocolEntity::topPortHead() {
	return topPort;
}


PortTail* ProtocolEntity::getTopPortTail() {
	return topPort -> getTail();
}


PortHead* ProtocolEntity::bottomPortHead() {
	return bottomPort;	
}


PortTail* ProtocolEntity::getBottomPortTail() {
	return bottomPort -> getTail();
}


/****************** port manipulation ***********************/

// read the ProtocolEntity's ports
PortElement* ProtocolEntity::readTop(){

	// trace
	getTrace() -> mark( this, markReadTop );
	
	PortElement* e = topPort -> get();
	
	//observer
	_obsForEach( ProtocolEntityObserver, ReadTop( this, e ) );
	
	return e;
}


PortElement* ProtocolEntity::readBottom() {

	// trace
	getTrace() -> mark( this, markReadBottom );
	
	PortElement* e = bottomPort -> get();
	
	//observer
	_obsForEach( ProtocolEntityObserver, ReadBottom( this, e ) );

	return e;
}


// call PortTail::put()  
void ProtocolEntity::putTop( PortElement* e ) {
	getTopPortTail() -> put( e );
}		


void ProtocolEntity::putBottom( PortElement* e ) {
	getBottomPortTail() -> put( e );
}





/********** setters and getters ******************************/

NetNode* ProtocolEntity::getNode() {
	return node;
}		

TransmissionMedium* ProtocolEntity::getTransmissionMedium() {
	return ( (ProtocolSimulation*)getSimulation() ) -> getTransmissionMedium();
}

void ProtocolEntity::setTopEntity( ProtocolEntity* e ) {
	top = e;
}

void ProtocolEntity::setBottomEntity( ProtocolEntity* e ) {
	bottom = e;
}

ProtocolEntity* ProtocolEntity::getTopEntity() {
	return top;
}

ProtocolEntity* ProtocolEntity::getBottomEntity() {
	return bottom;
}



const MarkTypeId ProtocolEntity::baseMarkId = 5150;

const MarkType ProtocolEntity::markReadTop( "handle top input", baseMarkId+1, typeid(ProtocolEntity) );
const MarkType ProtocolEntity::markReadBottom( "handle bottom input", baseMarkId+2, typeid(ProtocolEntity) );
const MarkType ProtocolEntity::markTransmit( "transmit", baseMarkId+3, typeid(ProtocolEntity) );
const MarkType ProtocolEntity::markPassUpward( "pass upward", baseMarkId+4, typeid(ProtocolEntity) );
