//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file NetTopology.cpp

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Implementation of classes in NetTopology.h

	\sa NetTopology.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/




#include <odemx/protocol/NetTopology.h>
#include <odemx/protocol/NetNode.h>
#include <odemx/util/ErrorHandling.h>
#include <string>

using namespace odemx;
using namespace std;


NetTopology::NetTopology( ProtocolSimulation* sim, NetTopologyObserver* o ):
	Observable<NetTopologyObserver>( o ),
	env( sim ) 
	{}


NetTopology::~NetTopology() 
	{}


// make a node known to transmission
void NetTopology::registerNode( NetNode* n ) {

	//trace
	getTrace() -> mark( this, markRegisterNode, dynamic_cast<TraceProducer*>(n) );
	
	// observer
	_obsForEach( NetTopologyObserver, RegisterNode( this, n ) );
	
	// store new node
	allNodes.push_back( n );
			
	// insert new line
	connectivity[n];
	// set all its connections to other nodes to 0
	int e = allNodes.size();
	for ( int a = 0; a != e; ++a ) {
		connectivity[n][allNodes[a]].loss = 0;
		connectivity[n][allNodes[a]].p = 0;
	}
	
	// for all columns (nodes) set connection to new node to 0
	for ( int a = 0; a != e; ++a ) {
		connectivity[allNodes[a]][n].loss = 0;
		connectivity[allNodes[a]][n].p = 0;				
	}
	
	// "connection" to self 0, no need to send messages to self
	connectivity[n][n].loss = 0;
	connectivity[n][n].p = 0;
}


// remove node from connectivity array and the node list 
void NetTopology::removeNode( NetNode* n ) {

	NodeVector::iterator del; 
	del = find( allNodes.begin(), allNodes.end(), n );

	if ( del == allNodes.end() ) {
		error( "NetTopology::removeNode(): node not stored in list" );
		return;
	}
	
	//trace
	getTrace() -> mark( this, markRemoveNode, dynamic_cast<TraceProducer*>(n) );
	
	// observer
	_obsForEach( NetTopologyObserver, RemoveNode( this, n ) );
	
	int e = allNodes.size();
	// erase node n from all columns (node maps)
	for ( int a = 0; a != e; ++a ) {
		connectivity[ allNodes[a] ].erase( n );		
	}
	
	// delete line of node n
	connectivity.erase( n );

	// erase node from node list
	allNodes.erase( del );
	
	// remove entities from layers ???
}



bool NetTopology::validProbability( double p ) {

	if ( p < 0 ) {
		error( "NetTopology::checkProbability(): negative connection value" );
		return false;
	}

	if ( p > 1 ) {
		error( "NetTopology::checkProbability(): connection probability greater 1" );
		return false;
	}
	
	return true;
}


// set connectivity for one direction
void NetTopology::makeConnection( NetNode* n1, NetNode* n2, double p ) {

	if ( !validProbability(p) )
		return;
			
	//trace
	getTrace() -> beginMark( this, markMakeConnection );
	getTrace() -> addTag( tagFrom, dynamic_cast<TraceProducer*>(n1) );
	getTrace() -> addTag( tagTo, dynamic_cast<TraceProducer*>(n2) );
	getTrace() -> addTag( tagQuality, p );
	getTrace() -> endMark();
	
	// observer
	_obsForEach( NetTopologyObserver, MakeConnection( this, n1, n2 ) );

	// set connectivity between two nodes
	// new Draw(..) will automatically receive 
	// a new seed within one Simulation context
	connectivity[n1][n2].loss = new Draw ( env, "Draw", p);
	connectivity[n1][n2].p = p;
}



// set connectivity for one direction
void NetTopology::changeConnection( NetNode* n1, NetNode* n2, double p ) {

	if ( !validProbability(p) )
		return;
			
	//trace
	getTrace() -> beginMark( this, markChangeConnection );
	getTrace() -> addTag( tagFrom, dynamic_cast<TraceProducer*>(n1) );
	getTrace() -> addTag( tagTo, dynamic_cast<TraceProducer*>(n2) );
	getTrace() -> addTag( tagQuality, p );
	getTrace() -> endMark();
	
	// observer
	_obsForEach( NetTopologyObserver, ChangeConnection( this, n1, n2 ) );
	
	// set connectivity between two nodes
	// new Draw(..) will automatically receive 
	// a new seed within one Simulation context
	connectivity[n1][n2].loss = new Draw ( env, "Draw", p);
	connectivity[n1][n2].p = p;
}



void NetTopology::disconnect( NetNode* n1, NetNode* n2 ) {
	
	if ( connectivity[n1][n2].loss != 0 && connectivity[n2][n1].loss != 0 ) {
	
		delete( connectivity[n1][n2].loss );
		connectivity[n1][n2].loss = 0;
		connectivity[n1][n2].p = 0;
		
		delete( connectivity[n2][n1].loss );
		connectivity[n2][n1].loss = 0;
		connectivity[n2][n1].p = 0;
	}
	else {
		warning( "NetTopology::disconnect(): no previous connection between nodes" );
	}
		
	//trace
	getTrace() -> beginMark( this, markDisconnect );
	getTrace() -> addTag( tagNode, dynamic_cast<TraceProducer*>(n1) );
	getTrace() -> addTag( tagNode, dynamic_cast<TraceProducer*>(n2) );
	getTrace() -> endMark();
	
	// observer
	_obsForEach( NetTopologyObserver, Disconnect( this, n1, n2 ) );

}



// set same connectivity for bi-directional connections
void NetTopology::setConnection( NetNode* n1, NetNode* n2, double p_both ) {

	if ( ! validProbability( p_both ) )
		return;
	
	// when changing connection settings,
	// the old draw object must be destroyed
	if ( connectivity[n1][n2].loss != 0 ) {
		delete( connectivity[n1][n2].loss );
		
		changeConnection( n1, n2, p_both );
		changeConnection( n2, n1, p_both );
	}
	else {
		makeConnection( n1, n2, p_both );
		makeConnection( n2, n1, p_both );
	}
}



void NetTopology::setConnection( Label l1, Label l2, double p_both ) {

	NetNode* node1 = 0;
	NetNode* node2 = 0;
	
	// find source and destination NetNode* by comparing label with src and dest
	for( unsigned int i = 0; i < allNodes.size(); ++i ) {
		if ( node1 && node2 )
			break;
		
		if( string( allNodes[ i ] -> getLabel() ).compare( l1 ) == 0 )
			node1 = allNodes[ i ];
		
		if( string( allNodes[ i ] -> getLabel() ).compare( l2 ) == 0 )
			node2 = allNodes[ i ];
	}
	
	// if either one not found output error message, else set connection
	if( node1 && node2 ) {
	
		setConnection( node1, node2, p_both );
	}
	else
		error( "NetTopology::setConnection(): cannot determine source or destination" );
}


// set connectivity for one direction only
void NetTopology::setConnection( NetNode* n1, NetNode* n2, double p_to, double p_from ) {

	if ( !validProbability( p_to ) )
		return;
	
	if ( !validProbability( p_from ) )
		return;

	// when changing connection settings,
	// the old draw object must be destroyed
	if ( connectivity[n1][n2].loss != 0 ) {
		delete( connectivity[n1][n2].loss );
		
		changeConnection( n1, n2, p_to );
		changeConnection( n2, n1, p_from );
	}
	else {
		makeConnection( n1, n2, p_to );
		makeConnection( n2, n1, p_from );
	}
}




// set connectivity for one direction only
void NetTopology::setConnection( Label l1, Label l2, double p_to, double p_from ) {

	NetNode* node1 = 0;
	NetNode* node2 = 0;
	
	// find source and destination NetNode* by comparing label with src and dest
	for( unsigned int i = 0; i < allNodes.size(); ++i ) {
		if ( node1 && node2 )
			break;
		
		if( string( allNodes[ i ] -> getLabel() ).compare( l1 ) == 0 )
			node1 = allNodes[ i ];
		
		if( string( allNodes[ i ] -> getLabel() ).compare( l2 ) == 0 )
			node2 = allNodes[ i ];
	}
	
	// if either one not found output error message, else set connection
	if( node1 && node2 ) {
	
		setConnection( node1, node2, p_to, p_from );
	}
	else
		error( "NetTopology::setConnection(): cannot determine source or destination" );
}


bool NetTopology::connected( NetNode* node1, NetNode* node2 ) {
	
	// beware of uninitialized random number pointer
	if ( !connectivity[ node1 ][ node2 ].loss ) {
		return false;
	}

	return true;
}



NodeVector& NetTopology::getAllNodes() {
	return this -> allNodes;
}


// print current Matrix for debug purposes		
void NetTopology::printNodeMatrix() {
	
	NodeMatrix::iterator i;
	std::map< NetNode*, ConnInfo >::iterator j;    // *j is a pair

	// print headline
	std::cout << setw(8) << "";
	for (i = connectivity.begin(); i!=connectivity.end(); ++i) {
		std::cout << setw(8) << (*i).first->getLabel();
	}
	std::cout << std::endl;

	// print matrix entries
	for (i = connectivity.begin(); i!=connectivity.end(); ++i) {
		std::cout << setw(8) << (*i).first->getLabel();
		
		// print matrix line
		for (j = connectivity[(*i).first].begin(); j != connectivity[(*i).first].end(); ++j)
			std::cout << setw(8) << connectivity[(*i).first][(*j).first].p;
			
		std::cout << std::endl;
	}
	std::cout << std::endl;
}



const MarkTypeId NetTopology::baseMarkId = 5050;

const MarkType NetTopology::markRegisterNode( "register node", baseMarkId+3, typeid(NetTopology) );
const MarkType NetTopology::markRemoveNode( "remove node", baseMarkId+4, typeid(NetTopology) );
const MarkType NetTopology::markMakeConnection( "make connection", baseMarkId+5, typeid(NetTopology) );
const MarkType NetTopology::markChangeConnection( "change connection", baseMarkId+6, typeid(NetTopology) );
const MarkType NetTopology::markDisconnect( "disconnect", baseMarkId+7, typeid(NetTopology) );


const TagId NetTopology::baseTagId = 5050;

const Tag NetTopology::tagFrom( "from" );
const Tag NetTopology::tagTo( "to" );
const Tag NetTopology::tagQuality( "quality" );
const Tag NetTopology::tagNode( "node" );
