//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Coroutine.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/01/22

	\brief Declaration of Coroutine and CoroutineObserver

	\sa Coroutine.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_COROUTINE_INCLUDED
#define ODEMX_COROUTINE_INCLUDED

#include <typeinfo>

#include <odemx/coroutine/System.h>
#include <odemx/coroutine/CoroutineContext.h>
#include <odemx/util/Observable.h>
#include <odemx/util/LabeledObject.h>

namespace odemx {
	class CoroutineObserver;

	/** \class Coroutine

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Coroutine implements a function capable of switching execution to and back from another Coroutine inside function body.

		\note Coroutine supports Observation.

		<!-- [\sa {references to other classes}]* -->

		Coroutine is a system dependable but portable coroutine implementation. A coroutine is
		a function capable of switching execution to a point inside another coroutine and back to
		a point inside its own function body.
		The implementation is stack-based for Linux (based on concepts from Stroustrup "Task-Library",
		AT&T 1991 and Hansen "The C++ -Answer Book", Addison Wesley 1990) and Fiber-based for
		Microsoft Windows 98 and later (based on process implementation in ODEM by Martin von Lwis).

		The stack-based implementation should be portable to any platform using linear, continuous stacks
		(growing bottom up or top down) with a C++ (and exception handling) compatible setjmp()/longjmp()
		implementation.

		\warning Sun (Ultra)Sparc platform has a circular stack and is not supported at the moment.
		\warning IA-64 platform stores the return address of a function in a special branch register; not supported.

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Coroutine : public DefLabeledObject, public Observable<CoroutineObserver> {
		// Types
	public:
		/**
			\brief Coroutine states

			A Coroutine transits through 3 states during its lifetime.
			After creation it is in state CREATED. First time the Coroutine
			is activated it transits to RUNNABLE. When Coroutine returns
			it becomes TERMINATED.
		*/
		enum State {
			CREATED, ///< initial state
			RUNNABLE, ///< working state
			TERMINATED ///< final state
		};

		/**
			\brief start/continue execution of this Coroutine.
			\note The Coroutine will become the active Coroutine in its CoroutineContext.
			\note There is no more than one Coroutine active in its context at any time.
		*/
		void switchTo();

		/**
			\brief start/continue execution of this Coroutine.

			\copydoc Coroutine::switchTo
		*/
		void operator()();

		// Interface
	public:
		/**
			\brief Construction
			\param label
				label of Coroutine
			\param c
				pointer to CoroutineContext of this Coroutine
				(if c==0 a default CoroutineContext is used)
			\param o
				pointer to a CoroutineObserver

			\sa DefaultContext
		*/
		Coroutine(const char* label="", CoroutineContext* c=0, CoroutineObserver* o=0);

		~Coroutine(); ///< Destruction

		/**
			\brief coroutine state
			\return current state of Coroutine
			\sa Coroutine::State
		*/
		State getState() const {return state;}

		/**
			\brief get CoroutineContext of Coroutine
			\return CoroutineContext of this Coroutine
		*/
		CoroutineContext* getContext();

		/**
			\brief parent of coroutine
			\return parent (first caller) of coroutine
			\retval 0
				parent is CoroutineContext
			\retval !=0
				pointer to parent coroutine
		*/
		Coroutine* getParent() {return parent;}

		/**
			\brief last caller
			\return last caller of coroutine
			\retval 0
				last caller is CoroutineContext
			\retval !=0
				pointer to coroutine
		*/
		Coroutine* getCaller() {return caller;}

	protected:
		// Coroutine entry point
		virtual void start() = 0; ///< define this method to implement Coroutine behaviour

		// Implementation
	private:
		CoroutineContext* context; ///< CoroutineContext of this Coroutine
		Coroutine* parent; ///< return point after start() finished
		Coroutine* caller; ///< last caller of coroutine

		Coroutine::State state; ///< Coroutine state

		void initialize(); ///< called before first execution
		void clear(); ///< called after execution finished

		Coroutine::State setState(Coroutine::State newState); ///< state transitions

		// System dependent
#ifdef HAVE_FIBERS
		/// \name Fiber based implementation
		//@{
		LPVOID myFiber;
		LPVOID fiber;
		//@}

		/// \name Fiber based implementation
		//@{
		/// prepare execution switch (save active fiber).
		void saveFiber();
		//@}

		friend VOID CALLBACK ExecFiber(PVOID);
		friend void FiberSwitch(LPVOID fiber);
#else
		/// \name stack based implementation
		//@{
		static StackAdress adress1;
		static StackAdress adress2;
		//@}

		/// \name stack based implementation
		//@{
		StackAdress base; ///< begin of runtime stack
		StackAdress last; ///< top of runtime stack
		StackBuffer oldStack; ///< saved runtime stack
		int checksum; ///< stack buffer checksum
		jmp_buf env; ///< setjmp, longjmp data
		//@}

		/// \name stack based implementation
		//@{
		/**
			\brief prepare execution switch (save stack and registers)
			\retval true
				Coroutine has been restored,
				continue execution of Coroutine.
			\retval false
				Coroutine has saved execution state.
		*/
		bool saveEnvironment();
		void restoreEnvironment(); ///< switch to Coroutine (load stack and registers)
		//@}

		friend class CoroutineContext;
#endif
	};

	/** \interface CoroutineObserver

		\author RalfGerstenberger
		<!-- [\author <author>]* -->

		\brief Observer for Coroutine specific events.

		<!-- [\note {notes}]* -->

		\sa Coroutine

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class CoroutineObserver {
	public:
		virtual ~CoroutineObserver() { }
	public:
		virtual void onCreate(Coroutine* sender) {}; ///< creation
		virtual void onDestroy(Coroutine* sender) {}; ///< destruction

		virtual void onInitialize(Coroutine* sender) {}; ///< initialisation
		virtual void onClear(Coroutine* sender) {}; ///< Coroutine is cleared

		/// execution switches between Coroutines
		virtual void onSwitchTo(Coroutine* sender, Coroutine* previousActive) {};
		/// execution switches from Context to Coroutine
		virtual void onSwitchTo(Coroutine* sender, CoroutineContext* previousActive) {};

		/// state transition
		virtual void onChangeState(Coroutine* sender, Coroutine::State oldState, Coroutine::State newState) {};
	};

#ifdef HAVE_FIBERS
	/**
		\internal
		\brief Fiber start function
		\param p
			pointer to a Coroutine
	*/
	VOID CALLBACK ExecFiber(PVOID p);

	/**
		\internal
		\brief encapsulates SwitchToFiber() call
		\param fiber
			pointer to fiber
	*/
	void FiberSwitch(LPVOID fiber);
#endif
}

#endif

