//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
//	Coroutine.cpp
//
//	Created			by
//	2002/01/22		Ralf Gerstenberger
//----------------------------------------------------------------------------
/**	\file Coroutine.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/01/22

	\brief Implementation of Coroutine

	\sa Coroutine.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#if defined(_MSC_VER) && defined(FORCE_STACK_IMPL)
#pragma warning(disable : 4530)
#endif

#include <odemx/coroutine/Coroutine.h>
#include <odemx/util/ErrorHandling.h>

#include <exception>

using namespace odemx;

Coroutine::Coroutine(const char* label, CoroutineContext* c, CoroutineObserver* o) :
	Observable<CoroutineObserver>(o)
{
	// initial state
	state=CREATED;
	parent=0;
	caller=0;

	context=c;
	if (context==0)
		// use default context
		context=getDefaultContext();

	// set label
	DefLabeledObject::setLabel(context, label);

	// register Coroutine to context
	getContext()->beginCoroutine(this);

	_obsForEach(CoroutineObserver, Create(this));

	// System dependent
#ifdef HAVE_FIBERS
	myFiber=fiber=0;
#else
	base=0;
	last=0;
	oldStack=0;
	checksum=0;
#endif
}

Coroutine::~Coroutine() {
	if (context->getActiveCoroutine()==this) {
		// Error: self destruction, next execution is unknown
		fatalError("~Coroutine(); next execution is unknown", -1);
	}

	// clear Coroutine if necessary
	if (getState()!=TERMINATED)
		clear();

	_obsForEach(CoroutineObserver, Destroy(this));

	// System dependent
#ifdef HAVE_FIBERS
	if (fiber!=0) {
		DeleteFiber(fiber);
	}
#else
	if (oldStack!=0) {
		delete[] oldStack;
	}
#endif
}

void Coroutine::initialize() {
	if (state!=CREATED) {
		// Error: already initialized
		error("initialize(); Coroutine already initialized");
		return;
	}

#ifdef HAVE_FIBERS
	myFiber = fiber = CreateFiber(0, ExecFiber, this);
	if (fiber==0) {
		// Error: failed to create fiber
		error("initialize(); failed to create fiber");
		return;
	}
#else
	char a;
	char b;

	// calculate size of initialize() call stack (ip + this)
	if (adress1==0) {
		adress1 = &b+sizeof(b);
		initialize();
	} else if (adress2==0) {
		adress2 = &a;
		return;
	}

	base = &a;

	// adjust 'base' to point in stack before call to _switchTo()_
	if (adress1<adress2) {
		// stack is growing bottom up
		base -= 2*(adress2-adress1);

		// adjust CoroutineContext
		if (getContext()->base==0 || getContext()->base>base)
			// CoroutineContext has to save the parts of stack that will
			// be used by this Coroutine.
			getContext()->base=base;
	} else {
		// stack is growing top down
		base += 2*(adress1-adress2);

		// adjust CoroutineContext
		if (getContext()->base==0 || getContext()->base<base)
			// CoroutineContext has to save the parts of stack that will
			// be used by this Coroutine.
			getContext()->base=base;
	}
#endif

	_obsForEach(CoroutineObserver, Initialize(this));

	setState(RUNNABLE);
}

CoroutineContext* Coroutine::getContext() {
	return context;
}

void Coroutine::switchTo() {
	if (state==CREATED) {
		// store return point
		parent = context->getActiveCoroutine();

		initialize();
	}
	caller=context->getActiveCoroutine();

	if (context->getActiveCoroutine()!=0)
		_obsForEach(CoroutineObserver, SwitchTo(this, context->getActiveCoroutine()))
	else
		_obsForEach(CoroutineObserver, SwitchTo(this, context));

	if (state!=RUNNABLE) {
		// Warning: Coroutine is not runnable
		warning("switchTo(); switch to Coroutine failed: Coroutine is not RUNNABLE, continue in parent");

		if (parent!=0)
			parent->switchTo();
		else
			getContext()->switchTo();
	}

	if (context->getActiveCoroutine()==this)
		return;

#ifdef HAVE_FIBERS
	Coroutine* oldActive = context->setActiveCoroutine(this);
	if (oldActive!=0) {
		// switchTo from Coroutine;
		// Coroutine has to save 'switch in' point.
		oldActive->saveFiber();
	} else {
		// switchTo from context;
		// CoroutineContext has to save 'switch out' point.
		context->saveFiber();
	}

	FiberSwitch(fiber);

#else
	Coroutine* oldActive = context->setActiveCoroutine(this);
	if (oldActive!=0) {
		if (oldActive->saveEnvironment())
			return;
	} else {
		if (context->saveEnvironment())
			return;
	}

	// first execution
	if (last==0) {

		try {
			// start execution
			start();
		} catch (...) {
#ifdef _WIN32
			// Error: exception handling not supported
			fatalError("exception handling not supported",1);
#else
			// Error: unhandled exception in coroutine
			error("unhandled exception in coroutine");
#endif
		}

		// execution finished
		clear();

		// return
		if (parent!=0)
			parent->switchTo();
		else
			getContext()->switchTo();

		// This point should never be reached.
		fatalError("internal Error; return from coroutine failed",-1);
	}

	restoreEnvironment();
#endif
}

void Coroutine::operator ()() {
	switchTo();
}

void Coroutine::clear() {
	_obsForEach(CoroutineObserver, Clear(this));

	setState(TERMINATED);

	// leave context
	getContext()->endCoroutine(this);
}

#ifdef HAVE_FIBERS
void Coroutine::saveFiber() {
	// get active fiber
	fiber = GetCurrentFiber();

	if (fiber==0) {
		// Error: could not get a fiber
		fatalError("saveFiber(); Coroutine saveFiber failed: could not get active fiber",-1);
	}
}

VOID CALLBACK
odemx::ExecFiber(PVOID param){
	Coroutine *p = static_cast<Coroutine*>(param);

	try {
		// start execution
		p->start();
	} catch (...) {
		// Error: unhandled exception in coroutine
		error("unhandled exception in coroutine");
	}

	// execution finished
	p->clear();

	// return
	if (p->parent!=0)
		p->parent->switchTo();
	else
		p->context->switchTo();

	// This point should never be reached.
	fatalError("internal Error; return from coroutine failed",-1);
}

/**
	\internal
	\note FiberSwitch is a 'Work around'.

	In Visual C++ 6.0, when running a debug session,
	SwitchToFiber() does not restore EIP-register (Instruction pointer). [Maybe correct]
	FiberSwitch() ensures, that all calls to SwitchToFiber()
	come from the same instruction point in code.
*/
void odemx::FiberSwitch(LPVOID fiber) {
	SwitchToFiber(fiber);
}

#else
bool Coroutine::saveEnvironment() {
	char b;
	last = &b;

	if (state==TERMINATED)
		return false;

	// save stack
	if (oldStack!=0)
		delete[] oldStack;

	if (base < last) {
		// stack is growing bottom up
		oldStack = new char[last-base];
#ifdef sparc
		asm("t 3");
#endif
		memcpy(oldStack, base, (unsigned int)(last-base));
		checksum = calcChecksum(base, last-base);
	} else {
		// stack is growing top down
		oldStack = new char[base-last];
#ifdef sparc
		asm("t 3");
#endif
		memcpy(oldStack, last, (unsigned int)(base-last));
		checksum = calcChecksum(last, base-last);
	}

	// save environment
	if (setjmp(env)!=0) {
		return true;
	} else {
		return false;
	}
}

void Coroutine::restoreEnvironment() {
	char c;

	// restore stack
	if (oldStack==0) {
		// Error: no stack available
		fatalError("restoreEnvironment(); Coroutine restoration failed: no stack available",-1);
	}

	if (base < last) {
		// stack is growing bottom up
		// Recurse to get space in stack for Coroutine stack and
		// to have a valid this-pointer after memcpy().
		if (&c-(adress2-adress1) < last) // recursion to get space in stack
			restoreEnvironment();

		if (checksum!=calcChecksum(oldStack, last-base)) {
			// Error: stack corrupted
			fatalError("restoreEnvironment(); Coroutine restoration failed: stack corrupted",-1);
		}
#ifdef sparc
		asm("t 3");
#endif
		memcpy(base, oldStack, (unsigned int)(last-base));
	} else {
		// stack is growing top down
		// Recurse to get space in stack for Coroutine stack and
		// to have a valid this-pointer after memcpy().
		if (&c+(adress1-adress2) > last)
			restoreEnvironment();

		if (checksum!=calcChecksum(oldStack, base-last)) {
			// Error: stack corrupted
			fatalError("restoreEnvironment(); Coroutine restoration failed: stack corrupted",-1);
		}
#ifdef sparc
		asm("t 3");
#endif
		memcpy(last, oldStack, (unsigned int)(base-last));
	}

	// restore environment
	longjmp(env, 1);
}
#endif

Coroutine::State Coroutine::setState(Coroutine::State newState) {
	Coroutine::State oldState = state;
	state = newState;

	_obsAForEach(CoroutineObserver, State, oldState, newState);

	return oldState;
}

#ifndef HAVE_FIBERS
StackAdress odemx::Coroutine::adress1 = 0;
StackAdress odemx::Coroutine::adress2 = 0;
#endif

