//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file HtmlTrace.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/06/09

	\brief Declaration of odemx::HtmlTrace

	\sa HtmlTrace.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifdef _MSC_VER
#pragma warning(disable : 4786)
#endif

#ifndef ODEMX_HTMLTRACE_INCLUDED
#define ODEMX_HTMLTRACE_INCLUDED

#include <odemx/util/Trace.h>
#include <odemx/util/ErrorHandling.h>
#include <odemx/base/Process.h>
#include <odemx/base/Simulation.h>
#include <odemx/base/TraceFilter.h>

#include <iostream>
#include <string>

namespace odemx {
	/** \class HtmlTrace

		\ingroup base

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief HTML trace

		%HtmlTrace logs simulation events of all model components to a
		html file or an open output stream. If the user provides a filename
		%HtmlTrace opens and closes this file automatically.

		\par Example:
		To use a %HtmlTrace the user has to create an object of type %HtmlTrace
		first. A good place to do this is initSimulation().
\code
class UserSimulation : public Simulation {
...
public:
HtmlTrace* myTrace;
...
virtual void initSimulation()
{
	myTrace = new HtmlTrace(this, "MyTrace.html");
\endcode
		After that the created trace object has to be registered to the simulation
		to receive events with addConsumer(). It can be unregistered with
		removeConsumer().
\code
	addConsumer(myTrace);
\endcode
		%HtmlTrace provides a crude filter to reduce the number of logged simulation
		events. The filter is set with setFilter(). This has to be done	before
		the trace is started or while it is paused. (see setFilter() for details)
\code
	myTrace->setFilter("none; st:class odemx::Condq;");
\endcode
		Finally to activate the logging call startTrace() on the simulation. You can use
		pauseTrace() and continueTrace() to break the trace temporarily.
\code
	startTrace();
...
}
...
int main()
{
	UserSimulation sim;
	sim.runUntil(20.0);
	sim.pauseTrace();
	sim.runUntil(25.0);
	sim.continueTrace();
	sim.runUntil(31.0);
\endcode
		Use stopTrace() to stop tracing manually.
\code
	sim.stopTrace();
	sim.run();
}
\endcode
		It is possible to have more than one %HtmlTrace registered to a simulation.
		Each could use a different filter.
\code
in main()
{
	UserSimulation sim;
	HtmlTrace mySecondTrace(&sim, "My2Trace.html");
	mySecondTrace.setFilter("all; sl: MyCondq");
	sim.addConsumer(&mySecondTrace);
	...
}
\endcode

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class HtmlTrace : public virtual TraceConsumer,
					  public TraceFilter
	{
	public:
		/**
			\name Construction
			@{
		*/
		/**
			\param s
				Simulation
			\param filename
				file for output

			%HtmlTrace opens the file \p filename and writes into
			this file. When %HtmlTrace is deleted the file is closed.
			If %HtmlTrace fails to open the file the output is send
			to \c stdout.
		*/
		HtmlTrace(Simulation* s, const char* filename = 0);
		/**
			\param s
				Simulation
			\param os
				ostream for output

			%HtmlTrace writes its output to the ostream \p os.
		*/
		HtmlTrace(Simulation* s, std::ostream& os);
		//@}

		/**
			\name Destruction
			@{
		*/
		/**
			In case %HtmlTrace has opened a file it is closed by the destructor.
		*/
		virtual ~HtmlTrace();
		//@}

		/**
			\name Filter
			@{
		*/
		/**
			\brief Set trace filter
			\param f
				filter string

			%HtmlTrace provides a filter to remove marks from the trace.
			This filter is set with the string \p f. The string is read
			from left to right. If the keywords \c none or \c all are
			found the filter-policy is set. The last occurrence of one of
			these keywords is used. If none of these keywords is found
			\c all is used by default. The \c none policy blocks all marks
			except those specified in the string \p f. The \c
			all policy passes all marks except those specified in the
			string \p f.
			A mark can be specified by its MarkType or its sender. The
			MarkType consists of the MarkTypeId, the MarkTypeName and the
			scope of the mark. Each of these fields can be used to specify
			Marks. The keyword \c mti: is used to specify MarkTypeIds:
			\code
mti:1
mti: 345; mti:1334;
mti: 15, 2354, 2;
			\endcode
			The keyword \c mtn: can be used to specify MarktTypeNames:
			\code
mtn:begin
mtn:end , activate; mtn: destroy;
			\endcode
			The keyword \c mts: is used to specify a scope:
			\code
mts:class odemx::Process
mts:class odemx::Res, class odemx::Bin; mts: class odemx::Waitq;
			\endcode
			The sender can be specified by its label or its type. The
			keyword \c sl: is used to specify the label:
			\code
sl:MyWaitq, Car
sl: Client 2, Client 3; sl: Servant;
			\endcode
			The keyword \c st: specifies the sender type:
			\code
st:class ServantType, class Bin
st:class MyTestProcess1; st:class Storage
			\endcode
			A mark is specified if the MarkTypeId, the MarkTypeName, the
			mark scope, the sender label or the sender type are specified
			by the filter.

			\note The filter can only be set in pause mode or
			before trace starts.
		*/
		virtual void setFilter(const char* f);
		//@}

		// Implementation
	private:
		/**
			\name Simple trace marks

			Simple trace marks are handled by the following methods.

			@{

		*/
		virtual void mark(const TraceProducer* sender, const MarkType* m, const char* comment = 0);

		// Marks that include information about producer associations:
		virtual void mark(const TraceProducer* sender, const MarkType* m, const TraceProducer* partner, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, const TraceProducer* firstPartner, const TraceProducer* secondPartner, const char* comment = 0);

		// Marks that include information about producer attributes:
		virtual void mark(const TraceProducer* sender, const MarkType* m, bool newValue, bool oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, char newValue, char oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, short newValue, short oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, int newValue, int oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, long newValue, long oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, float newValue, float oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, double newValue, double oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, const char* newValue, const char* oldValue, const char* comment = 0);

		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned char newValue, unsigned char oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned short newValue, unsigned short oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned int newValue, unsigned int oldValue, const char* comment = 0);
		virtual void mark(const TraceProducer* sender, const MarkType* m, unsigned long newValue, unsigned long oldValue, const char* comment = 0);
		//@}

		/**
			\name Complex trace marks

			Complex trace marks are transmitted with the following methods.
			A complex trace mark can be interpreted as a tree of tag-value pairs.

			@{
		*/
		/// beginMark indicates the construction of a new mark
		virtual void beginMark(const TraceProducer* sender, const MarkType* m);
		/// endMarks signals the end of construction
		virtual void endMark(const char* comment = 0);

		// This methods are called to describe the construction of the complex mark.
		virtual void beginTag(Tag t);
		virtual void endTag(Tag t);

		virtual void addTag(Tag t, bool value);
		virtual void addTag(Tag t, char value);
		virtual void addTag(Tag t, short value);
		virtual void addTag(Tag t, int value);
		virtual void addTag(Tag t, long value);
		virtual void addTag(Tag t, float value);
		virtual void addTag(Tag t, double value);
		virtual void addTag(Tag t, const char* value);
		virtual void addTag(Tag t, const TraceProducer* value);

		virtual void addTag(Tag t, unsigned char value);
		virtual void addTag(Tag t, unsigned short value);
		virtual void addTag(Tag t, unsigned int value);
		virtual void addTag(Tag t, unsigned long value);
		//@}

		/**
			\name Trace Control
			@{
		*/
		virtual void startTrace(); ///< HtmlTrace initialises a new trace
		virtual void stopTrace(); ///< HtmlTrace cleans up
		virtual void pauseTrace(); ///< HtmlTrace starts a break
		virtual void continueTrace(); ///< HtmlTrace ends a break
		//@}

	private:
		Simulation* sim; ///< pointer to simulation
		bool manageOut; ///< manage the outstream
		bool pause; ///< in pause mode
		bool active; ///< is active
		std::ostream* out; ///< out-stream for HTML text
		SimTime lastTime; ///< last written SimTime;
		const TraceProducer* lastSender; ///< last sender

		static const char* colors[]; ///< colours
		unsigned int colorSelect; ///< colour selection

		/**
		 \name Help functions
		 @{
		*/
		/// begin table line
		void beginLine(const TraceProducer* sender, const MarkType* m);

		/// end table line
		void endLine(const char* comment);

		const char* printState( short val ) {
		
			switch( val ) {
				case( 0 ): return "CREATED";
				case( 1 ): return "CURRENT";
				case( 2 ): return "RUNNABLE";
				case( 3 ): return "IDLE";
				case( 4 ): return "TERMINATED";
			}

			// never reached
			error( "HtmlTrace::printState(): unknown process state" );
			return "Trace ERROR";
		}

		/// write value-changed line
		template <typename T>
		void writeLine(const TraceProducer* sender, const MarkType* m, T newValue, T oldValue, const char* comment) {
			pass(m, sender);
			if (skipOutput())
				return;

			beginLine(sender, m);
			table(1);
			*out << "<tr>";
			cell(2,0);
			*out << "old Value" << "</td>";
			cell(2,1);
			if ( m -> getId() == 1030 )
				*out << printState( (short)oldValue ) << "</td></tr>";
			else
				*out << oldValue << "</td></tr>";
				
			*out << "<tr>";
			cell(2,0);
			*out << "new Value" << "</td>";
			cell(2,1);
			if ( m -> getId() == 1030 )
				*out << printState( (short)newValue ) << "</td></tr></table>";
			else			
				*out << newValue << "</td></tr></table>";
			endLine(comment);
		}

		void writeLine(const TraceProducer* sender, const MarkType* m, bool newValue, bool oldValue, const char* comment);
		void writeLine(const TraceProducer* sender, const MarkType* m, const char* newValue, const char* oldValue, const char* comment);

		/// write tag-value
		template <typename T>
		void writeValue(Tag t, T value) {
			if (skipOutput()) return;

			*out << "<tr>";
			cell(2,0);
			if (t.getName()!=Tag::NONAME)
				*out << translate(t.getName());
			else
				*out << t.getId();
			*out << "</td>";
			cell(2,1);
			*out << value << "</td></tr>";
		}

		void writeValue(Tag t, bool value);
		void writeValue(Tag t, const char* value);
		void writeValue(Tag t, const TraceProducer* value);

		/// begin table
		void table(int i);

		/// begin cell
		void cell(int i, int j);

		/// generate output
		bool skipOutput() {return pause || !hasPassed();}

		/// translate string to html string
		const char* translate(const std::string& s);
		/// \overload
		const char* translate(const char* s);
		//@}
	};
}

#endif

