//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file HtmlReport.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/05/23

	\brief Declaration of odemx::HtmlReport

	\sa HtmlReport.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_HTMLREPORT_INCLUDED
#define ODEMX_HTMLREPORT_INCLUDED

#include <odemx/util/Report.h>
#include <odemx/base/Simulation.h>

#include <iostream>
#include <string>

namespace odemx {
	/** \class HtmlReport

		\ingroup base

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief HTML report

		%HtmlReport gathers information from report generating model
		components and generates a html output. This output is written
		to a file or an output stream.

		\par Example:
		At first the user has to create an instance of %HtmlReport in order
		to use it. A good place to do this is the initSimulation() function
		of a simulation class.
\code
class UserSimulation : public Simulation {
...
public:
HtmlReport* myReport;
Waitq* myWaitq;
Accum* myStatistics;
Uniform* myRandom;
...
virtual void initSimulation {
	myReport = new HtmlReport(this, "MyReport.html");
	myWaitq = new ...

\endcode
		%HtmlReport gathers information only from registered model components.
		To add a model component to an instance of %HtmlReport use addProducer().
		You can remove a registered object with removeProducer(). Only objects
		which support Report (as noted in documentation) can be added to a
		%HtmlReport.
\code
	myReport.addProducer(myWaitq);
	myReport.addProducer(myStatistics);
	myReport.addProducer(myRandom);
...
}
...
\endcode
		The generation of a report is triggered with generateReport().
\code
int main()
{
	UserSimulation mySim;
	mySim.run();
	mySim.myReport->generateReport();
}
\endcode
		You can have several reports in a single simulation, which gather information
		from different components. The content of the report does not depend on
		the time when a model component is added to a report but only on the model
		component itself and on the time when the generation is triggered. Most of
		the model components that support report are statistical objects which have
		a function called reset(). Such a reset	can be done for every object individually
		or for all objects at the simulation object.
\code
int main()
{
	UserSimulation mySim;
	mySim.runUntil(10.0);
	mySim.reset();
	mySim.runUntil(40.0);
	mySim.myStatistics->reset();
	mySim.run();

	HtmlReport mySecondReport(&mySim, "My2Report.html");
	mySecondReport.addProducer(myRandom);

	mySim.myReport->generateReport();
	mySecondReport.generateReport();
}
\endcode

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class HtmlReport : public Report {
	public:
		/**
			\name Construction
			@{
		*/
		/**
			\param s
				pointer to the Simulation object
			\param filename
				filename of output file

			%HtmlReport opens the file \p filename and writes into
			this file. When %HtmlReport is deleted the file is closed.
			If %HtmlReport fails to open the file the output is send
			to \c stdout.
		*/
		HtmlReport(Simulation* s, const char* filename = 0);
		/**
			\param s
				pointer to the Simulation object
			\param os
				ostream for output

			%HtmlReport writes to the output stream \p os.
		*/
		HtmlReport(Simulation* s, std::ostream& os);
		//@}

		/**
			\name Destruction
			@{
		*/
		/**
			In case %HtmlReport has opened a file it is closed by the
			destructor.
		*/
		virtual ~HtmlReport();
		//@}

	protected:
		/**
			\brief processTables implementation

			%HtmlReport is derived from the Report class. Report implements
			management of report producer and data transfer. The data
			are provided by registered report producer and transferred as
			tables. %HtmlReport implements the processTables() function to
			generate html output from this tables.
		*/
		virtual void processTables();

		// Implementation
	private:
		Simulation* sim; ///< Simulation
		bool manageOut; ///< manage out
		std::ostream* out; ///< outstream

		/// \overload
		const char* translate(const char* s);
		/// translate string to xml string
		const char* translate(const std::string& s);
	};
}

#endif

