//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file HtmlReport.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/05/23

	\brief Implementation of classes in HtmlReport.h

	\sa HtmlReport.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifdef _MSC_VER
#pragma warning(disable : 4786)
#endif

#include <odemx/base/HtmlReport.h>
#include <odemx/util/Version.h>
#include <odemx/util/ErrorHandling.h>

#include <fstream>
#include <cassert>

using namespace std;
using namespace odemx;

HtmlReport::HtmlReport(Simulation* s, const char* fileName/*= 0*/) :
	sim(s), manageOut(true), out(&cout)
{
	assert(sim!=0);

	if (fileName==0)
		manageOut = false;
	else {
		out = new ofstream(fileName);
		if (out==0 || !(*out)) {
			// Error: cannot open file
			error("HtmlReport cannot open file; sending output to stdout.");
			manageOut = false;
			out = &cout;
		}
	}
}

HtmlReport::HtmlReport(Simulation* s, std::ostream& os) :
	sim(s), manageOut(false), out(&os)
{
	assert(sim!=0);

	if (!(*out)) {
		// Error: invalid ostream
		error("HtmlReport invalid ostream; sending output to stdout.");
		out = &cout;
	}
}

HtmlReport::~HtmlReport() {
	if (manageOut) {
		static_cast<ofstream*>(out)->close();
		delete out;
	}
}

void HtmlReport::processTables() {
	Table* tab=0;
	vector<Table*>::iterator it;

	// write DOCTYPE
	*out << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\" \"http://www.w3.org/TR/html4/strict.dtd\">" << endl;

	// write HTML-Header
	*out << "<html>" << endl << "<title>" << endl;
	*out << "HtmlReport for simulation: " << translate(sim->getLabel()) << ' ' << "at simtime: " << sim->getTime() << endl;
	*out << "</title>" << endl;

	// begin body
	*out << "<body>" << endl;

	// write Headlines
	*out << "<table width=\"100%\" border=\"1\" rules=\"rows\" frame=\"hsides\">" << endl;
	*out << "<thead>" << endl;

	// Simulation name and simtime
	*out << "<tr><th colspan=\"3\" align=\"left\"><h1> Simulation: ";
	*out << translate(sim->getLabel());
	*out << "</h1></th><th align=\"right\"><h1>SimTime:</h1></th>";
	*out << "<th align=\"right\"><h1>" << sim->getTime() << "</h1></th></tr>" << endl;

	// HtmlReport and ODEMx version
	*out << "<tr><th colspan=\"3\" align=\"left\"><h2>" << "HtmlReport";
	*out << "</h2></th><th align=\"right\"><h2>ODEMx Version:</h2></th>";
	*out << "<th align=\"right\"><h2>" << translate(Version::getString()) << "</h2></th></tr>" << endl;

	// Table head and foot
	*out << "<tr height=\"20\"><th colspan=\"5\"></th></tr>" << endl;
	*out << "</thead><tfoot>" << endl;
	*out << "</tfoot><tbody></tbody></table>" << endl;

	// write tables
	for (it=ts.begin(); it!=ts.end(); ++it) {
		tab=*it;

		if (tab==0)
			continue;

		// open table
		*out << "<p>" << endl;
		*out << "<table frame=\"box\" border=\"1\" cellspacing=\"0\" cellpadding=\"2\">" << endl;

		// title
		*out << "<thead>" << endl;
		*out << "<tr>" << "<th align=\"left\" colspan=\"" << tab->getNumberOfColumns() << "\">" << translate(tab->getLabel()) << "</th>" << endl;

		// write column labels
		*out << "<tr bgcolor=\"darkturquoise\">" << endl;
		for (unsigned int i=0; i<tab->getNumberOfColumns(); ++i) {
			*out << "<th align=\"center\" width=\"70\">" << translate(tab->getLabelOfColumn(i)) << "</th>";
		}
		*out << endl << "</tr>" << endl;
		*out << "</thead>" << endl;

		*out << "<tfoot>" << endl;
		if (tab->getNumberOfLines()>10)
		{
			// repeat column labels at foot
			*out << "<tr bgcolor=\"darkturquoise\">" << endl;
			for (unsigned int i=0; i<tab->getNumberOfColumns(); ++i) {
				*out << "<th align=\"center\" width=\"70\">" << translate(tab->getLabelOfColumn(i)) << "</th>";
			}
			*out << endl << "</tr>" << endl;
		}
		*out << "</tfoot><tbody>" << endl;

		// write lines
		const char* colors[] = {"azure", "beige"};
		for (unsigned int j=0; j<tab->getNumberOfLines(); ++j) {
			*out << "<tr>" << endl;

			// write cells
			for (unsigned int i=0; i<tab->getNumberOfColumns(); ++i) {
				*out << "<td bgcolor=\"" << colors[i%2] << "\"";
				switch (tab->getTypeOfColumn(i)) {
				case INTEGER:
				case REAL:
					*out << "align=\"right\"";
					break;
				case STRING:
					break;
				default:
					break;
				}

				*out << ">" << translate(tab->getSTRING(i, j)) << "</td>";
			}

			*out << "</tr>" << endl;
		}

		// close table-tree
		*out << "</tbody></table>" << endl;
		*out << "</p>" << endl;

	}

	// close body
	*out << "</body>" << endl;

	// close Html
	*out << "</html>" << endl;
}

const char* HtmlReport::translate(const char* s) {
	string tmp=s;
	return translate(tmp);
}

const char* HtmlReport::translate(const std::string& s) {
	static string ret;
	string::const_iterator i;

	// prepare ret
	ret.erase();

	// replace forbidden characters
	for (i=s.begin(); i!=s.end(); ++i) {
		switch (*i) {
		case '<':
			ret+="&lt;";
			break;
		case '>':
			ret+="&gt;";
			break;
		default:
			ret+=*i;
		}
	}
	ret+='\0';

	return ret.c_str();
}
