//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Event.cpp

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Implementation of classes in Event.h

	\sa Event.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/



#include <odemx/base/Event.h>
#include <odemx/util/ErrorHandling.h>
#include <string>

#include <odemx/Debug.h>

	
using namespace odemx;

Event::Event ( Simulation* sim, Label l, EventObserver* eo ):
	Sched( EVENT ),
	Observable<EventObserver>( eo ),
	scheduled( false ),
	eventTime( 0.0 ),
	env( sim ),
	priority( 0.0 ) // same default priority like Process
	{
		// set label as in Coroutine
		DefLabeledObject::setLabel( sim, l );
		
		// trace
		getTrace() -> mark( this, markCreate );
		
		// observer
		_obsForEach(EventObserver, Create(this));
	}


Event::~Event () 
	{
		// trace
		getTrace() -> mark( this, markDestroy );
		
		// observer
		_obsForEach(EventObserver, Destroy(this));
		
		if ( !env -> isFinished() ) {
			
			// warning: Event should not be destroyed while simulation is running.
			warning("~Event(): simulation not finished");
		}	
	}
	

SimTime Event::getExecutionTime() const {

	if ( !scheduled )
		return 0.0;

	return eventTime;
}


SimTime Event::setExecutionTime( SimTime newTime ) {

	SimTime oldTime = eventTime;
	eventTime = newTime;

	// trace
	getTrace()->mark(this, markChangeExecutionTime, newTime, oldTime);

	// observer
	_obsAForEach(EventObserver, ExecutionTime, oldTime, eventTime);

	return oldTime;
}


Priority Event::getPriority() const {

	return priority;
}


SimTime Event::getCurrentTime() {

	return env -> getTime();
}


bool Event::isScheduled() {

	return scheduled;
}


Simulation* Event::getSimulation() {
	
	return env;
}


void Event::schedule() {
	
	//trace
	getTrace() -> mark( this, markSchedule, getPartner() );

	// observer
	_obsForEach( EventObserver, Schedule( this ) );
	
	// beware of moving this statement; getExecutionTime() needs it here!
	scheduled = true;

	// scheduling
	setExecutionTime( getCurrentTime() );
	env -> insertSched( this );
}


void Event::scheduleIn( SimTime delay ) {

	// cannot schedule Event
	if ( delay < 0 ) {
		error( "Event::scheduleIn(): cannot schedule Event, invalid SimTime < 0" );
		return;
	}

	// beware of moving this statement getExecutionTime() uses it!
	scheduled = true;
		
	// trace
	getTrace() -> beginMark( this, markScheduleIn );
	getTrace() -> addTag( tagCurrent, getPartner() );
	getTrace() -> addTag( tagRelSimTime, delay );
	getTrace() -> endMark();

	// observer
	_obsForEach( EventObserver, ScheduleIn( this, delay ) );

	// scheduling
	setExecutionTime( getCurrentTime() + delay );
	env -> insertSched( this );
}


void Event::scheduleAt( SimTime absoluteTime ) {
	
	// cannot schedule Event
	if ( absoluteTime < getCurrentTime() ) {
		error( "Event::scheduleAt(): cannot schedule Event, SimTime already passed" );
		return;
	}
	
	// trace
	getTrace() -> beginMark( this, markScheduleAt );
	getTrace() -> addTag( tagCurrent, getPartner() );
	getTrace() -> addTag( tagAbsSimTime, absoluteTime );
	getTrace() -> endMark();

	// observer
	_obsForEach( EventObserver, ScheduleAt( this,  absoluteTime ) );

	// beware of moving this statement; getExecutionTime() uses it!
	scheduled = true;

	// scheduling
	setExecutionTime( absoluteTime );
	env -> insertSched( this );
}


void Event::scheduleAppend() {

	//trace
	getTrace() -> mark( this, markScheduleAppend, getPartner());

	// observer
	_obsForEach( EventObserver, ScheduleAppend( this ) );
	
	// beware of moving this statement; getExecutionTime() needs it here!
	scheduled = true;

	// scheduling
	setExecutionTime( getCurrentTime() );
	env -> addSched( this );
}


void Event::scheduleAppendIn( SimTime delay ) {
	
	// cannot schedule Event
	if ( delay < 0 ) {
		error( "Event::scheduleAppendIn(): cannot schedule Event, invalid SimTime < 0" );
		return;
	}
		
	// trace
	getTrace() -> beginMark( this, markScheduleAppendIn );
	getTrace() -> addTag( tagCurrent, getPartner() );
	getTrace() -> addTag( tagRelSimTime, delay );
	getTrace() -> endMark();

	// observer
	_obsForEach( EventObserver, ScheduleAppendIn( this, delay ) );

	// beware of moving this statement; getExecutionTime() uses it!
	scheduled = true;
	
	// scheduling
	setExecutionTime( getCurrentTime() + delay );
	env -> addSched( this );
}


void Event::scheduleAppendAt( SimTime absoluteTime ) {
		
	// cannot schedule Event
	if ( absoluteTime < getCurrentTime() ) {
		error( "Event::scheduleAppendAt(): cannot schedule Event, SimTime already passed" );
		return;
	}

	// beware of moving this statement; getExecutionTime() uses it!
	scheduled = true;
		
	// trace
	getTrace() -> beginMark( this, markScheduleAt );
	getTrace() -> addTag( tagCurrent, getPartner() );
	getTrace() -> addTag( tagAbsSimTime, absoluteTime );
	getTrace() -> endMark();

	// observer
	_obsForEach( EventObserver, ScheduleAppendAt( this,  absoluteTime ) );

	// scheduling
	setExecutionTime( absoluteTime );
	env -> addSched( this );
}


void Event::removeFromSchedule() {

	if ( isScheduled() ) {
		
		env -> removeSched( this );
		
		scheduled = false;
		
		// trace
		getTrace() -> mark( this, markRemoveFromSchedule, getPartner() );
		
		// observer
		_obsForEach( EventObserver, RemoveFromSchedule( this ) );
	}
	else {
		
		error( (std::string("Event::removeFromSchedule(): Event not scheduled: " ) + 
				getLabel()).c_str() );
	}
}


void Event::execute() {

	if ( !scheduled ) {
		// error: Event has to be scheduled to be executed
		fatalError("Event::executeEvent(); event is not in schedule", -1);
	}

	// remove Event from ExecutionList
	env -> removeSched( this );
	scheduled = false;

	// trace
	getTrace() -> mark( this, markExecute, getPartner() );
	
	// observer
	_obsForEach( EventObserver, ExecuteEvent( this ) );

	// current must be set here, the old value is needed in getPartner() 
	env -> setCurrentSched( this );

	// execute the action represented by this Event
	eventAction();
}


Priority Event::setPriority( Priority newPriority ) {
	
	Priority oldPriority = priority;
	priority = newPriority;

	// trace
	getTrace() -> mark( this, markChangePriority, newPriority, oldPriority );

	// observer
	_obsAForEach( EventObserver, Priority, oldPriority, newPriority );

	if( isScheduled() ) {
		// setPriority() influences ExecutionList
		env -> inSort( this );
	}
	
	return oldPriority;
}


TraceProducer* Event::getPartner() {

	if ( env -> getCurrentSched() != 0 ) {
		
		return dynamic_cast< TraceProducer* >(env -> getCurrentSched());
	}
	else {
		
		return env;
	}
}


const MarkTypeId Event::baseMarkId = 2000;

const MarkType Event::markCreate("create", baseMarkId+1, typeid(Event));
const MarkType Event::markDestroy("destroy", baseMarkId+2, typeid(Event));

const MarkType Event::markSchedule("schedule", baseMarkId+3, typeid(Event));
const MarkType Event::markScheduleAt("scheduleAt", baseMarkId+4, typeid(Event));
const MarkType Event::markScheduleIn("scheduleIn", baseMarkId+5, typeid(Event));

const MarkType Event::markScheduleAppend("scheduleAppend", baseMarkId+6, typeid(Event));
const MarkType Event::markScheduleAppendAt("scheduleAppendAt", baseMarkId+7, typeid(Event));
const MarkType Event::markScheduleAppendIn("scheduleAppendIn", baseMarkId+8, typeid(Event));

// in case these Process functions are adapted for Events, too:
// const MarkType Event::markActivateBefore("", baseMarkId+9, typeid(Event));
// const MarkType Event::markActivateAfter("", baseMarkId+10, typeid(Event));

const MarkType Event::markRemoveFromSchedule("removeFromSchedule", baseMarkId+11, typeid(Event));
const MarkType Event::markExecute("execute", baseMarkId+12, typeid(Event));

const MarkType Event::markChangePriority("changePriority", baseMarkId+13, typeid(Event));
const MarkType Event::markChangeExecutionTime("changeExecutionTime", baseMarkId+14, typeid(Event));

const TagId Event::baseTagId = 2000;

const Tag Event::tagCurrent( "current" );
const Tag Event::tagAbsSimTime( "absolute Time" );
const Tag Event::tagRelSimTime( "relative Time" );
const Tag Event::tagPartner( "partner" );

/* 
const Tag Event::tagCurrent(baseTagId+1);
const Tag Event::tagAbsSimTime(baseTagId+2);
const Tag Event::tagRelSimTime(baseTagId+3);
const Tag Event::tagPartner(baseTagId+4);
*/
