//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Timer.cpp

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Implementation of classes in Timer.h

	\sa Timer.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/


#include <iostream>
#include <odemx/synchronization/Timer.h>
#include <odemx/util/ErrorHandling.h>

#include <odemx/Debug.h>

using namespace odemx;

Timer::Timer( Simulation* sim, Label l, TimerObserver* to ):
	Event( sim, l ),
	Memo( sim, Memo::TIMER ),
	Observable<TimerObserver>( to )
	{
		setPriority( MAXPRIO );
		
		// trace in Event
		
		// observer
		_obsForEach(TimerObserver, Create(this));
	}


Timer::~Timer() {
	
		// trace in Event
	
		// observer
		_obsForEach(TimerObserver, Destroy(this));
	}


bool Timer::isSet() {

	return isScheduled();
}


bool Timer::available() {

	return !isScheduled();
}


void Timer::setIn( SimTime t ) {

	// cannot schedule Timer
	if ( t < 0 ) {
		error( "Timer::set(): cannot schedule Timer, invalid SimTime < 0" );
		return;
	}

	// trace
	getTrace() -> mark( this, markSet );

	// observer
	_obsForEach( TimerObserver, Set( this, t ) );

	// scheduling
	scheduleIn( t );
}


void Timer::setAt( SimTime t ) {
	
	// cannot schedule Event
	if ( t < getCurrentTime() ) {
		error( "Timer::setAbsoluteTime(): cannot schedule Timer, SimTime already passed" );
		return;
	}

	// scheduling
	setIn( t - getCurrentTime() );
}


void Timer::reset( SimTime t ) {

	if ( ! isScheduled() ) {
		// warning: Timers should only be reset when scheduled		
		warning( "Timer::reset(): scheduling idle Timer by reset" );
	}
	
	// cannot schedule Timer
	if ( t < 0 ) {
		error( "Timer::set(): cannot schedule Timer, invalid SimTime < 0" );
		return;
	}

	// trace
	getTrace() -> mark( this, markReset );

	// observer
	_obsForEach( TimerObserver, Reset( this, getExecutionTime(), t ) );

	// scheduling
	scheduleIn( t );
}


void Timer::eventAction() {

	// trace
	getTrace() -> mark( this, markTimeout );

	// observer
	_obsForEach( TimerObserver, Timeout( this ) ); 

	if ( memoryList.empty() ) { 

		// timers should have processes associated with them
		warning( "Timer::eventAction(): Memory empty, cannot alert any Process" );
	}
	else {

		// wake processes
		Memo::alert();
	}
}


void Timer::stop() {

	// trace
	getTrace() -> mark( this, markStop );
	
	// observer
	_obsForEach( TimerObserver, Stop( this ) );

	// error check done in removeFromSchedule
	removeFromSchedule();
	
	// remove all waiting processes from Memo
	Memo::eraseMemory();
}


bool Timer::registerProcess( Process* p ) {
		
	// observer
	_obsForEach( TimerObserver, RegisterProcess( this, p ) );
	
	// trace in Memo
	return Memo::remember( p );
}


bool Timer::removeProcess( Process* p ) {
		
	// observer
	_obsForEach( TimerObserver, RemoveProcess( this, p ) );
	
	// trace in Memo
	return Memo::forget( p );
}


const MarkTypeId Timer::baseMarkId = 2500;
const MarkType Timer::markSet( "set", baseMarkId+1, typeid(Timer) );
const MarkType Timer::markReset( "reset", baseMarkId+2, typeid(Timer) );
const MarkType Timer::markStop( "stop", baseMarkId+3, typeid(Timer) );
const MarkType Timer::markTimeout( "timeout", baseMarkId+5, typeid(Timer) );
