//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Idist.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/05/09

	\brief Declaration of random discrete number generators

	\sa Idist.cpp

	Discrete random number generators

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_IDIST_INCLUDED
#define ODEMX_IDIST_INCLUDED

#include <odemx/random/Dist.h>

namespace odemx {
	/**
		\defgroup idist Integer distributions
		\ingroup random

		A group of random number generators which produce integer
		random numbers in different distributions.
	*/

	/** \interface Idist
		
		\ingroup idist

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Interface for integer distributions

		<!-- [\note {notes}]* -->

		<!-- [\sa {references to other classes}]* -->

		%Idist declares the 'int sample()' method for discrete 
		distribution classes. All derived classes implement this
		method to provide the random numbers in different distributions.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Idist: public Dist {
	public:
		/**
			\brief Construction

			\param title
				Label of the generator

			\param c
				pointer to a DistContext object
		*/
		Idist(DistContext* c, Label title) : Dist(c, title) {};
		virtual ~Idist() {}

		/**
			\brief Get next random number

			This function returns the next random number.
			It is implemented in derived classes which generate
			random numbers in different distributions.
		*/
		virtual int sample()=0;
	};

	/** \class Randint
		
		\ingroup idist

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Uniform distributed discrete random numbers

		\note Randint from ODEM

		\note supports Report

		<!-- [\sa {references to other classes}]* -->

		Randint provides a series of uniform distributed 
		integer random numbers in the interval [a, b). The
		parameters a and b are set in the constructor.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Randint: public Idist {
	public:
		/**
			\brief Construction

			\param title
				Label of the generator

			\param c
				pointer to DistContext object

			\param ra
				lower bound - a

			\param rb
				upper bound - b

			The parameter a and b define the interval [a, b)
			of the uniform distributed random numbers generated
			by %Randint.
		*/
		Randint(DistContext* c, Label title, int ra, int rb);
		// Destruction
		virtual ~Randint(){};

		/// Get next random number
		virtual int sample();

		/// Get parameter a
		int getA() {return a;}
		/// Get parameter b
		int getB() {return b;}
		/// Get number of uses
		int getUses() {return uses;}

		/// Reset statistics
		virtual void reset() {StatisticObject::reset(); uses=0;}

		/// Generate report
		virtual void report(Report* r);

	protected:
		int a, b, uses;

	private:
		double zyqspan;
	};

	/** \class Poisson
		
		\ingroup idist

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Poisson distributed discrete random numbers

		\note Poisson from ODEM

		\note supports Report

		<!-- [\sa {references to other classes}]* -->

		Poisson provides a series of Poisson-distributed
		discrete random numbers. The parameter \c pa
		divergence is set in the constructor.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Poisson: public Idist {
	public:
		/**
			\brief Construction

			\param title
				Label of the generator

			\param c
				pointer to DistContext object

			\param pa
				divergence

			The parameter pa defines the divergence
			of the Poisson-distribution generated by
			this random number generator.
		*/
		Poisson(DistContext* c, Label title, double pa);
		/// Destruction
		virtual ~Poisson(){};

		/// Get next random number
		virtual int sample();

		/// Get parameter pa
		double getA() {return a;}
		/// Get number of uses
		int getUses() {return uses;}

		/// Reset statistics
		virtual void reset() {StatisticObject::reset(); uses=0;}

		/// Generate report
		virtual void report(Report* r);

	protected:
		double a;
		int uses;
	};


	/** \class Draw
		
		\ingroup idist

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Random series of 0 and 1 

		\note Draw from ODEM

		\note supports Report

		<!-- [\sa {references to other classes}]* -->

		Draw provides a random series of 1 and 0. The
		probability of the 1 against a 0 is set in the
		constructor.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Draw: public Idist {
	public:
		/**
			\brief Construction

			\param title
				Label of the generator

			\param c
				pointer to DistContext object

			\param da
				probability of a 1 against a 0

			The parameter \c da defines the probability
			of a 1 against a 0.
		*/
		Draw(DistContext* c, Label title, double da);
		/// Destruction
		virtual ~Draw(){};

		/// Get next random number
		virtual int sample();

		/// Get parameter \c da
		double getA() {return a;}
		/// Get number of uses
		int getUses() {return uses;}

		/// Reset statistics
		virtual void reset() {StatisticObject::reset(); uses=0;}

		/// Generate report
		virtual void report(Report* r);

	protected:
		double a;
		int uses;
	};

	/** \class Iconst
		
		\ingroup idist

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief Constant number generator

		\note Iconst from ODEM

		\note supports Report

		<!-- [\sa {references to other classes}]* -->

		Iconst provides a constant integer number through the
		Idist interface. The constant number is set in the 
		constructor.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Iconst: public Idist {
	public:
		/**
			\brief Construction

			\param title
				Label of the generator

			\param c
				RNG Context

			\param i
				constant integer returned by sample()

			The parameter i defines the number returned
			by this pseudo generator.
		*/
		Iconst(DistContext* c, Label title, int i);
		/// Destruction
		virtual ~Iconst(){};

		/// Get number
		virtual int sample(){ uses++; return a;};

		/// Get the parameter i
		int getA() {return a;}
		/// Get number of uses
		int getUses() {return uses;}

		/// Reset statistics
		virtual void reset() {StatisticObject::reset(); uses=0;}

		/// Generate report
		virtual void report(Report* r);

	private:
		const int a;
		int uses;
	};
}

#endif

