//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file TransmissionMedium.cpp

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Implementation of classes in TransmissionMedium.h

	\sa TransmissionMedium.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/



#include <odemx/protocol/ProtocolMessage.h>
#include <odemx/protocol/TransmissionMedium.h>
#include <odemx/protocol/TransmissionEvents.h>

using namespace odemx;

TransmissionMedium::TransmissionMedium( ProtocolSimulation* sim, Label l,
										TransmissionMediumObserver* o ): 
	NetTopology( sim ),
	Observable<TransmissionMediumObserver>( o ),
	env( sim ),
	messageCount( 0 ),
	broadcastCount( 0 ),
	unicastCount( 0 ),
	failedBroadcastCount( 0 ),
	failedUnicastCount( 0 ),
	collisionCount( 0 )
	{
		DefLabeledObject::setLabel( sim, l );
		
		sim -> setTransmissionMedium( this );
		
		//trace
		getTrace() -> mark( this, markCreate );
		
		// observer
		_obsForEach( TransmissionMediumObserver, Create( this ) );
	}
	
TransmissionMedium::~TransmissionMedium()
	{
		//trace
		getTrace() -> mark( this, markDestroy );
		
		// observer
		_obsForEach( TransmissionMediumObserver, Destroy( this ) );
	}
 

bool TransmissionMedium::transmit( ProtocolMessage* m ) {

	//trace
	getTrace() -> mark( this, markTransmit, m -> toString().c_str() );
	
	// observer
	_obsForEach( TransmissionMediumObserver, Transmit( this, m ) );
	
	++messageCount;
	
	m -> computeTransmissionTime( getCurrentTime() );
	
	// only unicasts can have a connection test here,
	// broadcasts are always transmitted,
	// failure is reported by BroadcastTransmission
	if ( m -> isUnicast() ) {	

		NetNode* sender = m -> getSendNode();
		NetNode* receiver = m -> getReceiveNode();
		
		// error handling
		if ( noConnectionSet( sender, receiver ) ) {

			return false;
		}
		
		++unicastCount;
		
		// send message to receiveNode
		scheduleUnicastTransmission( m );
		
	}	
	else { // m is broadcast

		++broadcastCount;
		
		// send message to all neighbors
		scheduleBroadcastTransmission( m );
	}

	return true;
}


bool TransmissionMedium::noConnectionSet(  NetNode* sendNode, NetNode* receiveNode ) {

	if ( ! NetTopology::connected( sendNode, receiveNode ) ) {
		
		// dist variable between sender and receiver not initialized 
		error( "TransmissionMedium::noConnectionSet(): no dist variable for node pair" );
		return true;
	}
	
	return false;
}


bool TransmissionMedium::badConnectivity(  NetNode* sendNode, NetNode* receiveNode ) {
	
	if ( ! connectivity[ sendNode ][ receiveNode ].loss -> sample() )
		return true;

	return false;
}


void TransmissionMedium::scheduleUnicastTransmission( ProtocolMessage* m ) {
	
	SimTime t = m -> getTransmissionTime();
	
	//trace
	getTrace() -> beginMark( this, markScheduleUnicast );
	getTrace() -> addTag( tagFrom, dynamic_cast<TraceProducer*>(m -> getSendNode()) );
	getTrace() -> addTag( tagTo, dynamic_cast<TraceProducer*>(m -> getReceiveNode()) );
	getTrace() -> addTag( tagAt, t );
	getTrace() -> endMark( m -> toString().c_str() ); // msg handed over as comment
	
	// observer
	_obsForEach( TransmissionMediumObserver, ScheduleUnicast( this, m, t ) );
	
	Event* unicast = new UnicastTransmission( getSimulation(), "Unicast",  m );
	
	unicast -> scheduleAppendAt( t );
}


void TransmissionMedium::scheduleBroadcastTransmission( ProtocolMessage* m ) {

	SimTime t = m -> getTransmissionTime();
	
	//trace
	getTrace() -> beginMark( this, markScheduleBroadcast );
	getTrace() -> addTag( tagFrom, dynamic_cast<TraceProducer*>(m -> getSendNode()) );
	getTrace() -> addTag( tagAt, t );
	getTrace() -> endMark( m -> toString().c_str() ); // msg handed over as comment
	
	// observer
	_obsForEach( TransmissionMediumObserver, ScheduleBroadcast( this, m, t ) );
		
	Event* broadcast = new BroadcastTransmission( getSimulation(), "Broadcast",  m );
	
	broadcast -> scheduleAppendAt( t );
}


Trace* TransmissionMedium::getTrace() const { 
	return env;
}


ProtocolSimulation* TransmissionMedium::getSimulation() const {
	return env;
}


SimTime TransmissionMedium::getCurrentTime() const {
	return env -> getTime();
}


void TransmissionMedium::updateCollisionCount( unsigned int c ) {
	
	collisionCount += c;
}


const MarkTypeId TransmissionMedium::baseMarkId = 5100;

const MarkType TransmissionMedium::markCreate( "create medium", baseMarkId+1, typeid(TransmissionMedium) );
const MarkType TransmissionMedium::markDestroy( "destroy medium", baseMarkId+2, typeid(TransmissionMedium) );
const MarkType TransmissionMedium::markTransmit( "transmit", baseMarkId+3, typeid(TransmissionMedium) );
const MarkType TransmissionMedium::markScheduleUnicast( "schedule unicast", baseMarkId+6, typeid(TransmissionMedium) );
const MarkType TransmissionMedium::markScheduleBroadcast( "schedule broadcast", baseMarkId+7, typeid(TransmissionMedium) );

const TagId TransmissionMedium::baseTagId = 5100;

const Tag TransmissionMedium::tagFrom( "from" );
const Tag TransmissionMedium::tagTo( "to" );
const Tag TransmissionMedium::tagAt( "at time" );
