//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file ProtocolMessage.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Declaration of odemx::ProtocolMessage

	\sa ProtocolMessage.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/


#ifndef ODEMX_PROTOCOL_MESSAGE_INCLUDED
#define ODEMX_PROTOCOL_MESSAGE_INCLUDED

#include <odemx/util/SimTime.h>
#include <odemx/protocol/NetNode.h>
#include <string>

namespace odemx {

	/** \interface ProtocolMessage
	
		\ingroup protocol
	
		\author Ronald Kluth
		<!-- [\author <author>]* -->
	
		\brief %ProtocolMessage is a message interface required for protocol simulations 
	
		\sa TransmissionMedium, ProtocolSimulation
	
		%ProtocolMessage is the abstract base class (i.e. interface) for all 
		protocol messages to be sent via TransmissionMedium, whose implementation
		requires the existence of certain functions to automate the Event
		of sending a message.
		
		<!-- [\warning {warnings}]* -->
	
		<!-- [\todo {todos for this file}]* -->
	
		<!-- [\bug {bug description}]* -->
	
		<!-- [\test {testcase description}]* -->
	
		\since 2.0
	*/
	class ProtocolMessage: public PortElement { 
	
	public:

		/// Destruction
		virtual ~ProtocolMessage() {}
		
		/**
			\brief check for unicast or broadcast message
			\return
				\c true if the ProtocolMessage is a unicast type 
			
			For performance reasons there is a distinction between
			broadcast messages such as route requests, for example, and
			unicast messages such as data packets, which only take one
			chosen route through the network, even though in reality 
			the sending may always be in form of broadcasts, as is
			the case with wireless networks.
		*/
		virtual bool isUnicast() const = 0;
		
		/** \name Timing functions
		
			These functions guarantee access several required
			message features that are used in the scheduling of
			a message TransmissionEvent.
			
			@{ 
		*/
		/// get the duration a message needs to be transmitted between nodes
		virtual SimTime getTransmissionDuration() const = 0;
		/// determine the transmission time from current time plus duration
		virtual void computeTransmissionTime( SimTime currentTime );
		/// set the transmission time of a message
		void setTransmissionTime( SimTime t );
		
	public:
		/// get the transmission time of a message
		SimTime getTransmissionTime() const;
		//@}
		
		/** \name Node functions
			
			These functions provide access to the nodes involved in
			transmitting messages.
			
			@{
		*/
		/// get the original sender of a node
		virtual NetNode* getSource() const = 0;
		/// get the final destination node
		virtual NetNode* getDestination() const = 0;
		/// get the last sender that forwarded this message
		virtual NetNode* getSendNode() const = 0;
		/// get the next hop, i.e. the next node to receive the message
		virtual NetNode* getReceiveNode() const = 0;
		//@}
		
		/**
			\brief defines how to make a complete copy of the message
			\return
				a side-effect-free copy of the ProtocolMessage
				
			This function is required for broadcast messages. Whenever
			message types contain pointers or references as member data,
			a deep copy needs to be made, meaning the referenced objects
			also need to be copied so the pointers do not point to the 
			same dynamic object. Since this function is used internally by
			the library, it has to be provided.
		*/
		virtual ProtocolMessage* deepCopy() = 0;
		
		/// provide a string representation of this message 
		virtual std::string toString() = 0;
		
	private:
		
		/// time at which a TransmissionEvent will be triggered for this message
		SimTime transmissionTime;
	};


}

#endif /* ODEMX_PROTOCOL_MESSAGE_INCLUDED */
