//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file NetNode.cpp

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Implementation of classes in NetNode.h

	\sa NetNode.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/



#include <odemx/protocol/NetNode.h>
#include <odemx/protocol/ProtocolSimulation.h>
#include <odemx/protocol/ProtocolMessage.h>

using namespace odemx;

NetNode::NetNode( ProtocolSimulation* sim, Label l, NetNodeObserver* o ): 
	Process( sim, l ),
	Observable<NetNodeObserver>( o )
	{
		inPort = new PortHead( sim, (string(l)+" Input Port").c_str() );
		outPort = new PortTail( sim, (string(l)+" Output Port").c_str() );

		ProtocolLayerList& layers = sim -> getLayerList();
		
		if ( layers.size() == 0 )
			fatalError( "NetNode::NetNode(): No protocol layers defined, cannot create entities", -1 );
		
		initEntities( layers );
	}
	

NetNode::~NetNode() 
	{
		// port destructor will not remove the port 
		// if there is another process connected to head/tail
		delete inPort;
		delete outPort;
	}


// initialize this node's layer entities
void NetNode::initEntities( ProtocolLayerList& layers ) {

	ProtocolEntity* previous = 0;
	ProtocolEntity* created = 0;
	ProtocolLayer* currentLayer = 0;
	
	ProtocolLayerList::iterator iter;
	for( iter = layers.begin(); iter != layers.end(); ++iter ) {
		
		currentLayer = (*iter);
		
		// fill map entities for this node
		created = currentLayer -> createNewEntity( this );  
		
		// remember entity corresponding to its layer
		entities[ currentLayer ] = created;

		// link this node's entities
		if ( previous ) {
			previous -> setBottomEntity( created );
			created -> setTopEntity( previous );
		}
		
		// remember the new Entity for the next loop
		previous = created;
		
		created -> hold();
	}
	
	topEntity = entities[ layers.front() ];
	bottomEntity = entities[ layers.back() ];
}



void NetNode::transmit( PortElement* e ) {
	
	//trace
	getTrace() -> mark( this, markTransmit );
	// observer
	_obsForEach( NetNodeObserver, Transmit( this, e ) );
	
	getTopLayerEntity() -> putTop( e );
}


PortTail* NetNode::inputPort() {
	return inPort -> getTail();
}


PortHead* NetNode::outputPort() {
	return outPort -> getHead();
}


PortHead* NetNode::inputPortHead() {
	return inPort;
}


ProtocolEntity* NetNode::getTopLayerEntity() {
	return topEntity;
}


ProtocolEntity* NetNode::getBottomLayerEntity() {
	return bottomEntity;
}


ProtocolLayerManager* NetNode::getLayerManager() {
	
	return dynamic_cast<ProtocolSimulation*>(getSimulation()) -> getLayerManager();
}


ProtocolLayerList& NetNode::getLayerList() {
	return getLayerManager() -> getLayerList();
}


ProtocolEntity* NetNode::getEntityByLayer( ProtocolLayer* layer ) {
	
	if ( dynamic_cast<ProtocolEntity*>(entities[ layer ]) )
		return entities[ layer ];
	
	return 0;
}


ProtocolEntity* NetNode::getEntityByLayerName( Label l ) {
	
	ProtocolLayer* layer = getLayerManager() -> getLayerByLabel( l );
	
	if ( dynamic_cast<ProtocolEntity*>(entities[ layer ]) )
		return entities[ layer ];
	
	return 0;
}


void NetNode::input( PortElement* msg ) {
	
	//trace
	getTrace() -> mark( this, markInput );
	// observer
	_obsForEach( NetNodeObserver, Input( this, msg ) );
	
	inPort -> getTail() -> put( msg );	
}


PortElement* NetNode::readInputPort() {
	return inPort -> get();	
}


void NetNode::output( PortElement* msg ) {
	
	//trace
	getTrace() -> mark( this, markOutput );
	// observer
	_obsForEach( NetNodeObserver, Output( this, msg ) );
	
	outPort -> put( msg );	
}


PortElement* NetNode::readOutputPort() {
	return outPort -> getHead() -> get();	
}


void NetNode::receiveFromNet( ProtocolMessage* msg ) {
	
	//trace
	getTrace() -> mark( this, markReceiveFromNet, msg -> toString().c_str() );
	
	// observer
	_obsForEach( NetNodeObserver, ReceiveFromNet( this, msg ) );
	
	// retrieve lowest level entity
	ProtocolEntity* physicalEntity = getBottomLayerEntity();
	
	// send message
	physicalEntity -> putBottom( (PortElement*) msg );
}


const MarkTypeId NetNode::baseMarkId = 5000;

const MarkType NetNode::markInput( "input", baseMarkId+1, typeid(NetNode) );
const MarkType NetNode::markOutput( "output", baseMarkId+2, typeid(NetNode) );

const MarkType NetNode::markReceiveFromNet( "received message", baseMarkId+3, typeid(NetNode) );
const MarkType NetNode::markTransmit( "initiate transmission", baseMarkId+4, typeid(NetNode) );
