//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file CoroutineContext.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/02/04

	\brief Implementation of CoroutineContext
	
	\sa CoroutineContext.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#if defined(_MSC_VER) && defined(FORCE_STACK_IMPL)
#pragma warning(disable : 4530)
#endif

#include <odemx/coroutine/CoroutineContext.h>
#include <odemx/coroutine/Coroutine.h>
#include <odemx/util/ErrorHandling.h>

using namespace odemx;

CoroutineContext::CoroutineContext(const char* label, CoroutineContextObserver* o) :
	Observable<CoroutineContextObserver>(o),
	activeCoroutine(0),
	numberOfCoroutines(0)
{
	// set label
	DefLabeledObject::setLabel(this, label);

	_obsForEach(CoroutineContextObserver, Create(this));

#ifdef HAVE_FIBERS
	fiber = 0;
#else
	base = 0;
	last = 0;
	oldStack = 0;
	checksum = 0;
#endif
}

CoroutineContext::~CoroutineContext() {
	_obsForEach(CoroutineContextObserver, Destroy(this));

	if (numberOfCoroutines!=0) {
		// Warning: not all coroutines are finished
		warning("~CoroutineContext(); not all coroutines of this context have finished");
	}
	
	if (oldStack != 0) {
		delete[] oldStack;
	}
}

void CoroutineContext::switchTo() {
	_obsForEach(CoroutineContextObserver, SwitchTo(this, getActiveCoroutine()));

	if (getActiveCoroutine()==0)
		return;

#ifdef HAVE_FIBERS
	setActiveCoroutine(0);
	FiberSwitch(fiber);
#else
	Coroutine* oldActive = setActiveCoroutine(0);
	if (oldActive->saveEnvironment())
		return;

	restoreEnvironment();
#endif
}

Coroutine* CoroutineContext::getActiveCoroutine() {
	return activeCoroutine;
}

Coroutine* CoroutineContext::setActiveCoroutine(Coroutine* c) {
	Coroutine* oldActiveCoroutine = activeCoroutine;
	activeCoroutine = c;

	_obsAForEach(CoroutineContextObserver, ActiveCoroutine, oldActiveCoroutine, activeCoroutine);

	return oldActiveCoroutine;
}

void CoroutineContext::beginCoroutine(Coroutine* cr) {
	assert(cr->getContext()==this);
	++numberOfCoroutines;
}

void CoroutineContext::endCoroutine(Coroutine* cr) {
	assert(cr->getContext()==this);
	--numberOfCoroutines;
}

#ifdef HAVE_FIBERS
void CoroutineContext::saveFiber() {
	// Initialise Fibers
	if (initFibers) {
		ConvertThreadToFiber(0);
		initFibers=false;
	}

	// get context
	fiber=GetCurrentFiber();

	if (fiber==0) {
		// Error: could not get a fiber
		fatalError("saveFiber(); CoroutineContext saveFiber failed: could not get a fiber",-1);
	}
}

bool CoroutineContext::initFibers = true;
#else

int odemx::calcChecksum(StackAdress begin, unsigned long size) {
	int sum = 0;	

	while (size > 0) {
		sum += *begin;
		--size;
		++begin;
	}

	return sum;
}

bool CoroutineContext::saveEnvironment() {
	char a;
	char b;
	last = &a;

	// save stack
	if (oldStack!=0)
		delete[] oldStack;

	if (&a < &b) {
		// stack is growing bottom up
		if (last>base) {
			// context stack and coroutine stack are overlapping;
			// save stack
			oldStack = new char[last-base];
#ifdef sparc
			asm("t 3");
#endif
			memcpy(oldStack, base, (unsigned int)(last-base));
			checksum = calcChecksum(base, last-base);
		}
	} else {
		// stack is growing top down
		if (last<base) {
			// context stack and coroutine stack are overlapping;
			// save stack		
			oldStack = new char[base-last];
#ifdef sparc
			asm("t 3");
#endif
			memcpy(oldStack, last, (unsigned int)(base-last));
			checksum = calcChecksum(last, base-last);
		}
	}

	// save environment
	if (setjmp(env)!=0) {
		return true;
	} else {
		return false;
	}
}

void CoroutineContext::restoreEnvironment() {
	char c;	 

	// restore stack
	if (oldStack!=0) {
		if (base < last) {
			// stack is growing bottom up
			// Recurse to get space in stack for context and 
			// to have a valid this-pointer after memcpy().
			if (&c-(Coroutine::adress2-Coroutine::adress1) < last)
				restoreEnvironment();

			if (checksum!=calcChecksum(oldStack, last-base)) {
				// Error: stack corrupted
				fatalError("restoreEnvironment(); CoroutineContext restoration failed: stack corrupted",-1);
			}
#ifdef sparc
			asm("t 3");
#endif
			memcpy(base, oldStack, (unsigned int)(last-base));
		} else {
			// stack is growing top down
			// Recurse to get space in stack for context and 
			// to have a valid this-pointer after memcpy().
			if (&c+(Coroutine::adress1-Coroutine::adress2) > last)
				restoreEnvironment();

			if (checksum!=calcChecksum(oldStack, base-last)) {
				// Error: stack corrupted
				fatalError("restoreEnvironment(); CoroutineContext restoration failed: stack corrupted",-1);
			}
#ifdef sparc
			asm("t 3");
#endif
			memcpy(last, oldStack, (unsigned int)(base-last));
		}
	}

	// restore environment
	longjmp(env, 1);
}
#endif

DefaultContext* defContext=0;

/**	
	\return	pointer to the DefaultContext
	\relates DefaultContext

	default CoroutineContext for convinience	
*/
CoroutineContext* odemx::getDefaultContext() {
	if (defContext==0) {
		defContext=new DefaultContext;

		if (defContext==0) {
			// Error: could not create DefaultContext
			fatalError("getDefaultContext(); could not create DefaultContext", -1);
		}
	}

	return defContext;
}

