//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Simulation.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/02/22

	\brief Implementation of classes in Simulation.h

	\sa Simulation.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <odemx/base/Event.h>
#include <odemx/base/Process.h>
#include <odemx/base/Simulation.h>
#include <odemx/util/ErrorHandling.h>

// #include <odemx/Debug.h>

using namespace odemx;

Simulation::Simulation(Label l, SimulationObserver* o) :
	ExecutionList(o),
	CoroutineContext(l, o),
	Observable<SimulationObserver>(o),
	isInitialized(false),
	isStopped(false),
	executingEvent(false),
	simMode(UNTILEMPTY),
	endTime(0),
	now(0),
	current(0),
	currentSched(0)
{
	_obsForEach(SimulationObserver, Create(this));
}

Simulation::~Simulation() {
	_obsForEach(SimulationObserver, Destroy(this));
}

void Simulation::run() {
	// trace
	getTrace()->mark(this, markRun);

	// observer
	_obsForEach(SimulationObserver, Run(this));

	init();
	simMode = UNTILEMPTY;
	compSimulation(false);
}

void Simulation::step() {
	// trace
	getTrace()->mark(this, markStep);

	// observer
	_obsForEach(SimulationObserver, Step(this));

	init();
	simMode = STEPPING;
	compSimulation(false);
}

void Simulation::runUntil(SimTime t) {
	// trace
	getTrace()->mark(this, markRunUntil);

	// observer
	_obsForEach(SimulationObserver, RunUntil(this, t));

	init();
	simMode = UNTILTIME;
	endTime = t;
	compSimulation(false);
}

void Simulation::exitSimulation() {
	// trace
	getTrace()->mark(this, markExitCall);

	// observer
	_obsForEach(SimulationObserver, ExitSimulation(this));

	isStopped = true;
	switchTo();
}

bool Simulation::isFinished() {
	return isInitialized && (isStopped || ExecutionList::isEmpty());
}

bool Simulation::isExecutingEvent() {
	return executingEvent;	
}

void Simulation::init() {
	if (!isInitialized) {
		// trace
		getTrace()->mark(this, markInit);

		// observer
		_obsForEach(SimulationObserver, Initialization(this));

		initSimulation();
		isInitialized=true;
	}
}


// execute next Sched object
void Simulation::exec() {
	
	Sched* next = ExecutionList::getNextSched();

	// check time consistency
	if ( next -> getExecutionTime() < now ) {
		// Error: ExecutionList corrupted
		error( "exec(); ExecutionList corrupted: invalid execution time" );
		
		printExecutionList();
		
		fatalError( "exec(); ExecutionList corrupted: ending simulation", -1 );
	}

	// trace
	getTrace()->mark( this, markTime, next -> getExecutionTime(), now );

	// observer
	_obsAForEach( SimulationObserver, Time, now, next -> getExecutionTime() );

	// update current SimTime
	now = next -> getExecutionTime();
	
	// execute according to type of scheduled object
	switch ( next -> getSchedType() ) {
		
		case Sched::PROCESS:

			// trace
			getTrace() -> mark( this, markExecProc, next );
		
			// observer
			_obsForEach( SimulationObserver, ExecuteProcess( this, dynamic_cast< Process* >(next) ) );

			next -> execute();
			break;

		case Sched::EVENT:

			// trace
			getTrace() -> mark( this, markExecEvent, next );
		
			// observer
			_obsForEach( SimulationObserver, ExecuteEvent( this, dynamic_cast< Event* >(next) ) );
		
			// ensure complete execution, even when scheduling processes
			executingEvent = true;
			next -> execute();
			executingEvent = false;
			
			break;

		default:
			error( "Simulation::exec(): cannot determine SchedType" );
	}
}


Process* Simulation::getCurrentProcess() {
	return current;
}

Sched* Simulation::getCurrentSched() {
	return currentSched;	
}

std::list<Process*>& Simulation::getCreatedProcesses() {
	return created;
}

std::list<Process*>& Simulation::getRunnableProcesses() {
	return runnable;
}

std::list<Process*>& Simulation::getIdleProcesses() {
	return idle;
}

std::list<Process*>& Simulation::getTerminatedProcesses() {
	return terminated;
}

void Simulation::setCurrentSched( Sched* s ) {

	currentSched = s;	
}


void Simulation::setProcessAsCurrent(Process* p) {
	// trace
	getTrace()->mark(this, markCurrProc, p);

	// observer
	_obsAForEach(SimulationObserver, CurrentProcess, current, p);

	current = p;
	currentSched = p;
}

void Simulation::setProcessAsCreated(Process* p) {
	//changeProcessList(p, CREATED);
}

void Simulation::setProcessAsRunnable(Process* p) {
	//changeProcessList(p, RUNNABLE);
}

void Simulation::setProcessAsIdle(Process* p) {
	//changeProcessList(p, IDLE);
}

void Simulation::setProcessAsTerminated(Process* p) {
	//changeProcessList(p, TERMINATED);
}

void Simulation::removeProcessFromLists(Process* p) {
	created.remove(p);
	runnable.remove(p);
	idle.remove(p);
	terminated.remove(p);
}

void Simulation::changeProcessList(Process* p, List l) {
	removeProcessFromLists(p);

	_obsForEach(SimulationObserver, ChangeProcessList(this, p, l));

	switch (l) {
	case CREATED:
		created.push_back(p);
		break;
	case RUNNABLE:
		runnable.push_back(p);
		break;
	case IDLE:
		idle.push_back(p);
		break;
	case TERMINATED:
		terminated.push_back(p);
		break;
	}
}

// compute simulation according to simulation mode
void Simulation::compSimulation(bool inSimulation/* = true*/) {

	if ( executingEvent )
		return;

	// execute all events scheduled before the next process object
	while( !ExecutionList::isEmpty() &&
			getNextSched() -> getSchedType() == Sched::EVENT ) {

		switch( simMode ) {

			case UNTILEMPTY:

				if ( !isStopped )
					exec();
					
				break;

			case STEPPING:

				if ( !isStopped ) {
					
					if ( inSimulation )
						switchTo();	// called from process scheduling fct, jump to main()
									// remember this position for the current process
					else
						exec();		// coming from main(), execute Event
					return;			// do nothing else after returning
				}
				break;

			case UNTILTIME:
				if ( !isStopped && getTime() <= endTime ) {
	
					// check next Sched to avoid Event firing
					if ( getNextSched() -> getExecutionTime() > endTime ) {
						if ( inSimulation )
							switchTo();	// called from Process scheduling 
						else
							return;		// return to main()
					}
					else {
						exec(); // execTime <= endTime
					}
				}
			}
		} // done with events	

	// execute process object if there is one scheduled
	if (!inSimulation) {
		// if called from main program [run(), step(), runUntil()]:
		// start next Process object
		switch(simMode) {
		case UNTILEMPTY:
		case STEPPING:
			if (!ExecutionList::isEmpty() && !isStopped)
				exec();
			break;
		case UNTILTIME:
			if (!ExecutionList::isEmpty() && getTime() <= endTime && !isStopped) {

				// check next Sched to avoid Event firing
				if ( getNextSched() -> getExecutionTime() > endTime ) {

					break; // return to main
				}
				else {
					exec();
				}		
			}
		}
	} else {
		if (!isInitialized)
			return;

		// if called from Process: handle different simulation modes
		switch(simMode) {
		case UNTILEMPTY:
			// continue with next Process in Execution list
			if (!ExecutionList::isEmpty() && !isStopped)
				exec();
			else
				switchTo(); // return to main program
			break;
		case STEPPING:
			// return to main program
			switchTo();
			break;
		case UNTILTIME:
			// continue if possible
			if (!ExecutionList::isEmpty() && getTime() <= endTime && !isStopped) {
			
				// check next Sched to avoid Event firing
				if ( getNextSched() -> getExecutionTime() > endTime ) {
				
					switchTo(); // return to main
				}
				else {
					exec();
				}
			}
			else
				switchTo(); // return to main program
			break;
		}
	}
}


// ============== DEFAULT SIMULATION / MARK INITIALIZATION ====================


DefaultSimulation* defSim=0;

/**
	\return	pointer to the DefaultSimulation
	\relates DefaultSimulation

	default Simulation for convenience
*/
Simulation* odemx::getDefaultSimulation() {
	if (defSim==0) {
		defSim=new DefaultSimulation;

		if (defSim==0) {
			// Error: could not create DefaultSimulation
			fatalError("getDefaultSimulation(); could not create DefaultSimulation", -1);
		}
	}

	return defSim;
}

const MarkTypeId Simulation::baseMarkId = 1000;

const MarkType Simulation::markInit = MarkType("init", baseMarkId+1, typeid(Simulation));
const MarkType Simulation::markExitCall = MarkType("exit call", baseMarkId+2, typeid(Simulation));
const MarkType Simulation::markRun = MarkType("run", baseMarkId+3, typeid(Simulation));
const MarkType Simulation::markStep = MarkType("step", baseMarkId+4, typeid(Simulation));
const MarkType Simulation::markRunUntil = MarkType("run until", baseMarkId+5, typeid(Simulation));
const MarkType Simulation::markExecProc = MarkType("execute process", baseMarkId+6, typeid(Simulation));
const MarkType Simulation::markExecEvent = MarkType("execute event", baseMarkId+7, typeid(Simulation));
const MarkType Simulation::markCurrProc = MarkType("current process", baseMarkId+8, typeid(Simulation));
const MarkType Simulation::markTime = MarkType("time", baseMarkId+9, typeid(Simulation));
