// Roboterposition mit Constraints bestimmen
// -----------------------------------------
//
// In diesem Programm werden aus vom Image-Prozessor bermittelten Daten
// Constraints definiert und dann darauf Constraintpropagierung angewendet,
// um die Roboterposition zu bestimmen.
//
// Die Lsungsstrategie fr das Constraintproblem ist in Funktionen so allgemein
// implementiert, dass - ohne sie zu ndern - sie auf beliebige Constraints ber
// eine beliebigt Anzahl von Variablen angewendet werden knnen. Auch wurde viel
// Wert auf die Beachtung des Falles gelebt, dass die maximale lokal-konsistente
// Einschrnkung zu einer Wertebereichsteilmenge fhren kann, die aus mehreren
// nicht zusammenhngenden Komponenten bestehen kann.
//
// Erst im Abschnitt "Funktionen zur Lsung des spziellen durch die drei
// Bilder gegebenen Problems" am Ende dieses Files wird die Lsung des
// speziellen Problems aus der Aufgabenstellung angegangen.
//
// Um die Lsungsstrategie ein wenig zu beeinflussen, knnen die
// folgenden Parameter hier im Quelltext gendert werden (Sie sind als
// Klassenvariablen der alles einschlieenden Klasse robocopo implementiert):
// - suchTiefe
// - maxTeilTiefe
// Zur Erklrung dieser Parameter siehe die Dokumentation oder die betreffende
// Stelle in diesem Quelltext.
//
//
// Um das Programm zu compilieren (in der Kommandoeile):
// > javac robocopo.java
//
// Um das Programm zu starten (in der Kommandoeile):
//   -normaler Modus:
//    > java robocopo 
//
//   -debug-Modus, d.h. das Programm gibt aus, was es gerade tut, oder hnliches:
//    > java robocopo 1
//      (Jeder Eingabeparameter bewirkt den debug-Modus)
// 
// 
// Mitwirkende:
//  Debora Baumann (198687)
//  Patrick Ladig (197987)
//  Jonathan Fischer (194193)

//In dieser Klasse ist der gesamte Lsungsversuch implementiert.
public class robocopo {

	//////////////////////////////////////////////////////////
	//Konstanten und Variablen
	//////////////////////////////////////////////////////////
	
	// Konstanten aus der Aufgabenstellung
	//
	// Winkel
	public static final double KAM_WINKEL_H = 55.;
	public static final double KAM_WINKEL_V = 44.;
	//Fehler fr Peilungswinkel
	public static final double WINKEL_FEHLER = .25;
	//Mae der Bilder (in mm)
	public static final double BILD_HOEHE = 59;
	public static final double BILD_BREITE = 76;
	// Koordinaten der Fahnen
	public static final double FAHNE_KOORD [][] = 
		{{-1350.,1950.},
		{1350.,1950.},
		{-1350.,-1950.},
		{1350.,-1950.}};
	// Spielfeldmae
	public static final double SPFLD_RAND_MAX [] = {3000.,2000.};
	public static final double SPFLD_RAND_MIN [] = {-3000.,-2000.};
	// Mae der Fahnen
	public static final double FAHNE_HOEHE = 400.;
	public static final double FAHNE_BREITE = 100.;
	public static final double BALLDURCHM = 90.;

	// Numerierungen, um Indizes fr Arrays zu haben
	// 
	// Numerierung der Fahnen
	public static final int ROSA_BLAU = 0;
	public static final int ROSA_GELB = 1;
	public static final int BLAU_ROSA = 2;
	public static final int GELB_ROSA = 3;
	// Numerierung der Koordinaten auf dem Spielfeld
	// Gleichzeitig Numerierung der Variablen, auf der die Constraints definiert sind
	public static final int X = 0;
	public static final int Y = 1;
	//Die Anzahl der Variablen
	public static final int VAR_ANZAHL = 2;
	//Nuerierung von "obere Grenze" und "untere Grenze"
	public static final int MAX = 1;
	public static final int MIN = 0;
	// Numerierung der drei Bilder
	public static final int LINKS = 0;
	public static final int VORN = 1;
	public static final int RECHTS = 2;

	// Daten "vom Image-Prozessor"
	// 
	// welche Fahne ist auf welchem Bild zu sehen
	public static int fahneFarbe [] = {BLAU_ROSA, ROSA_BLAU, ROSA_GELB};
	// Hhe der Fahne auf dem Bild (in mm, da wir keine Angaben in Pixeln machen knnen)
	public static double fahneHoehe [] = {11.7,16.5,14.5};

	//Daten, die nicht "vom Image-Prozessor kommen", die aber vom "User" hier gendert werden knnen:
	//
	//Tiefe der Suche nach Lsungspunkten in einem durch Constraintpropagation
	//(bei der hier keine Lsungen verloren gehen knnen) weitestmglich eingeschrnkten Gebiet.
	//Das Minimum ist 1
	public static int suchTiefe = 3;
	//Gibt, an wie oft eine Menge, in der mit der Suche keine Lsung gefunden wurde, fr eine
	//weitere Constraintpropagation geteilt werden soll.
	//Das Minimum ist 1, dann wird nicht geteilt.
	public static int maxTeilTiefe = 2;

	//in dieser Klasse "globale" Variablen:
	//
	//Hier wird gespeichert, ob debugging-Ausgaben gemacht werden (zum Programmierfehler finden)
	public static boolean debug = false;


	//////////////////////////////////////////////////////////
	//Die Darstellung der Constraints in Klassen
	//////////////////////////////////////////////////////////
	
	// Unsere Darstellung von Constraints: Die Constraint-Klasse bildet die Basisklasse.
	// Die Constraint-Objekte bilden eine einfach verkettete Liste.
	//
	// Ein Objekt der Klasse Constraint, welches nicht Objekt einer von Constraint
	// abgeleiteten Klasse ist, schrnkt den Wertebereich nicht ein, d.h. ist immer erfllt.
	// Zur Definition "echter" Constraint erweitern wir diese Klasse, indem wir abgeleitete
	// Klassen implementieren. Diese berladen die Memberfunktionen (siehe nchster Paragraph).
	//
	// Ein Constraint wird hier durch zwei Memberfunktionen definiert:
	// - maxLokalKonsistEinschr() erzeugt eine maximale lokal-konsistente Einschrnkung
	// 	aus einer gegebenen Wertebereichmenge.
	// - erfuellt() prft, ob das Constraint in einem gegebenen Punkt erfllt ist.
	public static class Constraint{
		//static-Variablen
		//
		//Anzahl der Constraints
		public static int anzahl = 0;
		//Referenz auf das erste Constraint
		public static Constraint erster = null;
		//Referenz auf das letzte Constraint
		private static Constraint letzter = null;

		//non-static-Variablen
		//
		//Refenz auf das nchste Constraint
		public Constraint next;
		//Nummer des Constraints (wird nur fr die Debug-Ausgabe gebraucht).
		public int nummer;

		//Konstruktor
		public Constraint (){
			//Jetzt gibt es ein weiteres Constraint
			nummer = ++anzahl;
			//Constraint in die einfach verkettete Liste einbauen
			if (erster == null)
				erster = this;
			else
				letzter.next = this;
			letzter = this;
			this.next = erster;
		}

		//Diese Funktion erzeugt die maximale lokal-konsistente Einschrnkung
		//
		//Erklrung der Eingabe "double eingabe [][][]"
		//Dieser 3-dimensionale Array definiert die Wertebereichmenge, die weiter eingeschrnkt
		//	werden soll:
		//erste Dimension: Die Variable (z.B. X, Y)
		//zweite Dimension: Die Konponente (Evtl. ist der Wertebereich der Variablen nicht (mehr)
		//	zusammenhngend, sondern besteht aus mehreren Intervallen ("Komponenten").)
		//dritte Dimension: MAX und MIN (die obere oder untere Grenze des Intervalls)
		//also z.B. eingabe[X][2][MIN] ist die untere Grenze fr die dritte Komponente der Variablen X.
		//
		//Rckgabewert[0] gibt an, ob eingabe gendert wurde, also ob weiter eingeschrnkt wurde.
		//Rckgabewert[1] gibt an, ob die Einschrnkung nichtleer ist.
		//Falls Rckgabewert[0]==false, kann Rckgabewert[1] ohne Prfung auf true gesetzt sein.
		//	(Man nimmt hier also an, dass die Eingabe einen nichtleeren Bereich definiert hat.)
		//
		//Diese Funktion ndert den Eingabeparameter durch "call-by-reference".
		//D.h. nach Aufruf der Funktion zeigt die Refernezvariable "eingabe" auf die maximale
		//lokal-konsistente Einschrnkung durch dieses Constraint.
		public boolean [] maxLokalKonsistEinschr (double eingabe [][][]){
			return new boolean [] {false, true};
		}
		
		//Diese Funktion berprft, ob in einem Punkt das Constraint erfllt ist oder nicht:
		//
		//Eingabe ist ein double-Array mit den Koordinaten des Punktes.
		//Er hat so viele Eintrge, wie es Variablen gibt.
		public boolean erfuellt (double [] punkt){
			return true;
		}
	}

	//Diese Klasse definiert eine bestimmte Sorte von Constraints.
	//Sie sind ber die Variablen X und Y (Spielfeldposition) defiert.
	//
	//Wenn innen=true, so erfllt ein Punkt (X,Y) dieses Constraint genau dann, wenn
	//er innerhalb eines bestimmten Kreises liegt, also wenn (X-MX)^2 + (Y-MY)^2 <= R^2
	//fr Mittelpunktkoordinaten MX und MY und Radius R
	//
	//Wenn innen=false, so erfllen nur Punkte auerhalb des Kreises dieses Constraint.
	public static class Kreis extends Constraint {
		//Die Koordinaten des Mittelpunktes
		private double mittelpunkt [];
		//Der Radius
		private double radius;
		//Hier wird gespeichert, ob das Constraint das innere oder das uere des Kreises ist.
		private boolean innen;

		//Der Konstruktor
		public Kreis (double mittelpunkt [], double radius, boolean innen ){
			//Aufruf des Constraint-Konstruktors
			super();
			//Belegung der Membervariablen
			this.mittelpunkt = mittelpunkt;
			this.radius = radius;
			this.innen = innen;
			//Diese Ausgabe erscheint im Debug-Modus
			debugAusgabe("Constraint "+nummer+": Kreis ("+(innen ? "innen" : "auen")+"); Mittelpunkt = ("+mittelpunkt[X]+","+mittelpunkt[Y]+"); Radius = "+radius);
		}

		//Fr Erklrung von Sinn, Ein- und Ausgabe dieser Funktion siehe
		//Kommentar zu Constraint.maxLokalKonsistEinschr()
		public boolean [] maxLokalKonsistEinschr (double eingabe [][][]){
			debugAusgabe("maxLokalKonsistEinschr() durch Constraint "+ this.nummer +" auf Gebiet:");
			//gibt im debug-Modus das Eingabegebiet in lesbarer Form aus
			gebietAusgabe(eingabe);
			//soll speichern, ob die Eingabe gendert wird.
			boolean eingabeGeaendert = false;
			//Im Fall "innen" wird die Einschrnkung im folgenden Block implementiert.
			if (innen){
				//das Minimum aus der geringsten Entfernung zur jeweiligen Koordinate des Mittelpunkts
				//un dem Radius des Kreises ermittelt. Dieser Wert wird im Array "min" gespeichert und
				//dann zur Berechnung der Einschrnkung der jeweils anderen Variable genutzt.
				//
				//min wird zunchst initialisiert
				double [] min = {radius, radius};
				//Schleife ber die Variablen. Auch im Folgenden deutet die Laufvariable "var" auf
				//eine Schleife ber alle Variablen hin.
				//Diese Schleife luft nur ber die ersten zwei Variablen, weil dieses Constraint nur
				//X und Y betrifft.
				for (int var = X; var<=Y ; ++var)
					//Schleife ber die Komponenten. (einzelne Intervalle des Wertebereiches)
					//Auch im Folgenden deutet die Laufvariable "komp"
					//auf eine Schleife ber alle Komponenten hin.
					for (int komp = 0; komp<eingabe[var].length; ++komp)
						if (eingabe[var][komp][MIN] <= mittelpunkt[var])
							//In diesem Fall liegt die Mittelpunktkoordinate innerhalb der
							//Variablenkomponente, die Entfernung ist also 0
							if (eingabe[var][komp][MAX] >= mittelpunkt[var])
								min[var] = 0.;
							//In diesem Fall liegt die Komponente (das Iintervall) komplett 
							//unterhalb" der mittelpunktkoordinate, d.h. die obere Grenze
							//gibt den Abstand dieses Intervalls zum Mittelpunkt an.
							else
								min[var] = Math.min(min[var], mittelpunkt[var]-eingabe[var][komp][MAX]);
						//In diesem Fall ist nur die untere Intervallsgrenze relevant, weil das
						//Intervall vollstndig "oberhalb" der Mittelpunktkoordinate liegt.
						else
							min[var] = Math.min(min[var], eingabe[var][komp][MIN]-mittelpunkt[var]);
				//Nun wird die Einschrnkung der Variablen ermittelt. Die Grenze hngt direkt von dem
				//Wert der anderen Variablen in "min" ab.
				for (int var = X; var<=Y ; ++var){
					//Da es hier um eine Beziehung zwischen den beiden Variablen geht, brauchen
					//wir nicht nur die Nummer "var" der Variablen, sondern auch die der anderen.
					int andereVar = (var + 1) % 2;
					//Die Anzahl der Komponenten (Intervalle) von "var" im gegebenen Wertebereich
					int kompAnzahl = eingabe[var].length;
					//Dieser Array soll zunchst die durch dieses Constraint eingeschrnkten
					//Komponenten enthalten, spter werden dann nur die nichtleeren Komponenten
					//bernommen.
					double [][] ausgabeMax = new double [kompAnzahl][];
					//zhlt die NICHTLEEREN Komponenten
					int komponentenZaehler = 0;
					// komp durchluft alle Komponenten des gegebenen (einzuschrnkenden) Wertebereiches.
					for (int komp = 0; komp<kompAnzahl; ++komp){
						//Hier soll der Kandidat fr eine Ausgabekomponente gespeichert werden.
						//Spter wird dann geprft, ob er nichtleer ist.
						double [] ausgabekandidat = new double [2];
						//Die untere Grenze der Komponente wird durch die Kreisgleichung eingeschrnkt
						ausgabekandidat[MIN] = Math.max(eingabe[var][komp][MIN], mittelpunkt[var] - Math.sqrt(radius*radius-min[andereVar]*min[andereVar]));
						//Die obere Grenze der Komponente wird durch die Kreisgleichung eingeschrnkt
						ausgabekandidat[MAX] = Math.min(eingabe[var][komp][MAX], mittelpunkt[var] + Math.sqrt(radius*radius-min[andereVar]*min[andereVar]));
						//Mittels einer eigens implementierten Funktion wird geprft, ob eine echte
						//Einschrnkung statt gefunden hat, oder ob (andernfalls) fr jede Belegung
						//der Variablen aus diesem Intervall eine entsprechende Erfllende Belegung
						//der anderen Variable existiert, so dass die Komponente gleich blieb.
						if (!gleich (ausgabekandidat,eingabe[var][komp])){
							//Fall die Ausgabekomponente nicht gleich der Eingabekomponente ist,
							//wurde sie gendert.
							eingabeGeaendert = true;
						}
						//Falls die Komponente nicht leer ist, wird sie in die Ausgabe aufgenommen.
						if (ausgabekandidat[MIN] <= ausgabekandidat[MAX]){
							ausgabeMax[komponentenZaehler++] = ausgabekandidat;	
						}
					}
					//Hier wird der Eingabeparameter gendert. Er ist wegen "call by value" durch die
					//aufrufende Funktion nutzbar.) Fr jede Variable wird ein neuer Array angelegt,
					//da ja ein pahr Komponenten verschwunden sein knnten.
					eingabe[var] = new double[komponentenZaehler][];
					//Falls keine Komponente aufgenommen wurde (also alle eingeschrnkten Komponenten
					//leer waren), wird die Information zurckgegeben, dass die Einschrnkung leer ist.
					if (komponentenZaehler == 0){
						return new boolean [] {eingabeGeaendert,false};
					}
					//Ansosnten werden die nichtleeren Komponenten bernommen.
					for (int komp = 0; komp<komponentenZaehler; ++komp)
						eingabe[var][komp] = ausgabeMax[komp];
				}
			}
			//Im folgenden Block erfolgt die Einschrnkung fr den Fall "auen"
			else{
				//So wie im Fall "innen" die Einschrnkung einer Variablen von der Entfernung des Werte-
				//berechs der anderen von der entsprechenden Mittelpunktkoordinate entscheident ist,
				//ist es hier das Maximum, wobei eine Variable berhaupt nicht eingeschrnkt wird, wenn
				//die andere eine Belegung besitzt, deren Abstand zur entsprechenden Mittelpunktkoordinate
				//mindestens der Radius ist.
				//Die relevanten maximalen Abstnde zum Mittelpunkt werden im Array "max" gespeichert.
				double [] max = {0.,0.};
				//Wir betrachten wieder jede Komponente (also jeden zusammenhngenden Teil des Wertebereichs)
				//jeder Variablen.
				for (int var = X; var <=Y; ++var)
					for (int komp = 0; komp<eingabe[var].length; ++komp)
						//max[var] == -1 bedeutet hier, dass bereits eine Belegung der Variablen
						//gefunden wurde, die um mindestens den Radius vom Mittelpunkt entfern ist.
						//In diesem Fall ist das genaue Maximum nicht relevant und muss also nicht
						//mehre berechnet werden.
						if (max[var] != -1)
							//Wir berprfen, ob genau der eben beschriebene Fall eintritt
							//und setzten ggf max[var]:=-1
							if (eingabe[var][komp][MAX] >= mittelpunkt[var]+radius)
								max[var] = -1;
							else if (eingabe[var][komp][MIN] <= mittelpunkt[var]-radius)
								max[var] = -1;
							//Andernfalls wird das genaue Maximum berechnet.
							else
								max[var] = Math.max(max[var], Math.max(mittelpunkt[var]-eingabe[var][komp][MIN], eingabe[var][komp][MAX]-mittelpunkt[var]));
				//Im Folgenden wird die Einschrnkung einer jeden Variablen mit Hilfe des "max"-Wertes der
				//anderen Variable berechnet.
				for (int var = X; var <=Y; ++var){
					//Wieder brauchen wir die Nummer der jeweils anderen Variable
					int andereVar = (var + 1) % 2;
					//Falls der oben beschriebene Fall eintritt, wird die Variable nicht eingeschrnkt.
					if (max[andereVar] != -1){
						//Die Anzahl der Komponenten (Intervalle) von "var" im gegebenen Wertebereich
						int kompAnzahl = eingabe[var].length;
						//Dieser Array soll zunchst die durch dieses Constraint eingeschrnkten
						//Komponenten enthalten, spter werden dann nur die nichtleeren Komponenten
						//bernommen.
						//Im Fall "auen" kann es nicht nur passieren, dass Komponenten leer werden,
						//sondern auch, dass hchstens eine Komponente in zwei geteilt wird.
						//Geht man zu Beginn der Constraintpropagation von einer zusammenhngenden
						//Domain aller Variablen aus, so ist dies die einzeige Stelle, an der
						//mehrere Komponenten entstehen knnen. Das ist auch der Grund, warum wir
						//berhaupt jede Variable Komponentenweise betrachten.
						double [][] ausgabeMax = new double [kompAnzahl + 1][];
						//zhlt die nichtleeren neuen Komponenten.
						int komponentenZaehler = 0;
						//Wieder wird jede Komponente des gegebenen Wertebereichs betrachtet
						for (int komp = 0; komp < kompAnzahl; ++komp){
							//Falls die Komponente geteilt werden sollte, knnen zwei nichtleere
							//Komponenten entstehen. Wir legen einfach jedesmal zwei Ausgabekandi-
							//daten fest, von denen einer stets leer sein wird, falls die Kompo-
							//nente nicht geteilt wird.
							double [][] ausgabekandidaten = new double [2][2];
							//Der erste Kandidat fr eine neue Komponente wird nur in der oberen
							//Grenze eingeschrnkt. Das geschieht durch die Kreisgleichung in
							//Abhngigkeit von der anderen Variable.
							ausgabekandidaten[0][MIN] = eingabe[var][komp][MIN];
							ausgabekandidaten[0][MAX] = Math.min(eingabe[var][komp][MAX],mittelpunkt[var] - Math.sqrt(radius*radius-max[andereVar]*max[andereVar]));
							//Der zweite Kandidat fr eine neue Komponente wird nur in der unteren
							//Grenze eingeschrnkt.
							ausgabekandidaten[1][MIN] = Math.max(eingabe[var][komp][MIN],mittelpunkt[var] + Math.sqrt(radius*radius-max[andereVar]*max[andereVar]));
							ausgabekandidaten[1][MAX] = eingabe[var][komp][MAX];
							//nur falls eine Eingabekomponente ungeleich beider zugehriger
							//Ausgabekandidaten ist, ist die Einschrnkung kleiner als der
							//Eingangswertebereich.
							if (!gleich (ausgabekandidaten[0], eingabe[var][komp]) && !gleich (ausgabekandidaten[1], eingabe[var][komp]))
								eingabeGeaendert = true;
							//Die nichtleeren Ausgabekandidaten werden fr die Ausgabe bernommen.
							for (int i = 0; i<2; ++i)
								if (ausgabekandidaten[i][MIN] <= ausgabekandidaten[i][MAX])
									ausgabeMax[komponentenZaehler++] = ausgabekandidaten[i];
						}
						//Der Eingabeparameter wird gendert. Bei jeder Variable knnte sich die
						//Anzahl der verbliebenen Komponenten gendert haben.
						eingabe[var] = new double [komponentenZaehler][];
						//Falls keine Komponente aufgenommen wurde (also alle eingeschrnkten
						//Komponenten leer waren), wird die Information zurckgegeben, dass die
						//Einschrnkung leer ist.
						if (komponentenZaehler == 0)
							return new boolean [] {eingabeGeaendert,false};
						//Ansosnten werden die nichtleeren Komponenten bernommen.
						for(int komp = 0; komp<komponentenZaehler; ++komp)
							eingabe[var][komp] = ausgabeMax[komp];
					}
				}
			}
			//Falls diese Anweisung erreicht wird, wurde "eingabe" auf einen neuen, evtl. genderten Wert
			//gesetzt und jede Variable hat mindestens eine nichtleere Komponente.
			return new boolean [] {eingabeGeaendert, true};
		}

		//Fr Erklrung von Sinn, Ein- und Ausgabe dieser Funktion siehe
		//Kommentar zu Constraint.erfuellt().
		public boolean erfuellt (double [] punkt){
			//Es wird geprft, ob der gegebene Punkt innerhalb des Kreises liegt.
			if ((punkt[X]-mittelpunkt[X])*(punkt[X]-mittelpunkt[X])+(punkt[Y]-mittelpunkt[Y])*(punkt[Y]-mittelpunkt[Y]) < radius*radius)
				return innen;
			else
				return !innen;
		}
	}


	//////////////////////////////////////////////////////////
	//Funktionen zur Ausgabe von Informationen im debug-Modus
	//////////////////////////////////////////////////////////


	//Diese Funktion gibt den Text auf dem Terminal aus, falls debug auf true gesetzt ist.
	public static void debugAusgabe (String text){
		if (debug)
			System.out.println ("||"+text);
	}

	//Diese Funktion gibt den in "gebiet" definierten Werteberech aller Variablen
	//in leidlich lesbarer Form auf dem Terminal aus, wenn debug auf true gesetzt ist.
	//Die Grenzen der Werteintervalle der Variablen ist wie im Eingabeparameter "eingabe"
	//der Funktion Constraint.maxLokalKonsistEinschr() gespeichert.
	public static void gebietAusgabe(double[][][] gebiet){
		if (debug){
			//Die Werte werden Variable fr Variable ausgegeben.
			for (int var = 0; var < VAR_ANZAHL; ++var)
				//Alle Komponenten werden bercksichtigt
				for (int komp = 0; komp < gebiet[var].length; ++komp)
					System.out.print("|| var["+var+"] aus ["+gebiet[var][komp][MIN]+","+gebiet[var][komp][MAX]+"] ");
			System.out.println ("");
		}
	}


	//////////////////////////////////////////////////////////
	//Allgemeine Funktionen zum lsen des Constraintproblems
	//////////////////////////////////////////////////////////
	//
	//Streng genommen, ist auch die in der Constraint-Klasse implementierten Memberfunktion
	//maxLokalKonsistEinschr integraler Bestandteil des Constraintpropagationsalgorithmus.
	//Doch diese Funktion muss (in unserem Konzept) fr jede Sorte Constraint extra implementiert
	//werden. Die Folgenden Funktionen nutzen maxLokalKonsistEinschr, knnen aber mit allen
	//Objekten der Klasse Constraint umgehen.
	//

	//In dieser Funktion wird der Constraintpropagationsalgorithmus implementiert. Dabei wird
	//als Einschrnkung immer die maximale lokal-konsistente Einschrnkung gewhlt, die durch
	//die Funktion maxLokalKonsistEinschr() des jeweiligen Constraint-Objekts geliefert wird.
	//
	//Zur Erklrung der Bedeutung des Eingabeparameters "eingabe" und der Ausgabeparameter
	//siehe Kommentar zur Funktion Constraint.maxLokalKonsistEinschr().
	//Auch hier wird das Ziel des Eingabeparameters gendert.
	public static boolean [] maxLokalKonsistPropagation (double eingabe [][][]){
		//im debug-Modus erfolgt eine Ausgabe zum Nachvollziehen des Programmverlaufs
		debugAusgabe("maxLokalKonsistPropagation() auf Gebiet:");
		gebietAusgabe(eingabe);
		//Hier wird gespeichert, ob die Eingabe berhaupt von irgendeinem Constraint verndert wurde.
		boolean eingabeGeaendert = false;
		//Wird dieser Zhler so gro wie die Anzahl der Constraints, so wird die Eingabe durch
		//kein Constraint weiter eingeschrnkt.
		int nichtGeaendertZaehler = 0;
		//Referenzvariable auf das aktuelle Constraint
		Constraint aktuellerConstraint = Constraint.erster;
		//Solange noch eine nderung des Wertebereiches durch maxLokalKonsistEinschr() mglich ist ...
		while (nichtGeaendertZaehler < Constraint.anzahl){
			//Ruf der Objektfunktion fr die maximale lokal-konsistente Einschrnkung.
			boolean [] einschrErgebnis = aktuellerConstraint.maxLokalKonsistEinschr(eingabe);
			//Falls echt eingeschrnkt wurde ...
			if (einschrErgebnis[0]){
				debugAusgabe("... gendert");
				//wurde also innerhalb dieser Funktion die Eingabe gendert.
				eingabeGeaendert = true;
				//Ab jetzt ist wieder eine echte Einschrnkung durch jedes Constraint prinzipiell
				//mglich.
				nichtGeaendertZaehler = 0;
				//Falls die Einschrnkung die leere Menge ist, wird genau diese Information zurckgegeben.
				if (!einschrErgebnis[1])
					return einschrErgebnis;
			}
			//Falls die Einschrnkung nicht echt war...
			else{
				debugAusgabe("... nicht gendert");
				//wird der Zhler der nicht ndernden Constraintanwendungen um 1 erhht.
				++nichtGeaendertZaehler;
			}
			//Das nchste Contraint kommt an die Reihe.
			aktuellerConstraint = aktuellerConstraint.next;
		}
		//Es wurde soweit wie mglich durch wiederholte maximale lokal-konsistente Einschrnkung
		//eingeschrnkt. Der Wertebereich ist dabei fr keine Variable leer geworden.
		return new boolean [] {eingabeGeaendert, true};
	}

	//Diese Funktion ruft zunchst nur maxLokalKonsistPropagation() auf. Wenn diese einen
	//Wertebereich zurckgibt, des aus mehr als nur einer Komponente besteht, so soll fr
	//jede (zusammenhngende) Komponente (also fr jedes zusammenhngende Stck auf dem
	//Spielfeld) wiederum maxLokalKonsistPropagation() angewendet werden, etc.
	//Dieser Vorgang wird sich nicht unendlich oft wiederholen, da es nur endlich viele
	//Constraints gibt, die Wertebereichkomponenten teilen knnen. Jedes dieser Constraints
	//wird nur endlich viele Teile erzeugen: Jedes Kreis(auen)-Constraint kann den Wertebereich
	//in hchstens vier Teile teilen. Daher ist der Vorgang endlich.
	//
	//Wir nennen diese Lsungsstrategie "induktive Constraintpropagation".
	//
	//Diese Funktion ist rekursiv definiert.
	//
	//Zur Erklrung der Parameter:
	//Der Eingabeparameter "eingabe[][][][][]" speichert wieder den Wertebereich (also wieder
	//eine Teilmenge aus dem Kreuzprodukt aller Domains). Allerdings werden nun nicht mehr fr jede
	//Variable mehrere Komponenten betrachtet, sonern allgemeiner mehrere Komponenten des Wertebereiches,
	//also beliebige rechteckige Stcke des Spielfeldes (in unserer Aufgabe.) Die Funktion kann
	//aber immer nur auf eine solche Komponente (ein solches Rechteck) angewendet werden.
	//
	//Es sollen also beim Aufruf von rekursKompPropagation() die folgenden formalen Bedingungen
	//fr "eingabe" erfllt sein:
	//Die erste Dimension ist 1. Sie dient lediglich zur Indirektion, also als eine Art pointer,
	//	um mittels "call by value" das Ziel verndern zu knnen, also um in diesem Fall die zweite
	//	Dimension ndern zu knnen.
	//Die zweite Dimension ist ebenfalls 1. Sie kann aber innerhalb der Funktion gendert werden.
	//	Sie bezeichnet dann die erhaltenen Komponenten. Die Funktion kann also nur auf eine
	//	Komponente angewand werden, kann aber mehrere Wertebereichskomponenten zurck geben.
	//Die dritte Dimension gibt die Variable an.
	//Die vierte Dimension gibt die Komponente des Wertebereiches der jeweiligen Variablen an. Sie
	//	ist bei Ein- und Ausgang 1, da jede Variable nun genau eine Komponente besitzt.
	//Die fnfte Dimension gibt an, ob es sich bei dem Wert um die untere oder obere Intervallgrenze handelt
	//
	//Beispiel: "eingabe[0][1][X][0][MIN]==20" bedeutet, dass die untere Grenze der Variablen X der
	//zweiten Komponente des Wertebereiches gleich 20 ist. (Bei Aufruf der Funktion darf es allerdings
	//nur eine Wertebereichskomponente geben).
	//
	//Der Rckgabewert ist einfach die Anzahl der erhaltenen Komponenten, also gleich eingabe[0].length
	//nach Durchlaufen der Funktion.
	public static int rekursKompPropagation (double eingabe [][][][][]){
		debugAusgabe("rekursKompPropagation() auf Gebiet:");
		gebietAusgabe(eingabe[0][0]);
		//zunchst wird maxLokalKonsistPropagation() aufgerufen
		boolean [] propagErgebnis = maxLokalKonsistPropagation (eingabe[0][0]);
		//Falls nichts gendert wurde, heit das, dass es bei einer Komponente bleibt
		if (!propagErgebnis[0]){
			debugAusgabe ("... nichts gendert ->exit");
			return 1;
		}
		//Falls das Ergebnis die leere Menge ist, wurde die Eingabemenge auf null Komponenten reduziert.
		if (!propagErgebnis[1]){
			debugAusgabe ("... Menge ist leer geworden -> exit");
			for (int var= 0; var < VAR_ANZAHL; ++var)
				eingabe[0] = new double [][][][] {};
			return 0;
		}
		//mittels einer eigens implementierten Funktion werden die Komponenten des Wertebereiches
		//insgesamt gezhlt
		int kompAnzahl = kompZaehlen(eingabe[0][0]);
		debugAusgabe ("Anzahl der Komponenten: "+kompAnzahl);
		//Sollte die Anzahl 1 sein, so wurde das Gebiet (der Wertebereich) auf einen bereich eingeschrnkt,
		//der auch nur aus einer Komponente besteht.
		if (kompAnzahl == 1){
			debugAusgabe("... Es bleibt bei einer Komponente: (-> exit)");
			gebietAusgabe(eingabe[0][0]);
			return 1;
		}
		gebietAusgabe(eingabe[0][0]);
		//weiter geht es nur in dem Fall, dass durch maxLokalKonsistPropagation mehrere Komponenten
		//entstanden sind.
		//Der Array wird jetzt umgeformt: Jeder Eintrag soll wieder eine typische "eingabe" mit drei
		//Dimensionen sein, fr die jede Variable nur eine Komponente hat.
		//Auf jede dieser Komponenten soll dann rekursKompPropagation() wieder angewandt werden.
		//
		//Zunchst wird "kompIndizes" erstellt, ein Array (zwei Dimensionen), der fr jede Wertebereichs-
		//komponente speichert, die wievielte Komponente welcher Variablen an ihr beteiligt ist.
		//Dabei sollen alle mglichen Kombinationen von Variablenkomponenten einmal vorkommen.
		//
		//"gleicheDavor" wird fr jede Variable die Information speichern, wie oft die Variable davor
		//hintereinander mit der gleichen Komponente vertreten war. Die Intervalle der aktuellen Variable
		//sollen dann fr jede dieser Abschnitte gleichmig verteilt werden, um letztendlich alle mglichen
		//Kombinationen genau einmal zu erhalten.
		//Die erste Variable soll auf alle Arrayeintrge von "kompIndizes" einmal verteilt werden.
		int gleicheDavor = kompAnzahl;
		int [][] kompIndizes = new int [kompAnzahl][VAR_ANZAHL];
		for (int var = 0; var < VAR_ANZAHL; ++var){
			//Anzahl der Komponenten dieser Variablen.
			int kompAnzDieserVar = eingabe[0][0][var].length;
			//Wie oft soll die gleiche Komponente dieser Variablen hintereinander vorkommen.
			int gleicheHintereinander = gleicheDavor/kompAnzDieserVar;
			//Wie oft sollen alle Komponenten hintereinander vorkommen.
			int perioden = kompAnzahl/gleicheDavor;
			//Aktualisierung von "gleicheDavor" fr die nchste Runde der for-Schleife
			gleicheDavor = gleicheHintereinander;
			//Die Indizes der Variablenkomponenten werden eingetragen.
			for (int periode = 0; periode < perioden; ++periode)
				for (int komp = 0; komp < kompAnzDieserVar; ++komp)
					for (int i = 0; i < gleicheHintereinander; ++i)
						kompIndizes[i+gleicheHintereinander*(komp+kompAnzDieserVar*periode)][var] = komp;
		}
		//"kompIndizes" soll nun genutzt werden, um den Array "eingabe" wir oben beschrieben umzuformen.
		//
		//"kompArray" soll die umgewandelten Arrays "eingabe" enthalten, die als Eingabeparameter fr die
		//rekursiven Funktionsaufrufe von rekursKompPropagation() genutzt werden sollen.
		double [][][][][][] kompArray = new double [kompAnzahl][1][1][][][];
		//Der Array kompkompAnzahl soll das Ergebnis der rekursiven Aufrufe speichern: Fr jede Komponente
		//soll gespeichert werden, wie viele Komponenten beim Aufruf entstanden sind.
		int [] kompkompAnzahl = new int [kompAnzahl];
		//Hier soll zum Schluss die Anzahl der insgesamt zum Schluss erhaltenen Wertebereichskomponenten gespeichert
		//werden, also die Summe der Eintrge von "kompkompAnzahl".
		int kompAnzahlneu = 0;
		//Wir betrachten nun jede Wertebereichkomponente aus dem Ergebnis von maxLokalKonsistPropagation()
		for (int komp = 0; komp<kompAnzahl; ++komp){
			//der entsprechende Eintrag von "kompArray" wird erzeugt
			kompArray[komp][0][0] = new double [VAR_ANZAHL][1][];
			//unter Nutzung von kompIndizes wird nun die Komponente beschrieben, d.h. "kompArray" mit
			//Werten belegt.
			for (int var = 0; var < VAR_ANZAHL; ++var)
				kompArray[komp][0][0][var][0] = eingabe[0][0][var][kompIndizes[komp][var]];
			//Nun wird auf diese Komponente erneut diese Funktion "rekursKompPropagation" angewendet.
			kompkompAnzahl[komp] = rekursKompPropagation (kompArray[komp]);
			//Die Anzahl der deadurch entstandenen Komponenten wird zu "kompAnzahlneu" hinzuaddiert, 
			//um letztendlich auf die Gesamtzahl der Komponenten bekommen.
			kompAnzahlneu += kompkompAnzahl[komp];
		}
		//Nun wird der Eingabearray wieder aus den Ergebnisarrays der rekursiven Aufrufe "zusammengebaut".
		//Zunchst wird der Array "eingabe[0]" neu erstellt, weil sich die Anzahl der Eintrge gendert
		//haben kann.
		eingabe[0] = new double [kompAnzahlneu][][][];
		//Dies soll nur eine Laufvariable sein
		int kompneu = 0;
		//In dieser Schleife werden alle "ursprnglichen" Komponenten durchlaufen, d.h. alle die,
		//auf welche rekursKompPropagation() erneut angewandt wurde.
		for (int komp = 0; komp<kompAnzahl; ++komp)
			//Wir durchlaufen nun die von diesen Aufrufen erzeugten Komponenten und hngen sie
			//an "eingabe", damit die aurfufende Funktion dann darauf zugreifen kann.
			for (int kompkomp = 0; kompkomp < kompkompAnzahl[komp]; ++kompkomp)
				eingabe[0][kompneu++] = kompArray[komp][0][kompkomp];
		//Die Anzahl der insgesamt Erzeugten Komponenten wird zurckgegeben.
		return kompAnzahlneu;
	}

	//Diese Funktion zhlt die Komponenten des eingegebenen Bereichs. Der Eingabeparameter
	//"eingabe" ist dabei wie in Constraint.maxLokalKonsistEinschr() definiert.
	//Gezhlt wird nicht die Anzahl der Intervalle pro Variable, sondern die sich daraus ergebenden
	//zusammenhngenden Komponenten des Wertebereichs insgesamt, also die Komponenten im Ruam
	//der Tupel ber alle Variablen. Hat eine Variable z.B. einen Wertebereich mit drei Intervallen,
	//so verdreifacht sich dadurch die Anzahl der Komponenten des Werteberechs insgesamt.
	public static int kompZaehlen (double [][][] eingabe){
		//sollte der Wertebereich jeder Variable nur aus einem Intervall bestehen, so
		//besteht der Wertebereich insgesamt aus genau einer Komponente.
		int kompAnzahl = 1;
		//Fr jede Variable wird kompAnzahl mit der Anzahl der Intervalle ihres Wertebereiches
		//multipliziert.
		for (int var = 0; var < VAR_ANZAHL; ++var)
			kompAnzahl *= eingabe[var].length;
		return kompAnzahl;
	}

	//Diese Funktion wird von maxLokalKonsistEinschr genutzt um festzustellen, ob eine durchgefhrte
	//Einschrnkung eine echte Teilmenge des Ausgangswertebereiches war ("echte Einschrnkung").
	//Sie berprft lediglich, ob zwei Intervalle (Komponenten) eines Variablenwertebereiches bereinstimmen.
	public static boolean gleich (double[] varKomp1, double[] varKomp2){
		//geprft wird die obere und die untere Grenze der Komponenten. Sie gelten als gleich,
		//wenn ihre Darstellung als double gleich ist. Eine Einschrnkung mit einem Effekt kleiner als
		//die double-Genauigkeit wird als keine "echte" Einschrnkung gewertet.
		return ((varKomp1[MIN] == varKomp2[MIN])&& (varKomp1[MAX] == varKomp2[MAX]));
	}

	//Diese Funktion berechnet den Mittlepunkt zwischen zwei Punkten im Raum aller Variablen.
	//Der Eingabevektor enthlt fr jede Variable (erste Dimension) zwei Werte, von denen
	//bekannt ist, dass der erste (MIN) kleiner oder gleich dem zweiten (MAX) ist.
	//
	//Diese Funktion wird von der (weiter unten aufgefhrten) Funktion loesungspunkte() aufgerufen.
	//
	//Diese Funktion ndert die Eingabe nicht, fr die Ausgabe wird ein vllig neues Objekt erstellt.
	//Der Ausgabearry enthlt die Komponenten des Mittelpunktes
	public static double [] mittelpunkt (double [][] eingabe){
		double [] ausgabe = new double [eingabe.length];
		for (int var = 0; var < VAR_ANZAHL; ++var)
			ausgabe[var] = .5*(eingabe[var][MAX]+eingabe[var][MIN]);
		return ausgabe;
	}

	//Diese Funktion wird auch von loesungspunkte (siehe unten) benutzt. Sie berprft,
	//ob ein gegebener Punkt im Raum aller Variablen alle Constraints erfllt, also ob er das
	//Constraintproblem lst.
	public static boolean loesungspunkt (double [] punkt){
		boolean ausgabe = true;
		//eine Referenzvariable fr die Constraints
		Constraint constraint = Constraint.erster;
		//Die Prfung wird - streng genommen - mit dem zweiten Constraint begonnen.
		for (int i = 0; i < Constraint.anzahl; ++i)
			ausgabe = (ausgabe &&(constraint = constraint.next).erfuellt (punkt));
//		if (ausgabe)
//			System.exit(1);
		return ausgabe;
	}
	
	//Diese Funktion soll innerhalb einer Wertebereichkomponente (also innerhalb eines
	//Rechtecks auf dem Spielfeld) nach Lsungspunkten suchen. Zunchst wird dazu der Mittel-
	//punkt untersucht.
	//Die Funktion wird dann rekursiv auf die (im Falle des Spielfeldes vier) Teile der Komponente
	//angewendet, die den Mittelpunkt als Eckpunkt haben und einen Eckpunkt der Eingabe als Eckpunkt haben.
	//
	//Der Eingabeparameter "eingabe" enthlt die Randwerte des zu untersuchenden Gebiets (also der Wertebereich-
	//komponente), wie bei der Eingabe fr die oben beschriebene Funktion mittelpunkt().
	//Der Eingabeparameter "eingabe" wird von dieser Funktion nicht gendert.
	//"tiefe" gibt die Rekursionstiefe an und wird als Abbruchkriterium fr die Rekursion benutzt. Sie bestimmt
	//gewissermaen, mit welcher Auflsung in dem gegebenen Gebiet nach Lsungspunkten gesucht wird. Eine
	//Erhhung von "tiefe" um 1 bedeutet eine Verdopplung dieser Auflsung.
	//
	//Der Ausgabeparameter ist ein Array von Punkten, d.h. jeder Eintrag ist ein Array mit den Koordinaten des
	//Punktes.
	public static double [][] loesungspunkte (double [][] eingabe, int tiefe){
		//Hier wird der Mittelpunkt gespeichert.
		double [] mpkt = mittelpunkt (eingabe);
		//Falls die Tiefe 1 ist, wird nicht weiter rekursiv gesucht, sondern nur noch der eben
		//ermittelte Mittelpunkt untersucht.
		if (tiefe <= 1)
			//Falls der Mittelpunkt Lsung ist, wird es als (in dieser Rekursionstiefe einziger)
			//Lsungspunkt ausgegeben.
			if (loesungspunkt(mpkt)){
				double[][] ausgabe = new double [1][];
				ausgabe[0] = mpkt;
				return ausgabe;
			}
			//Andernfalls ist der Ausgabearray von der Gre 0 (enthlt keine Punkte)
			else
				return new double [][] {};
		//Falls die noch zu untersuchende Tiefe grer ist als 1, wird die Funktion sich rekursiv aufrufen.
		//Zunchst wird das zu untersuchende Gebiet in Teilgebiete aufgeteilt. Die Ergebnisse der Rekursions-
		//aufrufe der Funktion sollen im Array "loesungsArray" gespeichert werden (Fr jeden Aufruf ein Eintrag).
		//In diesem Durchlauf wird es 2^VAR_ANZAHL viele Selbstaufrufe von loesungspunkte() geben.
		double [][][] loesungsArray= new double [1<<VAR_ANZAHL][][];
		//Dieser Array soll spter die Anzahlen der Lsungen enthalten, die durch die einzelnen Rekursions-
		//aufrufe der Funktion gefunden wurden.
		int [] ipunktAnzahl = new int [loesungsArray.length];
		//Diese Variable soll die Lsungspunkte insgesamt (auer ggf den Mittelpunkt) zhlen, soll also zum
		//Schluss genau die Summe der Eintrge von "ipunktAnzahl" sein.
		int punktAnzahl = 0;
		//"i" durchluft die Teilgebiete.
		for (int i = 0; i < loesungsArray.length; ++i){
			//Es wird der Array erstellt, der als Eingabeparameter fr den rekursiven Aufruf fungieren soll.
			double [][] eingabeneu = new double [VAR_ANZAHL][2];
			//Mit Hilfe von "i" und der Hilfsvariable "m" soll bestimmt werden, um welchen der Teilgebiete es
			//sich handelt. Der Wertebereich jeder Variable ist in zwei Hlften geteilt. Die Binrdarstellung
			//von "i" soll hier bestimmen, welche der beiden Hlften das aktuelle Teilgebiet betrifft.
			//Zunchst hngen wir eine Null hinten an "i" an und speichern diese Zahl in "m"
			int m = i<<1;
			//Nun wird jede Variable betrachtet.
			for (int var = 0; var < VAR_ANZAHL; ++var)
				//Fr die Variable wird nun bestimmt, ob ihre obere oder untere Hlfte der Wertebereich
				//fr das aktuelle Teilgebiet sein soll.
				//Dazu wird die letzte Ziffer von "m" gestrichen. (beim ersten Durchgang ist dies die
				//zuvor angehngte 0. Dann wird geprft, ob die nun letzte Ziffer gleich 1 ist.
				if ((m>>=1)%2 == 1){
					//Falls ja, wird die untere Hlfte des Intervalls genommen.
					eingabeneu[var][MIN] = eingabe[var][MIN];
					eingabeneu[var][MAX] = mpkt[var];
				}
				else{
					//Falls nein, ist es die obere Hlfte.
					eingabeneu[var][MAX] = eingabe[var][MAX];
					eingabeneu[var][MIN] = mpkt[var];
				}
			//Nun erfolgt der rekursive Aufruf, es wird also im aktuelle Teilgebiet nach weiteren Lsungen
			//gesucht. Diese Punkte werden in "loesungsArray" gespeichert.
			loesungsArray[i] = loesungspunkte (eingabeneu, tiefe - 1); 
			//Die Anzahl der erhaltenen Punkte wird in "ipunktAnzahl" gespeichert.
			ipunktAnzahl [i] = loesungsArray[i].length;
			//Die Zhler fr die Anzahl aller Lsungspunkte insgesamt wird um den entsprechenden Wert erhht.
			punktAnzahl += ipunktAnzahl[i];
		}
		//Diese Variable soll die Lsungspunkte durchlaufen.
		int punkt = 0;
		//Der Ausgabearray wird erstellt.
		double[][] ausgabe = null;
		//Falls der Mittelpunkt ein Lsungpunkt ist, wird er an erster Stelle im Ausgabearray aufgefhrt
		if (loesungspunkt(mpkt)){
			ausgabe = new double [punktAnzahl +1][];
			ausgabe[punkt++] = mpkt;
		}
		else
			ausgabe = new double [punktAnzahl][];
		//Die aus den rekursiven Aufrufen erhaltenen Lsungspunkte werden bertragen.
		for (int i = 0; i < loesungsArray.length; ++i)
			for (int j = 0; j < ipunktAnzahl[i]; ++j)
				ausgabe[punkt++] = loesungsArray[i][j];
		return ausgabe;
	}

	//Diese Fuktion beinhaltet die Gesamtlsungsstrategie fr das Constraintproblem:
	//Sie ruft zunchst rekursKompPropagation() auf und dann fr jede der erhaltenen Komponenten die Funktion
	//loesungspunkte(), die in den eingeschrnkten Gebieten nach Lsungspunkten suchen soll.
	//Wird kein Lsungspunkt gefunden, obwohl die Wertebereichmenge durch die Constraintpropagation
	//nicht leer geworden war, so wird diese verbliebene Menge geteilt und auf diese Teile
	//erneut die Funktion loesen() (rekursiv) angewendet. Die maximale Rekursionstiefe ist durch
	//den "globalen" Parameter maxTeilTiefe gegeben. Jedoch wird bis zu dieser Tiefe wirklich nur dann
	//geteilt, wenn nicht vorher schon Lsungspunkte gefunden wurden, oder die Wertebereichsmenge auf
	//die leere Menge reduziert wurde.
	//
	//Die Funktion wird mit einem zusammenhngenden Anfangsgebiet aufgerufen (in unserem Beispiel das
	//gesamte Spielfeld); die Abmessungen desselben werden im Array "eingabe" gespeichert.
	//Der Parameter "stiefe" legt die gewnschte Suchtiefe fr die Suche nach den Lsungspunkten fest.
	//Die genaue Bedeutung ist im Kommentar zur Funktion loesungspunkte() erklrt.
	//Der Parameter "ttiefe" speichert die noch zu bearbeitende Rekursionstiefe. Beim ersten Aufruf dieser
	//Funktion ist er maxTeilTiefe, bei den rekursiven Aufrufen hat er einen entsprechend kleineren Wert.
	//Als Parameter "platzFuerGebiete" muss nur ein double-Array mit fnf Dimensionen bergegeben werden,
	//der die erste Dimension 1 hat. Wir nutzen ihn als Referenz auf (die Referenz auf) einen double-Array
	//mit vier Dimensionen, der die durch die induktive Constraintpropagation erhaltenen Komponenten
	//speichern soll. Die extra Indirektion ist ntig, da wir noch nicht wissen, wie viele Komponenten
	//wir erhalten werden. "platzFuerGebiete" ist also gewissermaen nur fr die Ausgabe bestimmt.
	//
	//Der formale Ausgabeparameter enthlt alle gefundenen Lsungspunkte, und zwar nach den Komponenten geordnet,
	//in denen sie gefunden wurden.
	//Ausgabeparameter[4][3][X]==20 bedeutet also zum Beispiel, dass die X-Koordinate des 4. Lsungspunktes
	//aus der 5. Komponente gleich 20 ist.
	public static double [][][] loesen (double [][] eingabe, int stiefe, int ttiefe, double [][][][][] platzFuerGebiete){
		//Ausgabebefehle fr den Debug-Modus
		debugAusgabe("loesen() auf Gebiet");
		if (debug){
			double [][][] gebiet = new double [VAR_ANZAHL][][];
			for (int var = 0; var < VAR_ANZAHL; ++var){
				gebiet[var] = new double[1][];
				gebiet[var][0] = eingabe[var];
			}
			gebietAusgabe(gebiet);
		}
			
		//Es soll rekursKompPropagation() aufgerufen werden. Dafr wird zunchst der Eingabeparameter
		//erzeugt.
		double [][][][][] propEingabe = new double [1][1][eingabe.length][1][];
		//Er wird mit den Grenzen des an diese Funktion bergebenen Anfangsgebiets belegt.
		for (int var = 0; var < VAR_ANZAHL; ++var)
			propEingabe[0][0][var][0] = eingabe[var];
		//Jetzt erfolgt der Aufruf. "propKomp" speichert die Anzahl der ermittelten Komponenten.
		int propKomp = rekursKompPropagation(propEingabe);
		//Fr die Aufrufende Funktion werden die Komponenten in "platzFuerGebiete" gespeichert.
		platzFuerGebiete[0] = propEingabe[0];
		//Als nchstes soll in den erhaltenen Komponenten nach Lsungspunkten gesucht werden.
		//Dieser Ausgabearray soll am Schluss zurckgegeben werden und alle ermittelten Lsungspunkte
		//enthalten.
		double [][][] ausgabe = new double [propKomp][][];
		//Wir betrachten jede Komponente
		for (int komp = 0; komp < propKomp; ++komp){
			//Wir erzeugen fr die aktuelle Komponente den passenden Eingabeparameter fr
			//loesungspunkte(), also einen Array mit zwei Dimensionen.
			double [][] suchEingabe = new double [VAR_ANZAHL][];
			//Dieser wird nun mit den entsprechenden Werten belegt, d.h. die Referenzvariablen
			//werden auf Unterarrays von "propEingabe" zeigen.
			for (int var = 0; var < VAR_ANZAHL; ++var)
				suchEingabe[var] = propEingabe[0][komp][var][0];
			//Die in dieser Komponente gefundenen Lsungspunkte werden in den Ausgabearray geschrieben.
			ausgabe[komp] = loesungspunkte (suchEingabe, stiefe);
		}
		//Es wird geprft, ob insgesamt mindestens ein Lsungspunkt gefunden wurde.
		boolean loesungGefunden = false;
		for (int komp = 0; komp < propKomp; ++komp)
			if (ausgabe[komp].length != 0)
				loesungGefunden = true;
		//Falls ja, oder falls ttiefe keine weitere Teilung zulsst, wird das Ergebnis zurckgegeben.
		if (loesungGefunden || ttiefe <=1)
			return ausgabe;
		//Falls nicht, soll versucht werden, die Wertebereichmenge weiter einzuschrnken.
		//Dazu soll die Constraintpropagation auf Teilen der verbliebenen Komponenten
		//ausgefhrt werden. Wir teilen jede verbliebene Komponente so, dass jede Teilkomponente
		//den Mittelpunkt der Komponente als Eckpunkt hat, also wie in loesungspunkte().
		//
		//Es wird ermittelt, wie viele Rekursionsaufrufe aus diesem Durchlauf heraus statt finden sollen.
		//Jede Komponente soll, wie eben beschrieben, in 2^VAR_ANZAHL viele Teile aufgeteilt werden.
		int aufrufeAnzahl = propKomp*(1<<VAR_ANZAHL);
		//Dieser Array enthlt die Eingabeparameter "platzFuerGebiete" fr die Rekursionsaufrufe.
		//Nach den Aufrufen soll er die erhaltenen verbleibenden Komponenten enthalten.
		double [][][][][][] platzFuerGebieteArray = new double [aufrufeAnzahl][1][][][][];
		//Dieser Array soll die Rckgabewerte aus den Rekursionsaufrufen speichern, also alle gefundenen
		//Lsungspunkte.
		double [][][][] loesungspunkteArray = new double [aufrufeAnzahl][][][];
		//Dieser Array soll fr jeden Aufruf speichern, wie viele Komponenten dadurch erhalten wurden.
		int [] iKompAnzahl = new int [aufrufeAnzahl];
		//Diese Variable zhlt die erhaltenen Komponenten insgesamt, also die Summe der Eintrge von
		//"loesungspunkteArray".
		int kompAnzahl = 0;
		//Diese Variable dient als Laufvariable ber die Aufrufe
		int aufrufAktuell = 0;
		//Um die Rekursionsaufrufe zu starten, betrachten wir jede Komponente, die wir Anfangs aus
		//rekursKompPropagation() erhalten haben.
		for (int komp = 0; komp < propKomp; ++komp){
			//Wir wollen den Mittelpunkt bestimmen. Dazu mssen wir die Komponente (also das
			//"Gebiet") in die entsprechende Form zum Aufrufen der Funktion mittelpunkt() bringen.
			double [][] gebiet = new double [VAR_ANZAHL][2];
			for (int var = 0; var < VAR_ANZAHL; ++var)
				gebiet [var] = platzFuerGebiete[0][komp][var][0];
			//Der Mittelpunkt wird berechnet
			double [] mpkt = mittelpunkt(gebiet);
			//Nun wird jede Teilkomponente betrachtet
			for (int i = 0; i < (1<<VAR_ANZAHL); ++i){
				//Es wird der Array erstellt, der als Eingabeparameter fr den rekursiven Aufruf fungieren
				//soll.
				double [][] eingabeneu = new double [VAR_ANZAHL][2];
				//Mit Hilfe von "i" und der Hilfsvariable "m" soll bestimmt werden, um welchen der
				//Teilgebiete es sich handelt. Der Wertebereich jeder Variable ist in zwei Hlften geteilt.
				//Die Binrdarstellung von "i" soll hier bestimmen, welche der beiden Hlften das aktuelle
				//Teilgebiet betrifft.
				//Zunchst hngen wir eine Null hinten an "i" an und speichern diese Zahl in "m".
				int m = i<<1;
				//Nun wird jede Variable betrachtet.
				for (int var = 0; var < VAR_ANZAHL; ++var)
					//Fr die Variable wird nun bestimmt, ob ihre obere oder untere Hlfte der
					//Wertebereich fr das aktuelle Teilgebiet sein soll.
					//Dazu wird die letzte Ziffer von "m" gestrichen. (beim ersten Durchgang ist dies die
					//zuvor angehngte 0. Dann wird geprft, ob die nun letzte Ziffer gleich 1 ist.
					if ((m>>=1)%2 == 1){
						//Falls ja, wird die untere Hlfte des Intervalls genommen.
						eingabeneu[var][MIN] = gebiet[var][MIN];
						eingabeneu[var][MAX] = mpkt[var];
					}
					else{
						//Falls nein, ist es die obere Hlfte.
						eingabeneu[var][MAX] = gebiet[var][MAX];
						eingabeneu[var][MIN] = mpkt[var];
					}
				//Hier erfolgt der Selbstaufruf von loesen()
				loesungspunkteArray[aufrufAktuell] = loesen (eingabeneu, stiefe, ttiefe - 1, platzFuerGebieteArray[aufrufAktuell]); 
				//Die oben beschriebenen Parameter werden aktualisiert.
				iKompAnzahl [aufrufAktuell] = platzFuerGebieteArray[aufrufAktuell][0].length;
				kompAnzahl += iKompAnzahl[aufrufAktuell];
				aufrufAktuell++;
			}
		}
		//Nun werden die aus den rekursiven Aufrufen erhaltenen Lsungen fr die Ausgabe "zusammengebaut"
		//Zunchst erhlt "platzFuerGebiete[0]" die richtige Dimension.
		platzFuerGebiete[0] = new double [kompAnzahl][][][];
		//Diese Laufvariable soll angeben, in der wievielten (neuen, auszugebenen) Komponente wir uns befinden.
		int kompAktuell = 0;
		//Der Ausgabearray wird erstellt.
		ausgabe = new double [kompAnzahl][][];
		//Die aus den rekursiven Aufrufen erhaltenen Ergebnisse werden bertragen.
		for (int i = 0; i < aufrufeAnzahl; ++i)
			for (int j = 0; j < iKompAnzahl[i]; ++j){
				platzFuerGebiete[0][kompAktuell] = platzFuerGebieteArray[i][0][j];
				ausgabe[kompAktuell++] = loesungspunkteArray[i][j];
			}
		return ausgabe;
	}


	//////////////////////////////////////////////////////////
	//Funktionen zur Lsung des spziellen durch die drei
	//Bilder gegebenen Problems.
	//////////////////////////////////////////////////////////
	
	//Diese Funktion definiert die Constraints fr das Problem, die Roboterposition aus den drei Bildern
	//zu ermitteln. Alle Constraints sind von der Klasse Kreis.
	public static void constraintsDefinieren (){
		//die folgenden Constraints legen den Entfernung zu den Fahnen auf den drei Bildern fest.
		for (int bild = LINKS; bild <= RECHTS; ++bild)
			//Fr jedes Bild sollen zwei Constraints definiert werden.
			for (int i = 0; i<2; ++i){
				//Falls i==0, soll das Constraint mit dem greren Kreis definiert werden,
				//es beinhaltet alle inneren Punkte.
				boolean innen = (i==1);
				//Es wird der vertikale Peilungswinkel fr die Fahne auf dem Bild aus
				//der gegebenen Information berechnet, wie hoch die Fahne auf dem Bild ist.
				double hoehenwinkel = ((innen)?-1. : 1.)*WINKEL_FEHLER + KAM_WINKEL_V * fahneHoehe[bild]/BILD_HOEHE;
				//Der entsprechende Radius wird aus diesem Peilungswinkel berechet.
				double radius = FAHNE_HOEHE/(2*Math.tan(Math.PI*hoehenwinkel/360.));
				//Das Constraint wird erstellt. Wir bentigen keine Referenzvariable darauf,
				//da auf die Liste aller Constraints stets ber Constraint.erster zugegriffen
				//werden kann.
				new Kreis(FAHNE_KOORD[fahneFarbe[bild]], radius, innen);
			}
	}

	//Diese Funktion ruft die Funktion loesen() auf und gibt die Ergebnisse davon auf das Terminal aus.
	public static void loesungAusgeben (){
		// Zunchst wird die Domain fr jede Variable definiert. Es werden also die Spielfeldbegrenzungen
		// so in die Form eines Arrays gebracht, dass dieser sich als Eingabeparameter fr loesen() eignet.
		double [][] eingabe = new double [2][2];
		for (int var = X; var <=Y; ++var){
			eingabe[var][MAX] = SPFLD_RAND_MAX[var];
			eingabe[var][MIN] = SPFLD_RAND_MIN[var];
		}
		//"gebiete" ist als Array mit nur einer Komponente als der Ort gedacht, an dem nach dem
		//Aufruf die gefunden eingeschrnkten Wertebereichkomponenten fr die Ausgabe zu finden sein werden.
		double [][][][][] gebiete = new double [1][][][][];
		//Loesungsalgorithmus starten
		double [][][] punkte = loesen (eingabe, suchTiefe, maxTeilTiefe, gebiete);
	
		//Ausgabe der Ergebnisse
		//
		//zunchst der Fall, dass die Wertebereichsmenge durch die Constraintpropagation leer wurde.
		if (punkte.length == 0){
			System.out.println ("Durch (induktive) Constraintpropagation wurde die leere Menge erhalten. Es existiert keine Lsung. Das System der Constraints ist inkonsistent.");
			return;
		}
		//Andernfalls kann man die ermittelten Gebiete ausgeben.
		System.out.println ("Die (induktive) Constraintpropagation ergibt "+punkte.length+" Gebiet(e):");
		for (int gebiet = 0; gebiet < punkte.length; ++gebiet)
			System.out.println("X aus ["+gebiete[0][gebiet][X][0][MIN]+","+gebiete[0][gebiet][X][0][MAX]+"; Y aus ["+gebiete[0][gebiet][Y][0][MIN]+","+gebiete[0][gebiet][Y][0][MAX]+"]");
		//Es wird geprft, ob ein Lsungspunkt gefunden wurde.
		boolean loesungGefunden = false;
		for (int komp = 0; komp < punkte.length; ++komp)
			if (punkte[komp].length != 0)
				loesungGefunden = true;
		//Falls nicht, wird diese Information auf das Terminal ausgegeben.
		if (!loesungGefunden){
			System.out.println ("Die Wertebereichmenge wurde durch die (induktive) Constraintpropagation nicht leer, aber in den verbleibenden Gebieten konnten keine Punkte gefunden werden, die allen Constraints gengen.");
			return;
		}
		//Falls Lsungspunkte gefunden wurden, werden sie ausgegeben.
		System.out.println ("Es wurden die folgenden Lsungspunkte gefunden (X|Y):");
		for (int komp = 0; komp < punkte.length; ++komp)
			for (int punkt = 0; punkt < punkte[komp].length; ++punkt)
				System.out.println("("+punkte[komp][punkt][X]+"|"+punkte[komp][punkt][Y]+")");
	}

	//Die Mainfunktion ruft die beiden obigen Funktionen auf, die nur aus Zwecken der bersichtlichkeit
	//ausgelagert wurden.
	public static void main (String args []){
		//Es wird geprft, ob ein Eingabeparameter dem Programm bergegeben wurde. In diesem Fall
		//wird das Programm im debug-Modus ausgefhrt, d.h. die Funktionen debugAusgabe() und
		//gebietAusgabe() werden aktiv.
		if (args.length != 0)
			debug = true;
		//Aufruf der der beiden oben beschriebenen Funktionen.
		constraintsDefinieren();
		loesungAusgeben();
	}
}
