%====================================================%
%  NUMERISCHER ONLINE-INTEGRATOR FUER GDGL           %
%  (C) 23.11.2005                                    %
%  Heinrich Mellmann                                 %
%====================================================%

function [ y, t ] = integrator(fkt_name, verfahren, x, t, h, varargin )

% integriert ein GDGL-System mit dem in 'verfahren'
% uebergebenem Verfahren;
%
% Parameter:
%   fkt_name - name der Funktion in der das GDGL abgelegt ist;
%   verfahren - Verfahren das zu Integration verwendet werden soll:
%      'expl-euler' - Explizites Euler-Verfahren;
%      'impl-euler' - Implizites Euler-Verfahren;
%      'imp-midpoint' - Implizite Mittelpunktregel;
%      'runge-kutta'  - Das Klassische Runge-Kutta Verfahren;
%   x - die Loesung der Gleichung im letzten Schritt;
%   t - alte Zeit;
%   h - Schrittweite fuer das Verfahren;
%   varargin - sonstige Variablen die fuer die Gleichung notwendig sind;
%
% Ausgabe:
%   y - die Approximation der Loesung der Gleichung zur Zeit t+h
%   t - die neue Zeit, d.h. t+h :)

switch  verfahren,
    case 'expl-euler',
        y = x + h * feval(fkt_name, t, x, varargin{:});
        
    case 'runge-kutta',
        %Koeffizienten berechnen
        k_1 = feval(fkt_name, t, x, varargin{:});
        k_2 = feval(fkt_name, t + h/2, x+h/2*k_1, varargin{:});
        k_3 = feval(fkt_name, t + h/2, x+h/2*k_2, varargin{:});
        k_4 = feval(fkt_name, t + h, x+h*k_3, varargin{:});

        F = (k_1+2*k_2+2*k_3+k_4)/6;

        % Runge-Kutta Schritt
        y = x + h * F;
    
    case 'impl-euler',
        % anzahl der Newton-Schritte
        k = 1;
        [n m] = size(x);
        y = x;
        for i = 1:k,
            JF = eye(n) - h*jacobi(fkt_name, y , t+h, h, varargin{:});
            F =  y-x-h*feval(fkt_name, t+h, y, varargin{:});
            Z = JF\F;
            y = y - Z;
        end
        
     case 'imp-midpoint',
        % anzahl der Newton-Schritte
        k = 1;
        [n m] = size(x);
        y = x;
        for i = 1:k,
            JF = eye(n)/h - jacobi(fkt_name, y/2+x/2 , t+h/2, h, varargin{:})/2;
            F =  (y-x)/h-feval(fkt_name, t+h/2, y/2+x/2, varargin{:});
            Z = JF\F;
            y = y - Z;
        end
        
     case 'impl-trapez',
        % anzahl der Newton-Schritte
        k = 1;
        [n m] = size(x);
        y = x;
        for i = 1:k,
            JF = eye(n)/h - jacobi(fkt_name, y/2+x/2 , t+h/2, h, varargin{:})/2;
            F =  (y-x)/h-0.5*(feval(fkt_name, t+h, y, varargin{:})+feval(fkt_name, t, x, varargin{:}));
            Z = JF\F;
            y = y - Z;
        end
end

% neue Zeit
t = t+h;



function [JF] = jacobi(fkt_name, zx, t, h, varargin)
% approximiere die jacobi-matrix an der stelle zx
epsilon = 1e-12;
[n m] = size(zx);

e = eye(n)*epsilon;
JF = zeros(n);

f0 = feval(fkt_name, t, zx, varargin{:});

for i = 1:n, 
    f1 = feval(fkt_name, t, zx + e(:,i) , varargin{:});
    JF(:,i) = ( f1 - f0 ) / epsilon;
end