//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**
 * @file Rate.h
 * @author Sascha Qualitz
 * @date created at 2009/10/26
 * @brief Declaration of odemx::base::Continuous::Rate
 * @sa Rate.cpp
 * @since 3.0
 */

#ifndef ODEMX_RATE_INCLUDED
#define ODEMX_RATE_INCLUDED


namespace odemx {
	namespace base {
		namespace continuous {

			//Forward declaration
			class Continuous;

			class ODEObject;

			class State;

			/** \class Derivates

				\ingroup base

				\author Sascha Qualitz

				\brief Object for handling an equation-element in the class ODEObject

				To describe ordinary differential equations the user have to to use this class. The user can use this class in different ways.
				With look and feel of a variable or with look and feel of an array. If used as a variable the user have to set the correct index, before using
				this object.

				\sa JacobiMatrix
				\sa DfDt
			 */
			class Rate {
			public:
				/**
					\brief Construction

					\note The variables continuous and index_ will be set to 0
				*/
				Rate();
				/**
					\brief Construction
					\param continuous
						pointer to the continuous object where the equation-element belongs to

					\note The variable index_ will be set to 0
				*/
				Rate(Continuous* continuous);

				/**
					\brief Construction
					\param continuous
						pointer to the continuous object where the equation-element belongs to
					\param index
						index of the variable to get/set (range 0..(Dimension-1))
				*/
				Rate(Continuous* continuous, unsigned index);

				/// destruction
				virtual ~Rate();

				/**
					\brief Overrides the assignment operator to store the value in the object of type Rate in the VariableContainer
					\param value
						The value that is set
				*/
				Rate& operator =(const State& value);

				/**
					\brief Overrides the assignment operator to store the value of type double in the VariableContainer
					\param value
						The value that is set
				*/
				Rate& operator =(const double value);

				/**
					\brief Overrides the index operator to set the internal variable index_
					\param index
						index of the variable to get/set (range 0..(Dimension-1))
				*/
				Rate& operator [](const unsigned i);

				/**
					\brief Set the value of type double
					\param index
						index of the variable to set (range 0..(Dimension-1))
					\param value
						The value that is set
				*/
				void setValue(unsigned index, double value);

				/**
					\brief Set the value of type State
					\param index
						index of the variable to set (range 0..(Dimension-1))
					\param value
						The value that is set
				*/
				void setValue(unsigned index, State& value);

				/**
					\brief Set the value of type State, at the index index_
					\param value
						The value that is set

					To set the value of the equation-element at the correct position you have to set index_ with the setIndex-Method, overwritten operator[]
					or the constructor.
				*/
				void setValue(State& value);

				/**
					\brief Set the value of type double, at the indexes _row and _column
					\param value
						The value that is set

					To set the value of the equation-element at the correct position you have to set index_ with the setIndex-Method, overwritten operator[]
					or the constructor.
				*/
				void setValue(double value);

				/**
					\brief Sets the variable index_
					\param index
						index of the variable to set/get (range 0..(Dimension-1))
				*/
				void setIndex(unsigned index);

				/**
					\brief Sets the variable continuous
					\param continuous
						pointer to the continuous object where equation-element belongs to
				*/
				void setContinuous(Continuous* continuous);

			private:
				/** pointer to the continuous object where equation-element belongs to
				*/
				Continuous* continuous;

				/** index of the variable to get/set (range 0..(Dimension-1))
				 */
				unsigned index_;
			};
		}
	}
}

#endif /* ODEMX_RATE_INCLUDED */
