//
// Pair.h
//
// $Id: //poco/Main/Foundation/include/Poco/Dynamic/Pair.h#9 $
//
// Library: Foundation
// Package: Dynamic
// Module:  Pair
//
// Definition of the Pair class.
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Foundation_Pair_INCLUDED
#define Foundation_Pair_INCLUDED


#include "Poco/Foundation.h"
#include "Poco/Dynamic/Var.h"
#include "Poco/Dynamic/VarHolder.h"
#include <utility>


namespace Poco {
namespace Dynamic {


template <typename K>
class Pair
	/// Pair allows to define a pair of values.
{
public:
	typedef typename std::pair<K, Var> Data;

	Pair(): _data()
		/// Creates an empty Pair
	{
	}

	Pair(const Pair& other): _data(other._data)
		/// Creates the Pair from another pair.
	{
	}

	Pair(const Data& val): _data(val)
		/// Creates the Pair from the given value.
	{
	}

	template <typename T>
	Pair(const std::pair<K, T>& val): _data(std::make_pair(val.first, Var(val.second)))
		/// Creates Pair form standard pair.
	{
	}

	template <typename T>
	Pair(const K& first, const T& second): _data(std::make_pair(first, Var(second)))
		/// Creates pair from two values.
	{
	}

	virtual ~Pair()
		/// Destroys the Pair.
	{
	}

	Pair& swap(Pair& other)
		/// Swaps the content of the two Pairs.
	{
		std::swap(_data, other._data);
		return *this;
	}

	Pair& operator = (const Pair& other)
		/// Copy constructs Pair from another pair.
	{
		Pair(other).swap(*this);
		return *this;
	}

	inline const K& first() const
		/// Returns the first member of the pair.
	{
		return _data.first;
	}

	inline const Var& second() const
		/// Returns the second member of the pair.
	{
		return _data.second;
	}

private:
	Data _data;
};


template <>
class VarHolderImpl<Pair<std::string> >: public VarHolder
{
public:
	VarHolderImpl(const Pair<std::string>& val): _val(val)
	{
	}

	~VarHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(Pair<std::string>);
	}

	void convert(Int8& val) const
	{
		throw BadCastException("Cannot cast Pair type to Int8");
	}

	void convert(Int16& val) const
	{
		throw BadCastException("Cannot cast Pair type to Int16");
	}
	
	void convert(Int32& val) const
	{
		throw BadCastException("Cannot cast Pair type to Int32");
	}

	void convert(Int64& val) const
	{
		throw BadCastException("Cannot cast Pair type to Int64");
	}

	void convert(UInt8& val) const
	{
		throw BadCastException("Cannot cast Pair type to UInt8");
	}

	void convert(UInt16& val) const
	{
		throw BadCastException("Cannot cast Pair type to UInt16");
	}
	
	void convert(UInt32& val) const
	{
		throw BadCastException("Cannot cast Pair type to UInt32");
	}

	void convert(UInt64& val) const
	{
		throw BadCastException("Cannot cast Pair type to UInt64");
	}

	void convert(bool& val) const
	{
		throw BadCastException("Cannot cast Pair type to bool");
	}

	void convert(float& val) const
	{
		throw BadCastException("Cannot cast Pair type to float");
	}

	void convert(double& val) const
	{
		throw BadCastException("Cannot cast Pair type to double");
	}

	void convert(char& val) const
	{
		throw BadCastException("Cannot cast Pair type to char");
	}

	void convert(std::string& val) const
	{
		// Serialize in JSON format: equals an object
		// JSON format definition: { string ':' value } string:value pair n-times, sep. by ','
		val.append("{ ");
		Var key(_val.first());
		appendJSONString(val, key);
		val.append(" : ");
		appendJSONString(val, _val.second());
		val.append(" }");	
	}

	void convert(Poco::DateTime&) const
	{
		throw BadCastException("Pair -> Poco::DateTime");
	}

	void convert(Poco::LocalDateTime&) const
	{
		throw BadCastException("Pair -> Poco::LocalDateTime");
	}

	void convert(Poco::Timestamp&) const
	{
		throw BadCastException("Pair -> Poco::Timestamp");
	}

	VarHolder* clone() const
	{
		return new VarHolderImpl(_val);
	}
	
	const Pair<std::string>& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isStruct() const
	{
		return false;
	}

	bool isInteger() const
	{
		return false;
	}

	bool isSigned() const
	{
		return false;
	}

	bool isNumeric() const
	{
		return false;
	}

	bool isString() const
	{
		return false;
	}

private:
	Pair<std::string> _val;
};


template <>
class VarHolderImpl<Pair<int> >: public VarHolder
{
public:
	VarHolderImpl(const Pair<int>& val): _val(val)
	{
	}

	~VarHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(Pair<int>);
	}

	void convert(Int8& val) const
	{
		throw BadCastException("Cannot cast Pair type to Int8");
	}

	void convert(Int16& val) const
	{
		throw BadCastException("Cannot cast Pair type to Int16");
	}
	
	void convert(Int32& val) const
	{
		throw BadCastException("Cannot cast Pair type to Int32");
	}

	void convert(Int64& val) const
	{
		throw BadCastException("Cannot cast Pair type to Int64");
	}

	void convert(UInt8& val) const
	{
		throw BadCastException("Cannot cast Pair type to UInt8");
	}

	void convert(UInt16& val) const
	{
		throw BadCastException("Cannot cast Pair type to UInt16");
	}
	
	void convert(UInt32& val) const
	{
		throw BadCastException("Cannot cast Pair type to UInt32");
	}

	void convert(UInt64& val) const
	{
		throw BadCastException("Cannot cast Pair type to UInt64");
	}

	void convert(bool& val) const
	{
		throw BadCastException("Cannot cast Pair type to bool");
	}

	void convert(float& val) const
	{
		throw BadCastException("Cannot cast Pair type to float");
	}

	void convert(double& val) const
	{
		throw BadCastException("Cannot cast Pair type to double");
	}

	void convert(char& val) const
	{
		throw BadCastException("Cannot cast Pair type to char");
	}

	void convert(std::string& val) const
	{
		// Serialize in JSON format: equals an object
		// JSON format definition: { string ':' value } string:value pair n-times, sep. by ','
		val.append("{ ");
		Var key(_val.first());
		appendJSONString(val, key);
		val.append(" : ");
		appendJSONString(val, _val.second());
		val.append(" }");	
	}

	void convert(Poco::DateTime&) const
	{
		throw BadCastException("Pair -> Poco::DateTime");
	}

	void convert(Poco::LocalDateTime&) const
	{
		throw BadCastException("Pair -> Poco::LocalDateTime");
	}

	void convert(Poco::Timestamp&) const
	{
		throw BadCastException("Pair -> Poco::Timestamp");
	}

	VarHolder* clone() const
	{
		return new VarHolderImpl(_val);
	}
	
	const Pair<int>& value() const
	{
		return _val;
	}

	bool isArray() const
	{
		return false;
	}

	bool isStruct() const
	{
		return false;
	}

	bool isInteger() const
	{
		return false;
	}

	bool isSigned() const
	{
		return false;
	}

	bool isNumeric() const
	{
		return false;
	}

	bool isString() const
	{
		return false;
	}

private:
	Pair<int> _val;
};


} // namespace Dynamic


} // namespace Poco


#endif // Foundation_Pair_INCLUDED
