//
// Nullable.h
//
// $Id: //poco/Main/Data/include/Poco/Data/Nullable.h#1 $
//
// Library: Data
// Package: DataCore
// Module:  Nullable
//
// Definition of the Nullable class.
//
// Copyright (c) 2008, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Data_Nullable_INCLUDED
#define Data_Nullable_INCLUDED

#include "Poco/Data/AbstractBinder.h"
#include <iostream>

namespace Poco {
namespace Data {
           
template <typename T>
class Nullable {
    /// Nullable class - template for field, that can be null
public:
    Nullable()
        : _value(), _isNull(true) 
        /// Creates the Nullable.
    {
    }

    Nullable(const T& value)
        : _value(value), _isNull(false) 
        /// Creates the Nullable from value
    {
    }

    Nullable(const NullData&)
        : _value(), _isNull(true) 
        /// Creates the Nullable from null
    {
    }

    Nullable& operator=(const T& value)
        /// Assigns new value to Nullable
    {
        _isNull = false;
        _value = value;
        return *this;
    }

    Nullable& operator=(const Nullable<T>& other)
        /// Assigns other Nullable to Nullable
    {
        _isNull = other._isNull;
        _value = other._value;
        return *this;
    }

    Nullable& operator=(const NullData&)
        /// Assigns null to Nullable
    {
        _isNull = true;
        return *this;
    }

    bool operator==(const Nullable<T>& other) const
        /// Compares two Nullable
    {
        return (_isNull && other._isNull) || (_isNull == other._isNull && _value == other._value);
    }

    bool operator==(const T& value) const
        /// Compares Nullable with value
    {
        return (!_isNull && _value == value);
    }

    bool operator==(const NullData&) const
        /// Compares Nullable with null
    {
        return _isNull;
    }

    bool operator!=(const NullData&) const
        /// Compares Nullable for non null
    {
        return !_isNull;
    }

    bool operator!=(const T& value) const
        /// Compares Nullable with value for non equal
    {
        return (_isNull || _value != value);
    }

    bool operator < (const Nullable<T>& other) const
        /// Compares two Nullable objects
    {
        if (_isNull < other._isNull)
            return true;
        return (_value < other._value);
    }

    operator T& ()
        /// Get reference to the value
    {
        return _value;
    }

    operator const T& () const
        /// Get const reference to the value
    {
        return _value;
    }

    bool isNull() const
        /// Test Nullable for null
    { 
        return _isNull; 
    }

    void setNull(bool isNull)
        /// Change Nullable "isNull" sign
    {
        _isNull = isNull; 
    }

    const T& getValue() const
        /// Get value
    { 
        return _value; 
    }

    void setValue(const T& value)
        /// Set value
    { 
        _isNull = false; _value = value; 
    }

private:

    T _value;
    bool _isNull;
};

//
// operators
//

template <typename T>
bool operator == (const T& value, const Nullable<T>& nValue) 
{
    return (!nValue.isNull() && value == nValue.getValue());
}

template <typename T>
bool operator != (const T& value, const Nullable<T>& nValue) 
{
    return (nValue.isNull() || value != nValue.getValue());
}

template <typename T>
bool operator == (const NullData&, const Nullable<T>& nValue) 
{
    return nValue.isNull();
}

template <typename T>
bool operator != (const NullData&, const Nullable<T>& nValue) 
{
    return !nValue.isNull();
}

template <typename T>
std::ostream& operator<<(std::ostream& out, const Nullable<T>& obj) 
{
    if (obj.isNull()) 
    {
        out << "NULL";
    }
    else 
    {
        out << obj.getValue();
    }
    return out;
}
  
} } // namespace Poco::Data

#endif // Data_Nullable_INCLUDED
