//
// Date.h
//
// $Id: //poco/Main/Data/include/Poco/Data/Date.h#7 $
//
// Library: Data
// Package: DataCore
// Module:  Date
//
// Definition of the Date class.
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Data_Date_INCLUDED
#define Data_Date_INCLUDED


#include "Poco/Data/Data.h"
#include "Poco/Dynamic/VarHolder.h"
#include "Poco/Exception.h"


namespace Poco {

class DateTime;

namespace Data {


class Time;


class Data_API Date
	/// Date class wraps a DateTime and exposes date related interface.
	/// The purpose of this class is binding/extraction support for date fields.
{
public:
	Date();
		/// Creates the Date

	Date(int year, int month, int day);
		/// Creates the Date

	Date(const DateTime& dt);
		/// Creates the Date from DateTime

	~Date();
		/// Destroys the Date.

	int year() const;
		/// Returns the year.

	int month() const;
		/// Returns the month.

	int day() const;
		/// Returns the day.

	void assign(int year, int month, int day);
		/// Assigns date.

	Date& operator = (const DateTime& dt);
		/// Assignment operator for DateTime.

	bool operator == (const Date& date);
		/// Equality operator.

	bool operator != (const Date& date);
		/// Inequality operator.

	bool operator < (const Date& date);
		/// Less then operator.

	bool operator > (const Date& date);
		/// Greater then operator.

private:
	int _year;
	int _month;
	int _day;
};


//
// inlines
//
inline int Date::year() const
{
	return _year;
}


inline int Date::month() const
{
	return _month;
}


inline int Date::day() const
{
	return _day;
}


inline Date& Date::operator = (const DateTime& dt)
{
	assign(dt.year(), dt.month(), dt.day());
	return *this;
}


inline bool Date::operator == (const Date& date)
{
	return _year == date.year() &&
		_month == date.month() &&
		_day == date.day();
}


inline bool Date::operator != (const Date& date)
{
	return !(*this == date);
}


inline bool Date::operator > (const Date& date)
{
	return !(*this == date) && !(*this < date);
}


} } // namespace Poco::Data


//
// VarHolderImpl<BLOB>
//


namespace Poco {
namespace Dynamic {


template <>
class VarHolderImpl<Poco::Data::Date>: public VarHolder
{
public:
	VarHolderImpl(const Poco::Data::Date& val): _val(val)
	{
	}

	~VarHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(Poco::Data::Date);
	}

	void convert(Poco::Timestamp& val) const
	{
		DateTime dt;
		dt.assign(_val.year(), _val.month(), _val.day());
		val = dt.timestamp();
	}

	void convert(Poco::DateTime& val) const
	{
		val.assign(_val.year(), _val.month(), _val.day());
	}

	void convert(Poco::LocalDateTime& val) const
	{
		val.assign(_val.year(), _val.month(), _val.day());
	}

	void convert(std::string& val) const
	{
		DateTime dt(_val.year(), _val.month(), _val.day());
		val = DateTimeFormatter::format(dt, "%Y/%m/%d");
	}

	VarHolder* clone() const
	{
		return new VarHolderImpl(_val);
	}
	
	const Poco::Data::Date& value() const
	{
		return _val;
	}

private:
	VarHolderImpl();
	Poco::Data::Date _val;
};


} } // namespace Poco::Dynamic


#endif // Data_Date_INCLUDED
