//
// LOBStream.h
//
// $Id: //poco/Main/Data/include/Poco/Data/LOBStream.h#3 $
//
// Library: Data
// Package: DataCore
// Module:  LOBStream
//
// Definition of the LOBStream class.
//
// Copyright (c) 2004-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Data_LOBStream_INCLUDED
#define Data_LOBStream_INCLUDED


#include "Poco/Foundation.h"
#include "Poco/UnbufferedStreamBuf.h"
#include "Poco/Data/LOB.h"
#include <istream>
#include <ostream>


namespace Poco {
namespace Data {


template <typename T>
class LOBStreamBuf: public BasicUnbufferedStreamBuf<T, std::char_traits<T> >
	/// This is the streambuf class used for reading from and writing to a LOB.
{
public:	
	LOBStreamBuf(LOB<T>& lob): _lob(lob), _it(_lob.begin())
		/// Creates LOBStreamBuf.
	{
	}


	~LOBStreamBuf()
		/// Destroys LOBStreamBuf.
	{
	}

protected:
	typedef std::char_traits<T> TraitsType;
	typedef BasicUnbufferedStreamBuf<T, TraitsType> BaseType;

	typename BaseType::int_type readFromDevice()
	{
		if (_it != _lob.end())
			return charToInt(*_it++);
		else
			return -1;
	}

	typename BaseType::int_type writeToDevice(T c)
	{
		_lob.appendRaw(&c, 1);
		return 1;
	}

private:
	LOB<T>& _lob;
	typename LOB<T>::Iterator _it;
};


template <typename T>
class LOBIOS: public virtual std::ios
	/// The base class for LOBInputStream and
	/// LOBOutputStream.
	///
	/// This class is needed to ensure the correct initialization
	/// order of the stream buffer and base classes.
{
public:
	LOBIOS(LOB<T>& lob, openmode mode): _buf(lob)
		/// Creates the LOBIOS with the given LOB.
	{
		poco_ios_init(&_buf);
	}

	~LOBIOS()
		/// Destroys the LOBIOS.
	{
	}

	LOBStreamBuf<T>* rdbuf()
		/// Returns a pointer to the internal LOBStreamBuf.
	{
		return &_buf;
	}

protected:
	LOBStreamBuf<T> _buf;
};


template <typename T>
class LOBOutputStream: public LOBIOS<T>, public std::basic_ostream<T, std::char_traits<T> >
	/// An output stream for writing to a LOB.
{
public:
	LOBOutputStream(LOB<T>& lob):
		LOBIOS<T>(lob, std::ios::out),
		std::ostream(LOBIOS<T>::rdbuf())
		/// Creates the LOBOutputStream with the given LOB.
	{
	}

	~LOBOutputStream()
		/// Destroys the LOBOutputStream.
	{
	}
};


template <typename T>
class LOBInputStream: public LOBIOS<T>, public std::basic_istream<T, std::char_traits<T> >
	/// An input stream for reading from a LOB.
{
public:
	LOBInputStream(LOB<T>& lob):
		LOBIOS<T>(lob, std::ios::in),
		std::istream(LOBIOS<T>::rdbuf())
		/// Creates the LOBInputStream with the given LOB.
	{
	}

	~LOBInputStream()
		/// Destroys the LOBInputStream.
	{
	}
};


typedef LOBOutputStream<unsigned char> BLOBOutputStream;
typedef LOBOutputStream<char> CLOBOutputStream;

typedef LOBInputStream<unsigned char> BLOBInputStream;
typedef LOBInputStream<char> CLOBInputStream;

} } // namespace Poco::Data


#endif // Data_LOBStream_INCLUDED
