//
// AbstractExtraction.h
//
// $Id: //poco/Main/Data/include/Poco/Data/AbstractExtraction.h#5 $
//
// Library: Data
// Package: DataCore
// Module:  AbstractExtraction
//
// Definition of the AbstractExtraction class.
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Data_AbstractExtraction_INCLUDED
#define Data_AbstractExtraction_INCLUDED


#include "Poco/Data/Data.h"
#include "Poco/Data/AbstractExtractor.h"
#include "Poco/Data/Limit.h"
#include "Poco/RefCountedObject.h"
#include "Poco/AutoPtr.h"
#include <vector>
#include <deque>
#include <list>
#include <cstddef>


namespace Poco {
namespace Data {


class AbstractPreparation;
class AbstractPreparator;


class Data_API AbstractExtraction: public Poco::RefCountedObject
	/// AbstractExtraction is the interface class that connects output positions to concrete values
	/// retrieved via an AbstractExtractor.
{
public:
	AbstractExtraction(Poco::UInt32 limit = Limit::LIMIT_UNLIMITED,
		Poco::UInt32 position = 0, bool bulk = false);
		/// Creates the AbstractExtraction. A limit value equal to EXTRACT_UNLIMITED (0xffffffffu) 
		/// means that we extract as much data as possible during one execute.
		/// Otherwise the limit value is used to partition data extracting to a limited amount of rows.

	virtual ~AbstractExtraction();
		/// Destroys the AbstractExtraction.

	void setExtractor(AbstractExtractor* pExtractor);
		/// Sets the class used for extracting the data. Does not take ownership of the pointer.

	AbstractExtractor* getExtractor() const;
		/// Retrieves the extractor object

	Poco::UInt32 position() const;
		/// Returns the extraction position.

	virtual std::size_t numOfColumnsHandled() const = 0;
		/// Returns the number of columns that the extraction handles.
		///
		/// The trivial case will be one single column but when
		/// complex types are used this value can be larger than one.

	virtual std::size_t numOfRowsHandled() const = 0;
		/// Returns the number of rows that the extraction handles.
		///
		/// The trivial case will be one single row but 
		/// for collection data types (ie vector) it can be larger.

	virtual std::size_t numOfRowsAllowed() const = 0;
		/// Returns the upper limit on number of rows that the extraction will handle.

	virtual std::size_t extract(std::size_t pos) = 0;
		/// Extracts a value from the param, starting at the given column position.
		/// Returns the number of rows extracted.

	virtual void reset();
		/// Resets the extractor so that it can be re-used.
		/// Does nothing in this implementation.
		/// Implementations should override it for different behavior.

	virtual bool canExtract() const;
		/// Returns true. Implementations should override it for different behavior.

	virtual AbstractPreparation* createPreparation(AbstractPreparator* pPrep, std::size_t pos) = 0;
		/// Creates a Preparation object for the extracting object

	void setLimit(Poco::UInt32 limit);
		/// Sets the limit.

	Poco::UInt32 getLimit() const;
		/// Gets the limit.

	virtual bool isNull(std::size_t row) const;
		/// In implementations, this function returns true if value at row is null, 
		/// false otherwise. 
		/// Normal behavior is to replace nulls with default values.
		/// However, extraction implementations may remember the underlying database
		/// null values and be able to later provide information about them.
		/// Here, this function throws NotImplementedException.

	bool isBulk() const;
		/// Returns true if this is bulk extraction.

	void setEmptyStringIsNull(bool emptyStringIsNull);
		/// Sets the empty string handling flag.

	bool getEmptyStringIsNull() const;
		/// Returns the empty string handling flag.

	void setForceEmptyString(bool forceEmptyString);
		/// Sets the force empty string flag.

	bool getForceEmptyString() const;
		/// Returns the force empty string flag.

	template <typename T>
	bool isValueNull(const T& str, bool deflt)
		/// Utility function to determine the nullness of the value.
		/// This generic version always returns default value
		/// (i.e. does nothing). The std::string overload does
		/// the actual work.
		///
	{
		return deflt;
	}

	bool isValueNull(const std::string& str, bool deflt);
		/// Overload for const reference to std::string.
		///
		/// Returns true when folowing conditions are met:
		///
		/// - string is empty 
		/// - getEmptyStringIsNull() returns true

private:
	AbstractExtractor* _pExtractor;
	Poco::UInt32       _limit;
	Poco::UInt32       _position;
	bool               _bulk;
	bool               _emptyStringIsNull;
	bool               _forceEmptyString;
};


typedef Poco::AutoPtr<AbstractExtraction> AbstractExtractionPtr;
typedef std::vector<AbstractExtractionPtr> AbstractExtractionVec;
typedef std::vector<AbstractExtractionVec> AbstractExtractionVecVec;
typedef std::deque<AbstractExtractionPtr> AbstractExtractionDeq;
typedef std::vector<AbstractExtractionDeq> AbstractExtractionDeqVec;
typedef std::list<AbstractExtractionPtr> AbstractExtractionLst;
typedef std::vector<AbstractExtractionLst> AbstractExtractionLstVec;


//
// inlines
//
inline void AbstractExtraction::setExtractor(AbstractExtractor* pExtractor)
{
	_pExtractor = pExtractor;
}


inline AbstractExtractor* AbstractExtraction::getExtractor() const
{
	return _pExtractor;
}


inline void AbstractExtraction::setLimit(Poco::UInt32 limit)
{
	_limit = limit;
}


inline Poco::UInt32 AbstractExtraction::getLimit() const
{
	return _limit;
}


inline bool AbstractExtraction::isNull(std::size_t row) const
{
	throw NotImplementedException("Check for null values not implemented.");
}


inline Poco::UInt32 AbstractExtraction::position() const
{
	return _position;
}


inline bool AbstractExtraction::isBulk() const
{
	return _bulk;
}


inline void AbstractExtraction::reset()
{
}


inline bool AbstractExtraction::canExtract() const
{
	return true;
}


inline void AbstractExtraction::setEmptyStringIsNull(bool emptyStringIsNull)
{
	_emptyStringIsNull = emptyStringIsNull;
}


inline bool AbstractExtraction::getEmptyStringIsNull() const
{
	return _emptyStringIsNull;
}


inline void AbstractExtraction::setForceEmptyString(bool forceEmptyString)
{
	_forceEmptyString = forceEmptyString;
}


inline bool AbstractExtraction::getForceEmptyString() const
{
	return _forceEmptyString;
}


} } // namespace Poco::Data


#endif // Data_AbstractExtraction_INCLUDED
