package betriebpraktikanten;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.vecmath.Point2d;
import javax.vecmath.Point2i;

import bsh.EvalError;
import bsh.Interpreter;

/**
 * Reprsentiert ein Menge von Posen ([x,y],theta), welche als Liste abgelegt wird. Der Richtungswinkel
 * wird dabei unabhngig von einem konkreten [x,y] gespeichert. Es handelt sich also nicht um eine
 * lupenreine Menge von Posen, vielmehr um eine Menge von Paaren (x,y) und Winkeln (theta).
 * Da zur Bestimmung der mglichen Blickwinkel von einem konkreten Paar (x,y) ausgegangen wird, ist
 * diese Einschrnkung zweckmig.
 * 
 * @author lars
 *
 */
public class PoseListe {

	/** Alias der x Koordinate der Pose in den Constraints */
	public final static String POSE_X = "posX";
	/** Alias der y Koordinate der Pose in den Constraints */
	public final static String POSE_Y = "posY";
	/** Alias des Richtungswinkels der Pose in den Constraints */
	public final static String POSE_THETA = "theta";

	/** positionen (x,y) */
	private List<Point2i> vectorSpace = null;
	/** blickwinkel (theta) */
	private List<Float>	angleSpace = null;
	
	/**
	 * Liefert die Anzahl der aktuell noch mglichen Positionen (x,y)
	 * @return Anzahl der aktuell noch mglichen Positionen
	 */
	public int vectorSize() {
		return vectorSpace.size();
	}
	
	/**
	 * Liefert die Menge der aktuell noch mglichen Positionen (x,y)
	 * @return Menge der aktuell noch mglichen Positionen (x,y)
	 */
	public Iterator<Point2i> vectorIterator() {
		return vectorSpace.iterator();
	}

	/**
	 * Liefert die Anzahl der aktuell noch mglichen Blickwinkel (theta)
	 * @return Anzahl der aktuell noch mglichen Blickwinkel (theta)
	 */
	public int angleSize() {
		return angleSpace.size();
	}
	
	/**
	 * Liefert die Menge der aktuell noch mglichen Blickwinkel (theta)
	 * @return Menge der aktuell noch mglichen Blickwinkel (theta)
	 */
	public Iterator<Float> angleIterator() {
		return angleSpace.iterator();
	}

	/**
	 * Erstellte eine Ausdruck D0 mit eine Menge von Posen entsprechend den initialen Variablendomains
	 * @param bsh Interpreter (Umgebung)
	 */
	public PoseListe (Interpreter bsh) {
		
		// winkel initialisieren
		double thetaInterval = 0.02;
		try {
			thetaInterval = (Double) bsh.get("thetaRaster");
		} catch (EvalError e1) {
			System.err.println("thetaRaster nicht definiert --> nutze 0.01");
		}
		angleSpace = new LinkedList<Float>();
		for (float theta=Spielfeld.thetaMinValue; theta<=Spielfeld.thetaMaxValue; theta+=thetaInterval ) {
			angleSpace.add(new Float(theta));
		}
		// x,y initialisieren
		double xInterval = 10.0;
		try {
			xInterval = (Double) bsh.get("spielfeldRasterX");
		} catch (Exception e) {
			System.err.println("spielfeldRasterX nicht definiert --> nutze 10.0");
		}
		double yInterval = 10.0;
		try {
			yInterval = (Double) bsh.get("spielfeldRasterY");
		} catch (Exception e) {
			System.err.println("spielfeldRasterY nicht definiert --> nutze 10.0");
		}
		vectorSpace = new LinkedList<Point2i>();
		for (int x=Spielfeld.xMinValue; x<=Spielfeld.xMaxValue; x+=xInterval ) {
			for (int y=Spielfeld.yMinValue; y<=Spielfeld.yMaxValue; y+=yInterval ) {
				vectorSpace.add(new Point2i(x,y));
			}
			if ( ConstraintPropagiser.isDebug()&& (x % 1000 == 0) && (x > 0) ) System.out.print(x+" ");
		}
		if ( ConstraintPropagiser.isDebug() ) System.out.println();
	}
	
	/**
	 * Schrnkt die mglichen Positionen (x,y) des Roboters durch die Anwendung
	 * des Contraints c ein
	 * @param bsh Interpreter (Umgebung)
	 * @param c Contraint
	 * @throws EvalError 
	 */
	public void runVectorConstraint(Interpreter bsh, String c) {
		if (ConstraintPropagiser.isDebug()) System.out.println(c);;
		Iterator<Point2i> iter = this.vectorIterator();
		// constaints anwenden
		int i = 0;
		try {
			while (iter.hasNext()) {
				Point2i p = iter.next();
					// x,y laden
					bsh.set(POSE_X,p.x);
					bsh.set(POSE_Y,p.y);
					// c evaluieren
					if (!(Boolean)bsh.eval(c)) {
						// p entfernen
						iter.remove();
					}
					else {
						// p bleibt
					}
				if ( ConstraintPropagiser.isDebug() && (i % 5000 == 0) && (i > 0) ) System.out.print(i+" ");
				i++;
			}
		}
		catch (EvalError e) {
			System.err.println(c+"\nkonnte nicht evaluiert werden");
			e.printStackTrace();
		}
		if ( ConstraintPropagiser.isDebug() ) System.out.println();
	}
	
	/**
	 * Schrnkt den mglichen Richtungswinkel (theta) des Roboters durch die Anwendung
	 * des Contraints c ein, wobei (x,y) fest/gegeben ist
	 * @param bsh Interpreter (Umgebung)
	 * @param c Contraint
	 * @param p Position (x,y)
	 * @throws EvalError 
	 */
	public void runAngleConstraint(Interpreter bsh, String c, Point2d p) {
		if (ConstraintPropagiser.isDebug()) System.out.println(c);;
		Iterator<Float> iter = this.angleIterator();
		try {
			// x,y laden
			bsh.set(POSE_X,p.x);
			bsh.set(POSE_Y,p.y);
			// constaints anwenden
			int i = 0;
			while (iter.hasNext()) {
				Float theta = iter.next();
				// theta laden
				bsh.set(POSE_THETA,theta);
				// c evaluieren
				if (!(Boolean)bsh.eval(c)) {
					// angle entfernen
					iter.remove();
				}
				else {
					// angle bleibt
				}
				if ( ConstraintPropagiser.isDebug() && (i % 5000 == 0) && (i > 0) ) System.out.print(i+" ");
				i++;
			}
		}
		catch (EvalError e) {
			System.err.println(c+"\nkonnte nicht evaluiert werden");
			e.printStackTrace();
		}
		if ( ConstraintPropagiser.isDebug() ) System.out.println();
	}
	
	/**
	 * Liefert den Zentroiden aller aktuellen Postition (x,y)
	 * 
	 * @return Zentroid aller aktuellen Postition (x,y)
	 */
	public Point2d getCentroid () {
		Point2d centroid = new Point2d(0.0,0.0);
		// x,y arithmetrisch mitteln
		Iterator<Point2i> vectorIter = this.vectorIterator();
		while (vectorIter.hasNext()) {
			Point2i p = vectorIter.next();
			centroid.x += p.x/(double)this.vectorSize();
			centroid.y += p.y/(double)this.vectorSize();
		}
		return centroid;
	}
	
	/**
	 * Liefert den Mittelwert aller aktuellen Blickwinkel (theta)
	 * 
	 * @return Mittelwert aller aktuellen Blickwinkel (theta)
	 */
	public double getAvgAngle () {
		double avg = 0.0;
		// theta arithmetrisch mitteln
		Iterator<Float> angleIter = this.angleIterator();
		while (angleIter.hasNext()) {
			Float theta = angleIter.next();
			avg += theta/this.angleSize();
		}
		return avg;
	}
}
