package betriebpraktikanten;
import java.awt.Color;
import java.awt.Point;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.vecmath.Point2d;
import javax.vecmath.Point2i;
import javax.vecmath.Point3d;

import bsh.EvalError;
import bsh.Interpreter;

/**
 * Startklasse, hier werden alle notwendigen Objekte eingebunden.
 * @author lars
 *
 */
public class ConstraintPropagiser {
	/** startzeit */
	private static long time = System.currentTimeMillis();
	
	/** globales Debug-Flag */
	private static boolean debug;
	
	/** Strichlnge fr den Peilingswinkel */
	private static int drawLength;
	
	/** globales Debug-Flag */
	public static boolean isDebug () {
		return debug;
	}

	/**
	 * Zeichnet das Spielfeld mit den aktuell noch mglichen Roboterpositionen
	 * und Richtungswinkeln.
	 * @param vectorIter Zu zeichnende Positionen
	 * @param angleIter Zu zeichnende Winkel
	 * @param name Name des Fensters
	 * @param position Position des Fensters (oben links entspricht 0;0)
	 * @throws IOException
	 */
	public static void paintField(Iterator<Point2i> vectorIter, Iterator<Float> angleIter, String name, Point position) throws IOException {
		Spielfeld picture = new Spielfeld(name,position);
		// (x,y)
		picture.setDrawColor(new Color(28,134,238));
		int pointCount = 0;
		Point2i pXY = new Point2i();
		while (vectorIter.hasNext()) {
			pXY = vectorIter.next();
			picture.drawPoint(pXY);
			pointCount++;
		}
		
		// (theta)
		int laenge = drawLength;
		Point2i center = new Point2i(0,0);
		// wenn nur ein punkt gezeichent wurde (final), dann zeiche winkel ab dort
		if (pointCount == 1) {
			// center verschieben
			center = pXY;
			// punkt nochmal breiter malen
			picture.drawPoint(center,5);
		}
		while (angleIter.hasNext()) {
			Float theta = angleIter.next();
			int x = (int) (laenge * Math.cos(theta));
			int y = (int) (laenge * Math.sin(theta));
			picture.setDrawColor(new Color(255,134,0));
			picture.drawLine(center, new Point2i(x+center.x,y+center.y));
		}
		picture.setVisible(true);
	}
	
	/**
	 * Gibt ein paar statistische Daten (Speicherverbrauch, verstrichene Zeit, Anzahl der Elemente) ber <i>pl</i> aus.
	 * @param pl eine PoseListe
	 */
	private static void printStats(PoseListe pl) {
		System.out.println("  memory: "+((Runtime.getRuntime().totalMemory()-Runtime.getRuntime().freeMemory()))/1024.0/1024+"mb");
		System.out.println("  time  : "+(System.currentTimeMillis() - time)/1000.0+"s");
		System.out.println("  vSize : "+pl.vectorSize());
		System.out.println("  aSize : "+pl.angleSize());
	}
	
	/**
	 * Liest die Constraints aus der Datei <i>filename</i> ein und gibt sie als Liste zurck.
	 * @param filename Constraint-Datei
	 * @return Liste von Constraints
	 * @throws IOException
	 */
	private static List<Constraint> getConstraints(String filename) throws IOException {
		BufferedReader bufferedReader = new BufferedReader(new FileReader(filename));
		List<Constraint> constraints = new ArrayList<Constraint>();
		String zeile = null;
		while ((zeile = bufferedReader.readLine()) != null) {
			zeile = zeile.trim();
			if ( (zeile.length() > 0) && !zeile.startsWith("#") ) {
				constraints.add(new Constraint(zeile));
			}
		}
		bufferedReader.close();
		return constraints;
	}
	
	/**
	 * Liest die Domains/Variablen aus der Datei <i>filename</i> ein und gibt sie als Liste zurck.
	 * @param filename Ausdrcke-Datei
	 * @return Liste von Ausdrcken
	 * @throws IOException
	 */
	private static List<Ausdruck> getExpressions(String filename) throws IOException {
		BufferedReader bufferedReader = new BufferedReader(new FileReader(filename));
		List<Ausdruck> domains = new ArrayList<Ausdruck>();
		String zeile = null;
		while ((zeile = bufferedReader.readLine()) != null) {
			zeile = zeile.trim();
			if ( (zeile.length() > 0) && !zeile.startsWith("#") ) {
				domains.add(new Ausdruck(zeile));
			}
		}
		bufferedReader.close();
		return domains;
	}
	
	/**
	 * Initialisiert den Java-Interpreter, welche die Ausdrcke/Constraints auswertet, und
	 * einigen globalen Variablen
	 * @param expressions Liste von Ausdrcken
	 * @return
	 */
	private static Interpreter initInterpreter(List<Ausdruck> expressions) {
		Interpreter bsh = new Interpreter();
		for (int v=0; v<expressions.size(); v++) {
			try {
				bsh.eval(expressions.get(v).getExpression());
			} catch (Exception e) {
				System.err.println("Ungltiger Ausdruck (wird ignoriert): "+expressions.get(v));
				e.printStackTrace();
			}
		}
		
		// debug + drawLength
		try {
			debug = (Boolean) bsh.get("debug");
		} catch (EvalError e) {
			debug = false;
		}
		try {
			drawLength = (Integer) bsh.get("strichLaenge");
		} catch (EvalError e) {
			drawLength = 300;
		}
		if (ConstraintPropagiser.debug) {
			try {
				System.out.println("NW");
				System.out.println("bildVornFlaggeMitteHorizontal="+bsh.get("bildVornFlaggeMitteHorizontal"));
				System.out.println("bildVornFlaggeMitteVertikal="+bsh.get("bildVornFlaggeMitteVertikal"));
				System.out.println("bildVornAbstandHorizontal="+bsh.get("bildVornAbstandHorizontal"));
				System.out.println("bildVornAbstandVertikal="+bsh.get("bildVornAbstandVertikal"));
				System.out.println("bildVornGroessenverhaeltnisHorizontal="+bsh.get("bildVornGroessenverhaeltnisHorizontal"));
				System.out.println("bildVornGroessenverhaeltnisVertikal="+bsh.get("bildVornGroessenverhaeltnisVertikal"));
				System.out.println("bildVornFlaggeWinkelHorizontal="+bsh.get("bildVornFlaggeWinkelHorizontal"));
				System.out.println("bildVornFlaggeWinkelVertikal="+bsh.get("bildVornFlaggeWinkelVertikal"));
				System.out.println("bildVornFlaggeAbstandHorizontal="+bsh.get("bildVornFlaggeAbstandHorizontal"));
				System.out.println("bildVornFlaggeAbstandVertikal="+bsh.get("bildVornFlaggeAbstandVertikal"));
				System.out.println("abstandFlaggeNw="+bsh.get("abstandFlaggeNw"));
				System.out.println("SW");
				System.out.println("bildLinksFlaggeMitteHorizontal="+bsh.get("bildLinksFlaggeMitteHorizontal"));
				System.out.println("bildLinksFlaggeMitteVertikal="+bsh.get("bildLinksFlaggeMitteVertikal"));
				System.out.println("bildLinksAbstandHorizontal="+bsh.get("bildLinksAbstandHorizontal"));
				System.out.println("bildLinksAbstandVertikal="+bsh.get("bildLinksAbstandVertikal"));
				System.out.println("bildLinksGroessenverhaeltnisHorizontal="+bsh.get("bildLinksGroessenverhaeltnisHorizontal"));
				System.out.println("bildLinksGroessenverhaeltnisVertikal="+bsh.get("bildLinksGroessenverhaeltnisVertikal"));
				System.out.println("bildLinksFlaggeWinkelHorizontal="+bsh.get("bildLinksFlaggeWinkelHorizontal"));
				System.out.println("bildLinksFlaggeWinkelVertikal="+bsh.get("bildLinksFlaggeWinkelVertikal"));
				System.out.println("bildLinksFlaggeAbstandHorizontal="+bsh.get("bildLinksFlaggeAbstandHorizontal"));
				System.out.println("bildLinksFlaggeAbstandVertikal="+bsh.get("bildLinksFlaggeAbstandVertikal"));
				System.out.println("abstandFlaggeSw="+bsh.get("abstandFlaggeSw"));
				System.out.println("NO");
				System.out.println("bildRechtsFlaggeMitteHorizontal="+bsh.get("bildRechtsFlaggeMitteHorizontal"));
				System.out.println("bildRechtsFlaggeMitteVertikal="+bsh.get("bildRechtsFlaggeMitteVertikal"));
				System.out.println("bildRechtsAbstandHorizontal="+bsh.get("bildRechtsAbstandHorizontal"));
				System.out.println("bildRechtsAbstandVertikal="+bsh.get("bildRechtsAbstandVertikal"));
				System.out.println("bildRechtsGroessenverhaeltnisHorizontal="+bsh.get("bildRechtsGroessenverhaeltnisHorizontal"));
				System.out.println("bildRechtsGroessenverhaeltnisVertikal="+bsh.get("bildRechtsGroessenverhaeltnisVertikal"));
				System.out.println("bildRechtsFlaggeWinkelHorizontal="+bsh.get("bildRechtsFlaggeWinkelHorizontal"));
				System.out.println("bildRechtsFlaggeWinkelVertikal="+bsh.get("bildRechtsFlaggeWinkelVertikal"));
				System.out.println("bildRechtsFlaggeAbstandHorizontal="+bsh.get("bildRechtsFlaggeAbstandHorizontal"));
				System.out.println("bildRechtsFlaggeAbstandVertikal="+bsh.get("bildRechtsFlaggeAbstandVertikal"));
				System.out.println("abstandFlaggeNo="+bsh.get("abstandFlaggeNo"));
				System.out.println("flaggeWichtungHorizontal="+bsh.get("flaggeWichtungHorizontal"));
//				System.out.println("alpha="+bsh.get("alpha"));
			} catch (EvalError e) {
				e.printStackTrace();
			}
		}
		return bsh;
	}
	
	/**
	 * Gibt die Nichtlsbarkeit bekannt und beendet das Programm
	 *
	 */
	static void noSolution() {
		System.out.println("Keine global konsistente Belegung vorhanden!");
	}

	/**
	 * Hier geht's los!
	 * <ol>
	 * <li>Ausdrcke + Constrains einlesen</li>
	 * <li>Ausdrcke auswerten und fr das sptere Benutzen bekannt machen</li>
	 * <li>C0 bestimmen</li>
	 * <li>Cn bestimmen --> (x,y) wird eingeschrnkt, (theta)-Contraints werden zwischengespeichert</li>
	 * <li>Zentroid bestimmen --> festes (x,y), um (theta) einzuschrnken</li>
	 * <li>Cn bestimmen(2) --> (theta) wird eingeschrnkt (mit festem (x,y))</li>
	 * <li>Den Durchschnitt aller (theta) bestimmen</li>
	 * <li>Finale Pose ausgeben</li>
	 * </ol>
	 * @param args Erwartet den Namen der Ausdrcke (args[0]) und Constraint-Datei (args[1]).
	 * Falls nicht angegeben wird ein Default-Wert benutzt (src/config/expressions.config + src/config/constraints.config)
	 * @throws IOException 
	 */
	public static void main(String[] args) throws IOException {

		// ausdrcke + constrains einlesen
		String expressionsConfig	= "src/config/expressions.config";
		String constraintsConfig	= "src/config/constraints.config";
		if (args.length >= 2) {
			expressionsConfig = args[0]; 
			constraintsConfig = args[1]; 
		}
		// alle constraints
		List<Constraint> constraints		= getConstraints(constraintsConfig);
		// alle ausdrcke
		List<Ausdruck> expressions			= getExpressions(expressionsConfig);
		// theta-constraints (zwischenspeicher)
		List<Constraint> angleConstraints	= new LinkedList<Constraint>();

		// ausdrcke bekannt machen
		Interpreter bsh = initInterpreter(expressions);
		
		// C0
		Point pos = new Point(0,0);
		String C0 = "D0 = Dom(x) x Dom(y) x Dom(theta)";
		PoseListe allPoses = new PoseListe(bsh);
		paintField(allPoses.vectorIterator(),allPoses.angleIterator(),C0,pos);
		pos.x += 40;
		pos.y += 20;
		if (debug) printStats(allPoses);
		
		// Cn --> (x,y) wird eingeschrnkt, (theta)-contraints werden zwischengespeichert
		int cCount = 0;
		for (int i=0; i<constraints.size(); i++) {
			Constraint c = constraints.get(i);
			// nur x,y im constraint --> evaluieren
			if (c.isVectorConstraint() && !c.isAngleConstraint()) {
				allPoses.runVectorConstraint(bsh,c.getConstraint());
				paintField(allPoses.vectorIterator(),allPoses.angleIterator(),"D"+(cCount+1)+" = D"+(cCount)+" - C("+c.getOriginal()+")",pos);
				pos.x += 40;
				pos.y += 20;
				cCount++;
			}
			// theta im constraint --> zwischenspeichern
			else if (c.isAngleConstraint()) {
				angleConstraints.add(c);
			}
			else {
				System.err.println("Achtung! Dieses Constraint enthlt weder "+PoseListe.POSE_X+", "+PoseListe.POSE_Y+" noch "+PoseListe.POSE_THETA+"!\n"+c);
			}
			if (debug) printStats(allPoses);
			if (allPoses.vectorSize() == 0) {
				noSolution();
				return;
			}
		}
		
		// Zentroid bestimmen --> festes (x,y), um (theta) einzuschrnken
		Point2d centroid = allPoses.getCentroid();
		
		// Cn --> (theta) wird eingeschrnkt (mit festem (x,y))
		for (int i=0; i<angleConstraints.size(); i++) {
			Constraint c = angleConstraints.get(i);
			allPoses.runAngleConstraint(bsh,c.getConstraint(),centroid);
			paintField(allPoses.vectorIterator(),allPoses.angleIterator(),"D"+(cCount+1)+" = D"+(cCount)+" - C("+c.getOriginal()+")",pos);
			pos.x += 40;
			pos.y += 20;
			cCount++;
			if (debug) printStats(allPoses);
			if (allPoses.angleSize() == 0) {
				noSolution();
				return;
			}
		}

		// durchschnittswinkel bestimmen
		double avgTheta = allPoses.getAvgAngle();
		
		// finale pose ausgeben
		Point3d pose = new Point3d(centroid.x, centroid.y, avgTheta);		
		System.out.println("Pose --> "+pose);
		List<Point2i> poseVector = new LinkedList<Point2i>();
		poseVector.add(new Point2i((int)pose.x,(int)pose.y));
		List<Float> angleVector = new LinkedList<Float>();
		angleVector.add(new Float(pose.z));
		paintField(poseVector.iterator(),angleVector.iterator(),"Finale Pose = "+pose,pos);
	}

}
