class FlagConstraint implements RobotPositionConstants {
	//Flagge zum Flaggen-Constraint
	Flag flag;
	
	//Minimal mgliche Entfernung zum Roboter (in mm)
	protected double dmin;
	
	//Maximal mgliche Entfernung zum Roboter (in mm)
	protected double dmax;
	
	FlagConstraint(Flag flag, int error) {
		this.flag = flag;
		setMinAndMaxDistance(error);
	}
	
	/**
	 * Berechnet die minimale und maximale Distanz zur Flagge, abhngig vom Messfehler
	 * @param error Betrag des Messfehlers
	 */
	public void setMinAndMaxDistance(int error) {
		this.dmin = getDistance(flag.getBh(), flag.getH(), error);
		this.dmax = getDistance(flag.getBh(), flag.getH(), -error);
		System.out.println(flag.getId()+" min/max : " + dmin + " / " + dmax);
	}
	
	/**
	 * Berechnet Distanz der Flagge in Abhngigkeit von der Hhe des Bildes,
	 * der Pixelhhe der Flagge auf dem Bild, dem Messfehler, des vertikalen
	 * Kameraffnungswinkels und der realen Flaggenhhe.
	 * @param bh Bildhhe in Pixel
	 * @param h Pixelhhe der Flagge
	 * @param error Messfehler (positiv oder negativ)
	 * @return Abstand der Flagge vom Brennpunkt der Kamera in mm
	 */
	protected double getDistance(int bh, int h, int error) {
		return FLAG_HEIGHT*(bh/(2*Math.tan(Math.toRadians(Y_VIEWING_ANGLE/2.0))))/(h+error);
	}
	
	//prft, ob angegebene Koordinaten den Constraint erfllen
	public boolean validate(int x, int y) {
		int deltaX = flag.getX()-x;
		int deltaY = flag.getY()-y; 
		double dist = Math.sqrt(deltaX*deltaX+deltaY*deltaY);
		
		return dmin <= dist && dist <= dmax;
	}
	
	/**
	 * Prft, ob es zu einer gegebenen X-Koordinate eine Y-Koordinate im
	 * Wertebereich von yc gibt, die den Flaggen-Constraint erfllt
	 * @param x X-Koordinate
	 * @param yc Wertebereich fr y
	 * @return true, falls y aus yc existiert, so dass Flaggen-Constraint erfllt,
	 * ansonsten false
	 */
	public boolean validateX(int x, IntervalConstraint yc) {
		/**
		 * Es gibt drei Flle zu unterscheiden :
		 * 1. Die senkrechte Gerade durch X schneidet den Constraint-Ring nicht
		 * 2. Die senkrechte Gerade durch X schneidet Aussen-und Innenring
		 * 3. Die senkrechte Gerade durch X schneidet nur den Ausssenring
		 *    
		 */
		
		//1. Fall : Die senkrechte Grade durch X schneidet den Constraint-Ring nicht
		if (x<flag.getX()-dmax || x>flag.getX()+dmax)
			return false;
		
		int deltaX = flag.getX()-x;
		
		//unterer Halbkreis
		int ymin1 = (int)Math.round(flag.getY()-Math.sqrt(dmax*dmax-deltaX*deltaX));
		int ymax1 = 0;
		if (x>flag.getX()-dmin)
			ymax1 = (int)Math.round(flag.getY()-Math.sqrt(dmin*dmin-deltaX*deltaX));
		
		//oberer Halbkreis
		int ymax2 = (int)Math.round(flag.getY()+Math.sqrt(dmax*dmax-deltaX*deltaX));
		int ymin2 = 0;
		if (x<flag.getX()+dmin)
			ymin2 = (int)Math.round(flag.getY()+Math.sqrt(dmin*dmin-deltaX*deltaX));
		
		//2.Fall : Die senkrechte Gerade durch X schneidet Aussen-und Innenring
		if (x>flag.getX()-dmin && x<flag.getX()+dmin) {
			IntervalConstraint yc2 = new IntervalConstraint(ymin1, ymax1);
			IntervalConstraint yc3 = new IntervalConstraint(ymin2, ymax2);
		
			return (yc.intersection(yc2)!=null || yc.intersection(yc3)!=null);
		}	
		
		//3.Fall : Die senkrechte Gerade durch X schneidet nur den Aussenring
		IntervalConstraint yc2 = new IntervalConstraint(ymin1, ymax2);
		return (yc.intersection(yc2)!=null);
	}
	
	/**
	 * Prft, ob es zu einer gegebenen Y-Koordinate eine X-Koordinate im
	 * Wertebereich von xc gibt, die den Flaggen-Constraint erfllt
	 * @param y Y-Koordinate
	 * @param xc Wertebereich fr x
	 * @return true, falls x aus xc existiert, so dass Flaggen-Constraint erfllt,
	 * ansonsten false
	 */
	public boolean validateY(int y, IntervalConstraint xc) {
		/**
		 * Es gibt drei Flle zu unterscheiden :
		 * 1. Die waagerechte Gerade durch Y schneidet den Constraint-Ring nicht
		 * 2. Die waagerechte Gerade durch Y schneidet Aussen-und Innenring
		 * 3. Die waagerechte Gerade durch Y schneidet nur den Ausssenring
		 *    
		 */
		
		//1. Fall : Die senkrechte Grade durch Y schneidet den Constraint-Ring nicht
		if (y<flag.getY()-dmax || y>flag.getY()+dmax)
			return false;
		
		int deltaY = flag.getY()-y;
		
		//linker Halbkreis
		int xmin1 = (int)Math.round(flag.getX()-Math.sqrt(dmax*dmax-deltaY*deltaY));
		int xmax1 = 0;
		if (y>flag.getY()-dmin)
			xmax1 = (int)Math.round(flag.getX()-Math.sqrt(dmin*dmin-deltaY*deltaY));
		
		//rechter Halbkreis
		int xmax2 = (int)Math.round(flag.getX()+Math.sqrt(dmax*dmax-deltaY*deltaY));
		int xmin2 = 0;
		if (y<flag.getY()+dmin)
			xmin2 = (int)Math.round(flag.getX()+Math.sqrt(dmin*dmin-deltaY*deltaY));
		
		//2.Fall : Die waagerechte Gerade durch Y schneidet Aussen-und Innenring
		if (y>flag.getY()-dmin && y<flag.getY()+dmin) {
			IntervalConstraint xc2 = new IntervalConstraint(xmin1, xmax1);
			IntervalConstraint xc3 = new IntervalConstraint(xmin2, xmax2);
		
			return (xc.intersection(xc2)!=null || xc.intersection(xc3)!=null);
		}	
		
		//3.Fall : Die waagerechte Gerade durch Y schneidet nur den Aussenring
		IntervalConstraint xc2 = new IntervalConstraint(xmin1, xmax2);
		return (xc.intersection(xc2)!=null);
	}
	
	public Bounds getOuterBounds() {
		int x1 = (int)Math.round(flag.getX()-dmax);
		int x2 = (int)Math.round(flag.getX()+dmax);
		int y1 = (int)Math.round(flag.getY()-dmax);
		int y2 = (int)Math.round(flag.getY()+dmax);
		return new Bounds(x1, y1, x2, y2);
	}
	
	/**
	 * Ermittelt den maximal lokal konsistenten Bereich fr den Flaggenconstraint
	 * in einem eingeschrnkten Bereich
	 * @param b Eingeschrnkter Bereich
	 * @return maximal lokal konsistenter Bereich oder null
	 */
	public Bounds getMaxConsistentBounds(Bounds b) {
		/** not implemented */
		return null;
	}
	
	public Flag getFlag() {
		return flag;
	}

	public void setFlag(Flag flag) {
		this.flag = flag;
	}
	
	public double getDmax() {
		return dmax;
	}

	public void setDmax(double dmax) {
		this.dmax = dmax;
	}

	public double getDmin() {
		return dmin;
	}

	public void setDmin(double dmin) {
		this.dmin = dmin;
	}
}