package a2;

import java.util.BitSet;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * Eine Menge von Positionen (da Set<Position> zu langsam ist und zuviel Speicher braucht).
 */
public class PositionSet implements Iterable<Position> {
	
	
	/* support fr:
	 *	for(Position p : positionSet) {
	 *		...
	 *	}
	 */
	private class PositionIterator implements Iterator<Position> {
		
		private int index;
		private PositionSet ps;
		
		public PositionIterator(PositionSet ps) {
			this.ps    = ps;
			this.index = 0;
		}
		
		public boolean hasNext() {
			while(!ps.bitset.get(index)) {
				index++;
				if(index >= ps.bitset.size()) {
					return false;
				}
			}
			return true;
		}
		
		public Position next() {
			if(!hasNext()) {
				throw new NoSuchElementException();
			} else {
				index++;
				return ps.indexToPosition(index-1);
			}
		}
		
		public void remove() {
			throw new UnsupportedOperationException();
		}
		
	}
	
	
	
	private BitSet bitset;		    
	private int nPositions;		    /* Anzahl mglicher Positionen */
	private int minX, maxX, minY, maxY; /* Wertebereich fr x,y */
	
	/**
	 * Erzeugt eine neue leere Menge von Positionen. 
	 * Mgliche Positionen liegen im Bereich (<I>minX</I>, <I>minY</I>) - (<I>maxX</I>,<I>maxY</I>)
	 * @param minX 
	 * @param maxX 
	 * @param minY 
	 * @param maxY 
	 */
	public PositionSet(int minX, int maxX, int minY, int maxY) {
		this(minX, maxX, minY, maxY, false);
	}
	/**
	 * 
	 * @param minX 
	 * @param maxX 
	 * @param minY 
	 * @param maxY 
	 * @param containsAll true - Menge enthlt alle mglichen Positionen<br>
	 * false - Menge ist leer.
	 */
	public PositionSet(int minX, int maxX, int minY, int maxY, boolean containsAll) {
		assert minX < maxX;
		assert minY < maxY;
		
		this.minX = minX;
		this.maxX = maxX;
		this.minY = minY;
		this.maxY = maxY;
		
		nPositions = (maxX - minX) * (maxY - minY);
		bitset = new BitSet(nPositions);
		
		if(containsAll) {
			bitset.set(0, bitset.size());
		}
	}
	
	/**
	 * 
	 * @return einen Iterator ber alle Positionen in der Menge.
	 */
	public Iterator<Position> iterator() {
		return new PositionIterator(this);
	}
	
	/**
	 * Berechne Index der Position <i>p</i> im Bitset.
	 * @param p
	 * @return bitsetIndex
	 */
	private int positionToIndex(Position p) {
		int x = p.getX() - minX;
		int y = p.getY() - minY;
		return x + ((maxX - minX) * y);
	}
	
	/**
	 * Berechne die zum Bitset-Index <i>index</i> gehrige Position <i>p</i>
	 */
	private Position indexToPosition(int index) {
		int x = index % (maxX - minX);
		int y = index / (maxX - minX);
		return new Position(x + minX, y + minY);
	}

	/**
	 * Fgt Position <i>p</i> der Menge hinzu.
	 * @param p 
	 */
	public void add(Position p) {
		bitset.set(positionToIndex(p));
	}
	
	/**
	 * Entfernt <i>p</i> aus der Menge.
	 * @param p 
	 */
	public void remove(Position p) {
		bitset.clear(positionToIndex(p));
	}
	
	/**
	 * 
	 * @param p eine Position
	 * @return true - falls <i>p</i> in der Menge.
	 * false - sonst.
	 */
	public boolean contains(Position p) {
		return bitset.get(positionToIndex(p));
	}
	
}
