import java.util.LinkedList;

public class CSP {

	LinkedList<BinConstraint> constraints;
		
	// Roboterposition
	Variable x, y;
	
	// Konstruktor
	public CSP() {
		constraints = new LinkedList<BinConstraint>();	
		x = new Variable("x", -3000,3000);
		y = new Variable("y", -2000,2000);
	}
	
	// Constraint zu CSP hinzufuegen
	public CSP addConstraint(BinConstraint c) {
		constraints.addLast(c);
		return this;	
	}
	
	// Rotation eines 2-dimensionalen Punktes um Nullpunkt (X-Koordinate)
	public static double rotx(double x, double y, double w) {
		return x * Math.cos(w) - y*Math.sin(w);
	}
	
	// analog fuer Y-Koordinate
	public static double roty(double x, double y, double w) {
		return y * Math.cos(w) + x*Math.sin(w);
	}
	
	// berechnet die Breite der Bounding-Box fuer Flagge bei gegebener Neigung
	public static double calcFlagWidth(double w) {
		return Math.abs(rotx(0,0,w)-rotx(Constants.flagWidth,-Constants.flagHeight,w));
	}	
	
	/* Berechnung der Distanz einer Flagge abhaengig vom Winkel zwischen den Raendern ihres Bildes und dem Neigungswinkel (Formeln siehe Dokumentation) */
	public static double calcDistToFlag(double w, double horiz) {
		return calcFlagWidth(horiz)*(Math.sin((Math.toRadians(180)-w)/2)/Math.sin(w));
	}
	
	/* fuege Constraints fuer eine "gesichtete" Flagge flag hinzu, bei gegebener Bildgroesse,
	linkem und rechten Rand des Bildes der Flagge (mm) und Bildneigung (Horizont) */
	public CSP addFlagConstraints(Flag flag, double sxmm, double symm, double leftmm, double rightmm, double horiz) {
		// berechne Winkel zu Flaggenraendern
		double leftA = (Constants.cameraAngleX/sxmm)*leftmm;
		double rightA = (Constants.cameraAngleX/sxmm)*rightmm;
		
		// berechne minimalen und maximalen Winkel zwischen Flaggenraendern
		double minWidth = rightA-leftA - Constants.cameraError*2;
		double maxWidth = rightA-leftA + Constants.cameraError*2;
		
		// setze den Distanzintervall zwischen Flagge und Kamera
		flag.dist = new Interval((int)calcDistToFlag(maxWidth,horiz), (int)calcDistToFlag(minWidth,horiz));
		
		// fuege entsprechende Constraints hinzu		
		addConstraint(new XDistToReferencePointConstraint(x,y,flag));
		addConstraint(new YDistToReferencePointConstraint(y,x,flag));	
		
		/* Bereichsbeschraenkung auf Schnittflaeche der Umquadrate der Flaggen und der groessten moeglichen Entfernung zur jeweiligen Flagge */
		x.lower=max(flag.minX(),x.lower);
		x.upper=min(flag.maxX(),x.upper);
		y.lower=max(flag.minY(),y.lower);
		y.upper=min(flag.maxY(),y.upper);
		
		return this;
	}		

	// berechnet die Hoehe der Bounding-Box fuer Ecke bei gegebener Neigung
	public static double calcCornerHeight(double w) {
		return Math.abs(roty(0,Constants.wallHeight,w));
	}	
	
	/* Berechnung der Distanz einer Ecke abhaengig vom Winkel zwischen den Raendern ihres Bildes und dem Neigungswinkel (Formeln siehe Dokumentation) */
	public static double calcDistToCorner(double w, double horiz) {
		return calcCornerHeight(horiz)*(Math.sin((Math.toRadians(180)-w)/2)/Math.sin(w));
	}
		
	public CSP addCornerConstraints(Corner corner, double sxmm, double symm, double uppermm, double lowermm, double horiz) {
		// berechne Winkel zu Flaggenraendern
		double lowerA = (Constants.cameraAngleY/symm)*lowermm;
		double upperA = (Constants.cameraAngleY/symm)*uppermm;
		
		// berechne minimalen und maximalen Winkel zwischen Flaggenraendern
		double minHeight = lowerA-upperA - Constants.cameraError*2;
		double maxHeight = lowerA-upperA + Constants.cameraError*2;
		
		// setze den Distanzintervall zwischen Flagge und Kamera
		corner.dist = new Interval((int)calcDistToCorner(maxHeight,horiz), (int)calcDistToCorner(minHeight,horiz));

		// fuege entsprechende Constraints hinzu		
		addConstraint(new XDistToReferencePointConstraint(x,y,corner));
		addConstraint(new YDistToReferencePointConstraint(y,x,corner));	
		
		/* Bereichsbeschraenkung auf Schnittflaeche der Umquadrate der Flaggen und der groessten moeglichen Entfernung zur jeweiligen Flagge */
		x.lower=max(corner.minX(),x.lower);
		x.upper=min(corner.maxX(),x.upper);
		y.lower=max(corner.minY(),y.lower);
		y.upper=min(corner.maxY(),y.upper);
		
		return this;
	
	}
	
	// versucht, das Constraint-Problem zu loesen
	public void solve() throws InconsistencyException {
		
		boolean isChanged;
		do {
			// noch keine Veraenderung der Variablenintervalle
			isChanged = false;
			// alle Constraints anwenden
			for(BinConstraint c : constraints)
				// Veraenderung eingetreten? -> merken
				isChanged = c.apply() || isChanged;
		}
		while (isChanged); // wenn sich noch etwas veraendert hat, wiederholen
	}
			
	// gibt die Variablen aus
	public void dump() {
		System.out.println(" " + x + " " + y);
	}
	
	// Hilfsmethoden, klar
	public static int max(int x, int y) {
		if (x > y) 
			return x;
		else
			return y;
	}
	
	public static int min(int x, int y) {
		if (x > y)
			return y;
		else
			return x;
	}
}
