//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Waitq.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/26

	\brief Implementation of classes in Waitq.h

	\sa Waitq.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <odemx/base/InSort.h>
#include <odemx/util/ErrorHandling.h>
#include <odemx/synchronization/Waitq.h>

#include <string>
#include <cassert>

using namespace std;
using namespace odemx;

Waitq::Waitq(Simulation* s, Label l, WaitqObserver* o/* = 0*/) :
	Observable<WaitqObserver>(o),
	StatisticObject(s),
	env(s),
	masterWait(s, (string(l) + "_master_queue").c_str(), &priOrder),
	slaveWait(s, (string(l) + "_slave_queue").c_str(),&priOrder)
{
	assert(s!=0);

	// set label
	DefLabeledObject::setLabel(s, l);

	// trace
	getTrace()->mark(this, markCreate);

	// observer
	_obsForEach(WaitqObserver, Create(this));

	// init statistic
	reset();
}

Waitq::~Waitq() {
	// trace
	getTrace()->mark(this, markDestroy);

	// observer
	_obsForEach(WaitqObserver, Destroy(this));
}

bool Waitq::wait() {
	Process* slave=getCurrentProcess();

	// trace
	getTrace()->mark(this, markWait, slave);

	// observer
	_obsForEach(WaitqObserver, Wait(this, slave));

	// awake master
	awakeAll(&masterWait);

	// slave waits for master
	slaveWait.inSort(slave);

	// entrance time in queue is stored equal to execution time
	// sleep
	slave->sleep();

	if (slave->isInterrupted()) {
		slaveWait.remove(slave);

		return false;
	}

	return true;
}

Process* Waitq::coopt(Selection sel/*=0*/) {
	return synch(true, sel);
}

Process* Waitq::avail(Selection sel/*=0*/) {
	return synch(false, sel);
}

Process* Waitq::synch(bool wait, Selection sel) {
	Process* master=getCurrentProcess();
	Process* slave=getSlave(master, sel);

	if (slave==0) {
		// trace
		if (wait) getTrace()->mark(this, (sel ? markCooptSelFail : markCooptFail), master);
		else getTrace()->mark(this, (sel ? markAvailSelFail : markAvailFail) , master);

		// observer
		if (wait) {
			if (sel==0)	_obsForEach(WaitqObserver, CooptFail(this, master))
			else _obsForEach(WaitqObserver, CooptSelFail(this, master));
		} else {
			if (sel==0)	_obsForEach(WaitqObserver, AvailFail(this, master))
			else _obsForEach(WaitqObserver, AvailSelFail(this, master));
		}

		if (wait) {
			// master is waiting for slave
			masterWait.inSort(master);

			do {
				master->sleep();

				if (master->isInterrupted()) {
					masterWait.remove(master);

					return 0;
				}

				slave=getSlave(master, sel);
			} while (slave==0);

			// block released
			masterWait.remove(master);
		} else
			return 0;
	}

	// get slave
	slaveWait.remove(slave);

	// statistics
	updateStatistics(master, slave);

	// trace
	if (wait) traceSUCCEED( (sel ? markCooptSelSucceed : markCooptSucceed), master, slave);
	else traceSUCCEED( (sel ? markAvailSelSucceed : markAvailSucceed), master, slave);

	// observer
	if (wait) {
		if (sel==0)	_obsForEach(WaitqObserver, CooptSucceed(this, master, slave))
		else _obsForEach(WaitqObserver, CooptSelSucceed(this, master, slave));
	} else {
		if (sel==0)	_obsForEach(WaitqObserver, AvailSucceed(this, master))
		else _obsForEach(WaitqObserver, AvailSelSucceed(this, master));
	}

	return slave;
}

Process* Waitq::getSlave(Process* master/*=0*/, Selection sel/*=0*/) {
	if (slaveWait.isEmpty())
		return 0;

	if (sel==0)
		return slaveWait.getTop();

	if (master==0)
		return 0;

	std::list<Process*>::const_iterator i;
	for (i=slaveWait.getList().begin(); i!=slaveWait.getList().end(); ++i) {
		if ((master->*sel)(*i))
			return *i;
	}

	return 0;
}

void Waitq::reset() {
	StatisticObject::reset();

	slaveWait.reset();
	masterWait.reset();

	synchs = zeroMasters = zeroSlaves = 0;
	sumMasterWait = sumSlaveWait = 0.0;
}

void Waitq::report(Report* r) {
	assert (r!=0);

	masterWait.report(r);
	slaveWait.report(r);

	static const char* labels[] = {"Name", "Reset at", "Master Queue", "Slave Queue",
								   "Number of Synch.",
								   "Zero wait masters", "Avg masters wait",
								   "Zero wait slaves", "Avg slaves wait"};

	static const ColumnType types[] = {STRING, REAL, STRING, STRING,
									   INTEGER,
									   INTEGER, REAL,
									   INTEGER, REAL};

	static utTableDef def(sizeof(labels)/sizeof(char*), labels, types);

	Table* t=r->createTable("Waitq Statistics", &def);
	assert(t);

	*t << getLabel() << getResetTime() << masterWait.getLabel() << slaveWait.getLabel();
	*t << getNumberOfSynch();
	*t << getZeroWaitMasters() << getAVMasterWaitTime();
	*t << getZeroWaitSlaves() << getAVSlaveWaitTime();
}

void Waitq::updateStatistics(Process* master, Process* slave) {
	SimTime now = env->getTime();

	// master and slave synchronised (served)
	synchs++;

	// calculate waiting times
	double masterWaitTime = master->getDequeueTime()-master->getEnqueueTime();
	double slaveWaitTime = slave->getDequeueTime()-slave->getEnqueueTime();

	// update waiting time statistics
	if (masterWaitTime<=0.0 || master->getDequeueTime()!=now)
		zeroMasters++;
	else
		sumMasterWait += masterWaitTime;

	if (slaveWaitTime == 0.0)
		zeroSlaves++;
	else
		sumSlaveWait+=slaveWaitTime;
}

void Waitq::traceSUCCEED(MarkType m, Process* master, Process* slave) {
	getTrace()->beginMark(this, m);
	getTrace()->addTag(tagMaster, master);
	getTrace()->addTag(tagSlave, slave);
	getTrace()->endMark();
}

const MarkTypeId Waitq::baseMarkId = 1000;

const MarkType Waitq::markCreate = MarkType("create", baseMarkId+1, typeid(Waitq));
const MarkType Waitq::markDestroy = MarkType("destroy", baseMarkId+2, typeid(Waitq));

const MarkType Waitq::markWait = MarkType("wait", baseMarkId+10, typeid(Waitq));
const MarkType Waitq::markCooptFail = MarkType("cooptFail", baseMarkId+11, typeid(Waitq));
const MarkType Waitq::markCooptSucceed = MarkType("cooptSucceed", baseMarkId+12, typeid(Waitq));
const MarkType Waitq::markAvailFail = MarkType("availFail", baseMarkId+13, typeid(Waitq));
const MarkType Waitq::markAvailSucceed = MarkType("availSucceed", baseMarkId+14, typeid(Waitq));

const MarkType Waitq::markCooptSelFail = MarkType("cooptSelFail", baseMarkId+15, typeid(Waitq));
const MarkType Waitq::markCooptSelSucceed = MarkType("cooptSelSucceed", baseMarkId+16, typeid(Waitq));
const MarkType Waitq::markAvailSelFail = MarkType("availSelFail", baseMarkId+17, typeid(Waitq));
const MarkType Waitq::markAvailSelSucceed = MarkType("availSelSucceed", baseMarkId+18, typeid(Waitq));

const TagId Waitq::baseTagId = 1000;

const Tag Waitq::tagMaster = Tag(baseTagId+1);
const Tag Waitq::tagSlave = Tag(baseTagId+2);

