//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Condq.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/07/11

	\brief Implementation of classes in Condq.h

	\sa Condq.h

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/


#include <odemx/base/InSort.h>
#include <odemx/util/ErrorHandling.h>
#include <odemx/synchronization/Condq.h>

#include <string>
#include <cassert>

using namespace std;
using namespace odemx;

Condq::Condq(Simulation* s, Label l, CondqObserver* o/* = 0*/) :
	Observable<CondqObserver>(o), StatisticObject(s),
	env(s), processes(s, (string(l) + "_queue").c_str(), &priOrder)
{
	assert(s!=0);

	// set label
	DefLabeledObject::setLabel(s, l);

	// trace
	getTrace()->mark(this, markCreate);

	// observer
	_obsForEach(CondqObserver, Create(this));

	// init statistics
	reset();
}

Condq::~Condq() {
	// trace
	getTrace()->mark(this, markDestroy);

	// observer
	_obsForEach(CondqObserver, Destroy(this));
}

bool Condq::wait(Condition cond) {
	// trace
	getTrace()->mark(this, markWait, getCurrentProcess());

	// observer
	_obsForEach(CondqObserver, Wait(this, getCurrentProcess()));

	// wait for cond is true
	processes.inSort(getCurrentProcess());

	// statistics
	SimTime t=env->getTime();

	while (!(getCurrentProcess()->*cond)() ) {
		getCurrentProcess()->sleep();

		if (getCurrentProcess()->isInterrupted()) {
			processes.remove(getCurrentProcess());

			return false;
		}
	}

	processes.remove(getCurrentProcess());

	// statistics
	t = env->getTime() - t;
	sumWaitTime += t;
	if (t==0)
		zeros++;

	users++;

	// trace
	getTrace()->mark(this, markContinue, getCurrentProcess());

	// observer
	_obsForEach(CondqObserver, Continue(this, getCurrentProcess()));

	return true;
}

void Condq::signal() {
	// trace
	getTrace()->mark(this, markSignal, getCurrentProcess());

	// observer
	_obsForEach(CondqObserver, Signal(this, getCurrentProcess()));

	// statistics
	signals++;

	if (processes.isEmpty())
		return;

	// test conditions
	awakeAll(&processes);
}

void Condq::reset() {
	StatisticObject::reset();
	processes.reset();

	users = processes.getLength();
	zeros = signals = 0;
	sumWaitTime = 0.0;
}

void Condq::report(Report* r) {
	assert (r!=0);
	processes.report(r);

	static const char* labels[] = {"Name", "Reset at", "Queue", "Users", "Signals",
								   "Zero waiting time user", "Average waiting time"};

	static const ColumnType types[] = {STRING, REAL, STRING, INTEGER, INTEGER,
									   INTEGER, REAL};

	static utTableDef def(sizeof(labels)/sizeof(char*), labels, types);
	Table* t=r->createTable("Condq Statistics", &def);
	assert(t);

	*t << getLabel() << getResetTime() << processes.getLabel() << getUsers() << getSignals();
	*t << getZeroWait() << getAVWaitTime();
}

const MarkTypeId Condq::baseMarkId = 1000;

const MarkType Condq::markCreate = MarkType("create", baseMarkId+1, typeid(Condq));
const MarkType Condq::markDestroy = MarkType("destroy", baseMarkId+2, typeid(Condq));

const MarkType Condq::markWait = MarkType("wait", baseMarkId+3, typeid(Condq));
const MarkType Condq::markContinue = MarkType("continue", baseMarkId+4, typeid(Condq));
const MarkType Condq::markSignal = MarkType("signal", baseMarkId+5, typeid(Condq));

