//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2003, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Idist.cpp

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2003/05/09

	\brief Implementation of classes in Idist.h

	\sa Idist.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#include <cmath>

#include <odemx/random/Idist.h>
#include <odemx/util/ErrorHandling.h>

using namespace odemx;

Randint::Randint (DistContext* c, Label title, int ra, int rb) : Idist(c, title) {
	if (ra>rb) {
		int help=ra; ra=rb; rb=help;
		warning("Randint: lower bound is > upper bound; bounds swapped");
	}
	a=ra;
	b=rb;
	uses=0;
	zyqspan=b-a+1;
}

int Randint::sample(){
	uses++;
	return (int)floor(zyqspan*getSample())+a;
}

void Randint::report(Report* r) {
	Table* t=getTable(r);

	(*t) << "Randint" << long(getUses()) << long(getSeed()) << getA() << getB() << ENDL;
}

Poisson::Poisson (DistContext* c, Label title, double pa) : Idist(c, title) {
	if (pa<=0.0) {
		pa=(pa<0.0) ? -pa : 0.0010;
		warning("Poisson: mean a is <= 0.0; changed to -a");
	}
	uses=0;
	a=pa;
}

int Poisson::sample() {
	double q,r;
	int m=0;

	uses++;
	r=exp(-a);
	q=1.0;
	while (q>=r) {
		q*=getSample();
		m+=1;
	};
	m-=1;
	return m;
};

void Poisson::report(Report* r) {
	Table* t=getTable(r);

	(*t) << "Poisson" << long(getUses()) << long(getSeed()) << a << ENDL;
}

Draw::Draw(DistContext* c, Label title, double da) : Idist(c, title) {
	uses=0;
	a=da;
};

int Draw::sample() {
	uses++;
	return (a>getSample() ? 1 : 0);
}

void Draw::report(Report* r) {
	Table* t=getTable(r);

	(*t) << "Draw" << long(getUses()) << long(getSeed()) << a << ENDL;
}

Iconst::Iconst(DistContext* c, Label title, int i) : Idist(c, title), a(i) {
	uses=0;
}

void Iconst::report(Report* r) {
	Table* t=getTable(r);

	(*t) << "Iconst" << long(getUses()) << long(getSeed()) << a << ENDL;
}

