//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Process.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/01/30

	\brief Declaration of odemx::Process and odemx::ProcessObserver

	\sa Process.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_PROCESS_INCLUDED
#define ODEMX_PROCESS_INCLUDED

#include <odemx/coroutine/Coroutine.h>
#include <odemx/base/Sched.h>
#include <odemx/base/Simulation.h>
#include <odemx/base/ProcessQueue.h>
#include <odemx/synchronization/Memo.h>
#include <odemx/util/Observable.h>
#include <odemx/util/LabeledObject.h>

namespace odemx {

	class ProcessObserver;

	/** \class Process

		\ingroup base

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Process is the base class for all user processes in a model.

		\note %Process supports Observation.
		\note %Process supports Trace.

		\sa Event, Condition, Selection and Simulation

		%Process is the base class for all user defined processes in a model. A user
		creates a derived class for every different type of process of its model. In
		its classes the user has to provide the behaviour by implementing the main()
		method. The behaviour is executed according to an execution schedule
		(ExecutionList) at a certain point in simulation time. If multiple processes
		share the same execution time, the order of execution is defined by the way
		the processes have been scheduled. A process can be scheduled with respect
		to its priority FirstInFirstOut or LastInFirstOut at a point in time, or
		in relation to an already scheduled partner process (immediately after or before
		that process).

		\par Example:
		\include Base1.cpp

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Process : public Sched,
					public Coroutine,
					public Observable<ProcessObserver> {
		// Types
	public:
		/**
			\brief Process states

			A process passes different process states during its life time
			(between new and delete). After construction a process is CREATED.
			In this state a process isn't scheduled for execution and hasn't been executed
			yet. In RUNNABLE a process is scheduled for execution. If a process is executed
			its state is CURRENT. In every simulation there is no more than one process
			executed at a time. A process in IDLE state is not scheduled for execution. It is
			inactive. If a process has finished or is terminated its state is TERMINATED.
			Such a process is not scheduled for execution and cannot be scheduled
			anymore.
		*/
		enum ProcessState {
			CREATED,	///< Initial state   0
			CURRENT,	///< Active state    1
			RUNNABLE,	///< Scheduled state 2
			IDLE,		///< Wait state      3
			TERMINATED	///< Final state     4
		};

		// Public Interface
	public:
		/**
			\brief Construction
			\param s
				pointer to the Simulation object
			\param l
				label of this object
			\param o
				initial observer
		*/
		Process(Simulation* s, Label l, ProcessObserver* o = 0);
		/// Destruction
		~Process();

		/**
			\brief Get process state
			\return current process state
		*/
		ProcessState getProcessState() const;

		/**
			\name Process Scheduling

			These functions are used to schedule a process
			in simulation time.

			@{
		*/
		/**
			\brief Immediate activation

			This function schedules the process at the current time before
			all other processes with the same priority (LIFO). If the current
			process (A) does not have a higher priority than the process the
			function is called at (B) the following will happen:
			\li B is scheduled before A
			\li the state of A changes to RUNNABLE while the state
				of B changes to CURRENT
			\li the main() function of A is stopped (the point is saved)
			\li the main() function of B is executed from the point it
				has been left last time (or from the beginning)
		*/
		void activate();

		/**
			\brief Activation in (relative) time \p t
			\param t
				delay for activation

			This function schedules the process at the simtime \c now + \p t
			before all other processes with the same priority (LIFO). If
			\p t is 0 this function is equal to activate().

			\sa activate()
		*/
		void activateIn(SimTime t);

		/**
			\brief Activation at (absolute) time \p t
			\param t
				time for activation

			This function schedules the process at the simtime \p t
			before all other processes with the same priority (LIFO). If
			\p t is \c now this function is equal to activate().

			\sa activate()
		*/
		void activateAt(SimTime t);

		/**
			\brief Activation before process \p p
			\param p
				pointer to partner process

			This function schedules the process just before
			process \p p. If necessary the priority of the process
			is changed. If \p p is this process this function
			does nothing. If process \p p is not scheduled ODEMx
			reports an error.
		*/
		void activateBefore(Process* p);

		/**
			\brief Activation after process \p p
			\param p
				pointer to refrence process

			This function schedules the process just after
			process \p p. If necessary the priority of this process
			is changed. If \p p is this process this function
			does nothing. If process \p p is not scheduled ODEMx
			reports an error.
		*/
		void activateAfter(Process* p);

		/**
			\brief Immediate activation (FIFO)

			This function schedules the process at the current time
			after all other processes with the same priority.
		*/
		void hold();

		/**
			\brief Activation (FIFO) in (relative) time \p t

			This function schedules the process at the time \c now + \p t
			after all other processes with the same priority. If \p t
			is 0 this function is equal to hold().

			\sa hold()
		*/
		void holdFor(SimTime t);

		/**
			\brief Activation (FIFO) at (absolute) time \p t

			This function schedules the process at simtime \p t
			after all other processes with the same priority. If \p t
			is \c now this function is equal to hold().

			\sa hold()
		*/
		void holdUntil(SimTime t);

		/**
			\brief Deactivation

			This function removes the process from execution schedule
			and changes its state into IDLE. The next RUNNABLE process
			in the schedule is than activated.
		*/
		void sleep();

		/**
			\brief Interrupt

			This function interrupts the process, which results in an immediate
			activation [activate()]. A Process can determine whether it was
			interrupted by isInterrupted(). A Process must handle interrupts
			on its own. It should do so after activate... and hold... operation.

			\sa activateAt(), activateUntil(), holdFor() and holdUntil()
		*/
		virtual void interrupt();

		/**
			\brief Termination

			This function terminates the process. The process is than removed from
			execution schedule and its state is changed to TERMINATED.
		*/
		void cancel();
		//@}

		/**
			\name Priority

			If the priority is changed the position of the process in the
			execution schedule and in queues is updated. This can cause a
			switch to another process.

			@{
		*/
		/**
			\brief Get priority
			\return
				process priority

			This function returns the current process priority.
		*/
		Priority getPriority() const;

		/**
			\brief Set new priority
			\return
				previous process priority

			This function changes the process priority. The position
			in the execution schedule and in queues is effected by
			priority changes.
		*/
		Priority setPriority(Priority newPriority);
		//@}

		/**
			\name Execution Time

			@{
		*/
		/**
			\brief Get execution time
			\return
				execution time; 0 if not RUNNABLE or CURRENT

			This function returns the execution time of the process.
			A process is scheduled for execution at its execution
			time.
		*/
		SimTime getExecutionTime() const;
		//@}

		/**
			\name Current System State

			@{
		*/
		/**
			\brief Get currently active process
			\return
				pointer to the current process

			This function returns the currently active process (the one
			in state CURRENT).
		*/
		Process* getCurrentProcess();

		/**
			\brief Get current simtime
			\return
				current simulation time

			This function returns the current simulation time.
		*/
		SimTime getCurrentTime();

		/**
			\brief Get simulation of this process
			\return
				pointer to simulation object of this process

			This function returns the pointer to the simulation
			object of this process.
		*/
		Simulation* getSimulation();
		//@}

		/**
			\name Interrupt handling

			An Interrupt is caused by the scheduling function interrupt().
			The interrupter (Process) is stored. The interrupt-state is
			automatically reset by the next scheduling function called. If the
			Process is interrupted but the interrupter is 0 than the Process
			was interrupted by the simulation (environment).

			@{
		*/
		/**
			\brief Get interrupt state
			\return
				interrupt state

			This function returns the interrupt state of the process.
		*/
		bool isInterrupted() const {return interrupted;}

		/**
			\brief Get process which called interrupt()
			\return
				pointer to process which called interrupt()

			This function returns the process which called interrupt().
		*/
		Process* getInterrupter() const {return interrupter;}

		/**
			\brief Reset interrupt state
			\return
				interrupt state

			This function returns the interrupt state of this
			process. The interrupt state is reset automatically by
			scheduling functions.
		*/
		void resetInterrupt() {interrupted=false; interrupter=0;}
		//@}


		/**
			\name Suspend and alert
			\author Ronald Kluth

			A concept of suspending a process until a resource becomes
			available or a timeout occurs.

			@{
		
		*/

		/**
			\brief Suspend process and wait for alert
			\param m0
				pointer to object of Memo-derived class, must be given
			\param m1
				pointer to object of Memo-derived class, optional
			\param m2
				pointer to object of Memo-derived class, optional
			\param m3
				pointer to object of Memo-derived class, optional
			\param m4
				pointer to object of Memo-derived class, optional
			\param m5
				pointer to object of Memo-derived class, optional
			
			\return
				pointer to alerting memo object 

			This function is used to suspend a process while waiting for
			one of the given Memo objects to become available. 
			That can either be a PortHead or PortTail, or a Timer event. 
			The function returns the pointer to	the first available Memo object.
			If the process was interrupted, this function returns 0. Hence,
			a process should always check for interruption after calling \p wait(). 
		*/

		Memo* wait( Memo* m0, Memo* m1 = 0, Memo* m2 = 0, 
				    Memo* m3 = 0, Memo* m4 = 0, Memo* m5 = 0 );

		/**
			\brief Suspend process and wait for alert
			\param memvec
				vector containing pointers to objects of Memo-derived classes 
			\return
				pointer to alerting memo object 

			Overloaded version of the above function. Instead of several
			pointers to Memo objects, a vector containing all the Memo
			objects can be given as a parameter.
		*/

		Memo* wait( MemoVector* memvec );

		/**
			\brief Reschedule process when a Memo object becomes available 
			\param theAlerter
				pointer to the calling Memo object

			This function is called when Memo-derived objects call alert().
			It wakes up a suspended process and reschedules it at the 
			current simulation time.
		*/
		virtual void alertProcess( Memo* theAlerter );

		/**
			\brief Check alert state of the process
			\return
				Alert state of the process
			
			This function checks the alert state of the process. It returns 
			true if alertProcess() was called on this object. The library uses
			it in Process::wait() to determine whether a process was alerted
			by a Memo object.
		*/
		bool isAlerted() const;
		
		/**
			\brief Determine who woke the process with an alert
			\return
				pointer to the available Memo object
			
			This function is used in Process::wait() to determine 
			the currently available Memo object. 
		*/
		Memo* getAlerter() const;

		/**
			\brief Reset the internal values alerted and alerter

			This function is used in Process::wait() to avoid the use
			of outdated alert state or a previous alerter. It is recommended
			to call this function before and after waiting and receiving
			an alert.
		*/
		void resetAlert();
		//@}

		/**
			\brief main() function status
			\return
				true if the main() function has returned

			This function returns true if the main() function of
			this process has returned (is finished).
		*/
		bool hasReturned() const {return validReturn;}

		/**
			\brief Get return value of main() function
			\return
				return value from main()

			This function returns the return value of the main()
			function. Use the hasReturned() function to check
			whether the main() function has already returned.
		*/
		int getReturnValue() const;

		/**
			\brief Get pointer to Trace
			\return
				pointer to Trace object
		*/
		virtual Trace* getTrace() const {return env;}

		/**
			\brief Get queue
			\return
				pointer to ProcessQueue

			If the process is in a queue this function returns
			the pointer to the queue.
		*/
		ProcessQueue* getQueue() const {return q;}

		/**
			\brief Get enqueue-time
			\return
				time at which the process entered a queue

			If the process is in a queue this function returns
			the time at which the process entered the queue. This
			function is used by synchronisation objects.
		*/
		SimTime getEnqueueTime() const {return q_int;}

		/**
			\brief Get dequeue-time
			\return
				time at which the process left a queue

			If the process was in a queue this function returns
			the time it left the queue. This function is used by
			synchronisation objects.
		*/
		SimTime getDequeueTime() const {return q_out;}

	protected:
		/**
			\brief User defined process behaviour

			This function must be implemented by the user to define the
			behaviour of a process. It is executed by ODEMx when the process
			becomes the current process. If the process calls a scheduling
			function (direct or indirect, at itself or another process)
			the %main() function is delayed until the process is executed
			again. The sequence in which different processes are executed
			is determined by the execution schedule.

			\sa Process Scheduling
		*/
		virtual int main() = 0;


		// Internal Interface

		friend class ExecutionList;
		/**
			\brief Set execution time
			\return
				previous execution time

		 	This function is used by the library. Don't
			call it directly! Use the scheduling functions
			instead.
		*/
		SimTime setExecutionTime( SimTime time );

		friend class Simulation;
		/**
			\brief Execution of this process

			This function is used by the library. Don't
			call is directly! The execution is managed
			by the process scheduling.
		*/
		void execute();
	public:
		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace Process events.
			A TraceConsumer can use these constants to identify trace
			events sent	by Process.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markCreate;
		static const MarkType markDestroy;

		static const MarkType markActivate;
		static const MarkType markActivateIn;
		static const MarkType markActivateAt;
		static const MarkType markActivateBefore;
		static const MarkType markActivateAfter;
		static const MarkType markHold;
		static const MarkType markHoldFor;
		static const MarkType markHoldUntil;
		static const MarkType markInterrupt;
		static const MarkType markSleep;
		static const MarkType markCancel;
		
		static const MarkType markExecute;

		static const MarkType markWait;
		static const MarkType markAvailable;
		static const MarkType markAlert;

		static const MarkType markChangeProcessState;
		static const MarkType markChangePriority;
		static const MarkType markChangeExTime;

		static const TagId baseTagId;

		static const Tag tagCurrent;
		static const Tag tagAbsSimTime;
		static const Tag tagRelSimTime;
		static const Tag tagPartner;
		//@}

		// Implementation
	private:
		virtual void start(); ///< coroutine entry-point implementation

	private:
		ProcessState processState; ///< process state
		Priority p; ///< process priority
		SimTime t; ///< process execution time
		Simulation* env; ///< process simulation
		int returnValue; ///< return value of main()
		ProcessQueue* q; ///< pointer to queue if process is waiting
		SimTime q_int; ///< enqueue-time
		SimTime q_out; ///< dequeue-time
		bool validReturn; ///< return value is valid
		bool interrupted; ///< Process was interrupted
		Process* interrupter; ///< Process was interrupted by interrupter (0 -> by Simulation)
		bool alerted; ///< Process was alerted
		Memo* alerter; ///< Process alerter
		bool internalMemVec; ///< check whether to delete Memo Vector in wait()

	private:
		/**
			\brief set process state
			\return true if successful
		*/
		bool setProcessState(ProcessState newState);

		friend class ProcessQueue;
		friend class Queue;
		/**
			\name ProcessQueue Management
			\note No more than one ProcessQueue at a time is supported. (provisional)

			@{
		*/
		void enqueue(ProcessQueue* nq); ///< process enters queue \p nq
		void dequeue(ProcessQueue* nq); ///< process leaves queue \p nq
		//@}

		/**
			\name Trace help functions
			@{
		*/
		/// get current sched object or simulation
		TraceProducer* getPartner();

		// Complex Trace Marks
		void traceActivateIn(SimTime t); ///< send activate in mark
		void traceActivateAt(SimTime t); ///< send activate at mark
		void traceActivateBefore(Process* p); ///< send activate before mark
		void traceActivateAfter(Process* p); ///< send activate after mark
		void traceHoldFor(SimTime t); ///< send hold for mark
		void traceHoldUntil(SimTime t); ///< send hold until mark
		void traceWait(MemoVector* memvec); ///< send wait mark with partners
		//@}
	};

	/**
		\brief Member-pointer type for coding conditions

		This pointer to member-function type is used for coding
		conditions. The condition-function should return true
		if the condition is fulfilled.
	*/
	typedef bool (Process::*Condition)();

	/**
		\brief Member-pointer type for coding selections
		\param partner
			pointer to process to select

		This pointer to member-function type is used for coding
		selections. The condition-function should return true
		if the \p partner process fits the selection.
	*/
	typedef bool (Process::*Selection)(Process* partner);

	/** \interface ProcessObserver

		\author RalfGerstenberger
		<!-- [\author <author>]* -->

		\brief Observer for Process specific events.

		<!-- [\note {notes}]* -->

		\sa Process

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class ProcessObserver : public CoroutineObserver {
	public:
		virtual ~ProcessObserver() {}
		virtual void onCreate(Process* sender) {}; ///< Construction
		virtual void onDestroy(Process* sender) {}; ///< Destruction

		// Activation (LIFO)
		virtual void onActivate(Process* sender) {}; ///< Immediate activation
		virtual void onActivateIn(Process* sender, SimTime t) {}; ///< Activation in
		virtual void onActivateAt(Process* sender, SimTime t) {}; ///< Activation at

		// Activation in Releation to a Partner Process
		virtual void onActivateBefore(Process* sender, Process* p) {}; ///< Activation before
		virtual void onActivateAfter(Process* sender, Process* p) {}; ///< Activation after

		// Activation (FIFO)
		virtual void onHold(Process* sender) {}; ///< hold
		virtual void onHoldFor(Process* sender, SimTime t) {}; ///< Hold for
		virtual void onHoldUntil(Process* sender, SimTime t) {}; ///< Hold until

		virtual void onInterrupt(Process* sender) {}; ///< Process interrupted
		virtual void onSleep(Process* sender) {}; ///< Process deactivated
		virtual void onCancel(Process* sender) {}; ///< Process terminated

		virtual void onExecute(Process* sender) {}; ///< Process executed

		virtual void onWait(Process* sender, MemoVector* memvec) {}; ///< Process waiting for alert
		virtual void onAlert(Process* sender, Memo* alerter) {}; ///< Process deactivated

		/// Process state change
		virtual void onChangeProcessState(Process* sender, Process::ProcessState oldState, Process::ProcessState newState) {};
		/// Process priority change
		virtual void onChangePriority(Process* sender, Priority oldPriority, Priority newPriority) {};
		/// Process execution time change
		virtual void onChangeExecutionTime(Process* sender, SimTime oldExecutionTime, SimTime newExecutionTime) {};
	};
}

#endif

