using namespace odemx;
using namespace std;

/********************* PortHead Template Implementation ***********************/

template < typename ElementType >
PortTailT< ElementType >::PortTailT( Simulation* sim, Label portName, PortMode m, 
					unsigned int max, PortTailObserverT< ElementType >* pto ):
	Memo( sim, Memo::PORTTAIL ),
	Observable< PortTailObserverT< ElementType > >( pto ),
	mode( m ),
	env( sim )
	{
		// set label and context
		DefLabeledObject::setLabel( sim, (string(portName) + " (tail)").c_str() );
		
		// trace
		getTrace() -> mark( this, markCreate );
		
		// observer
		_obsForEachT(PortTailObserverT< ElementType >, Create(this));
		
		if ( max != 0 )
			internalPort = new PortT< ElementType >( env, portName, max, 0, this );
	}
	

template < typename ElementType >
PortTailT< ElementType >::~PortTailT() 
	{
		// trace
		getTrace() -> mark( this, markDestroy );
		
		// observer
		_obsForEachT(PortTailObserverT< ElementType >, Destroy(this));
		
		// delete the PortTail and also the Port
		// if it is not also pointed to by a PortHead
		if ( internalPort && internalPort -> portHead )
			internalPort -> portTail = 0;
		else
			delete internalPort;
	}


template < typename ElementType >
int PortTailT< ElementType >::put( ElementType newElement ) {
	
	// trace
	getTrace() -> mark( this, markPut, getPartner() );
	
	// observer
	_obsForEachT(PortTailObserverT< ElementType >, Put(this, newElement));

	while ( internalPort -> elementList.size() >= internalPort -> maxCapacity ) {
		
		Sched* caller = env -> getCurrentSched();
		
		switch ( mode ) {
		case WAITING_MODE:

			if ( caller -> getSchedType() == Sched::EVENT ) {
				// error: cannot send an event to sleep
				error( "PortHead::put(): waiting mode not allowed for events" );
				return -1;
			}
			else if ( caller -> getSchedType() == Sched::PROCESS ) {
				Memo::remember( dynamic_cast<Process*>(caller) );
				dynamic_cast<Process*>(caller) -> sleep();
			}
			break;

		case ERROR_MODE:
			error( "PortTail::put(): cannot add element to full port" );
			return -1;
			
		case ZERO_MODE:
			return 0;
		}
	}
	
	// remember if port was empty
	bool wasEmpty = internalPort -> elementList.empty();

	// insert new element
	internalPort -> elementList.push_back( newElement );
	
	// update stats
	internalPort -> putCalls++;
	internalPort -> portAccum.update( internalPort -> getNowCount() );
	
	// notify waiting Processes of newly inserted element
	if ( wasEmpty && internalPort -> portHead ) {
		internalPort -> portHead -> alert();
	}
	
	return 1;
}


// provides the corresponding PortHead
// if it does not exit, create it
template < typename ElementType >
PortHeadT< ElementType >* PortTailT< ElementType >::getHead() {

	PortHeadT< ElementType >* head = internalPort -> portHead;

	// head still zero?
	if ( !head ) {
		
		// create new Head
		head = new PortHeadT< ElementType >( env, internalPort -> getLabel(), mode, 0 );  
		
		// register new Head with Tail's Port
		internalPort -> portHead = head; 
		
		// register Tail's internal Port with the Head
		head -> internalPort = internalPort;
	}
	
	return head;
}


// change
//      old_Port_head -- old_Port -- old_Port_tail
// into
//      old_Port_head -- old_Port -- new_Port_tail
//      (__getHead()___) -- new_Port -- old_Port_tail
// where
//      old_Port_tail == this
// return
//      new_Port_tail.
template < typename ElementType >
PortTailT< ElementType >* PortTailT< ElementType >::cut() {

	// trace
	getTrace() -> mark( this, markCut, getPartner() );
	
	// observer
	_obsForEachT(PortTailObserverT< ElementType >, Cut(this));

	// remember old Port
	PortT< ElementType >* oldPort = internalPort;
	
	// create a new Tail and thereby a new Port
	PortTailT< ElementType >* newTail = new PortTailT< ElementType >( env, (string("Cut ") + internalPort -> getLabel()).c_str(), mode, oldPort -> maxCapacity );
	
	// get the new Port
	PortT< ElementType >* newPort = newTail -> internalPort;

	// put the new tail on the old Port
	newTail -> internalPort = oldPort;
	oldPort -> portTail = newTail;

	// put the old tail on the new Port
	newPort -> portTail = this;
	internalPort = newPort;

	return newTail;
}


// this PortTail is supposed to be downstream to the PortHead oldHead
template < typename ElementType >
void PortTailT< ElementType >::splice( PortHeadT< ElementType >* oldHead ) {
	
	// trace
	getTrace() -> mark( this, markSplice, getPartner() );
	
	// observer
	_obsForEachT(PortTailObserverT< ElementType >, Splice(this, oldHead));
	
	oldHead -> splice( this );
}


template < typename ElementType >
unsigned int PortTailT< ElementType >::getMaxCapacity() {
	return internalPort -> maxCapacity; 
}


template < typename ElementType >
void PortTailT< ElementType >::setMaxCapacity( unsigned int newMaximum ) {
	
	unsigned int oldCapacity = internalPort -> maxCapacity;
	
	// trace
	getTrace() -> mark( this, markChangeMaxCapacity, newMaximum, oldCapacity );
	
	// observer
	_obsAForEachT(PortTailObserverT< ElementType >, MaxCapacity, oldCapacity, newMaximum);
	
	internalPort -> maxCapacity = newMaximum; 
}


template < typename ElementType >
unsigned int PortTailT< ElementType >::getAvailableSpace() {
	return ( internalPort -> maxCapacity ) - ( internalPort -> elementList.size() ); 
}


template < typename ElementType >
PortMode PortTailT< ElementType >::getMode() {
	return mode;
}


template < typename ElementType >
void PortTailT< ElementType >::setMode( PortMode newMode ) {
	
	PortMode oldMode = mode;
	
	// trace
	getTrace() -> mark( this, markChangeMode, newMode, oldMode );
	
	// observer
	_obsAForEachT(PortTailObserverT< ElementType >, Mode, oldMode, newMode);
	
	mode = newMode;
}


// does PortTail have room?
template < typename ElementType >
bool PortTailT< ElementType >::available () {
	return ( getAvailableSpace() > 0 );
}


template < typename ElementType >
void PortTailT< ElementType >::alert() {
	
	if ( memoryList.empty() ) {

		// warning: no object to wake
		if ( mode != ZERO_MODE )
			warning( "PortTail::alert(): no process saved" );
		return;
	}

	Sched* rememberedObject = memoryList.front();

	// trace
	getTrace() -> mark( this, markAlert, rememberedObject );

	// observer
	_obsForEachT( PortTailObserverT< ElementType >, Alert( this, rememberedObject ) );

	// check if rememberedObject really is a sleeping process
	if ( rememberedObject -> getSchedType() == Sched::PROCESS ) {

		Process* p = (Process*)rememberedObject;

		// allow only unscheduled processes to be alerted
		if ( p -> getProcessState() == Process::IDLE ||
			 p -> getProcessState() == Process::CREATED ) {

			// wake up the process
			p -> alertProcess( this );
			
			// update stats
			internalPort -> tailAlerts++; 
		}
		else if ( p -> getProcessState() == Process::TERMINATED ) {
		
			// warning: process terminated
			error( "PortTail::alert(): process already terminated" );
		}
		else if ( p -> getProcessState() == Process::RUNNABLE ) { 
			
			// warning: process scheduled
			warning( "PortTail::alert(): process already scheduled" );
		}
		// no warning for the CURRENT process,
		// events might be running on that stack
	}
	else {
		
		// warning: not a process, no action taken, but continue
		warning( "PortTail::alert(): remembered object is not a process" ); 
	}
	
	Memo::forget( rememberedObject );
}


template < typename ElementType >
TraceProducer* PortTailT< ElementType >::getPartner() {

	// return the current process or event, if one is active
	if ( env -> getCurrentSched() != 0 ) {
		
		return dynamic_cast< TraceProducer* >(env -> getCurrentSched());
	}
	else { // or return the simulation 
		
		return env;
	}
}


template < typename ElementType >
const MarkTypeId PortTailT< ElementType >::baseMarkId = 3200;

template < typename ElementType >
const MarkType PortTailT< ElementType >::markCreate( "create", baseMarkId+1, typeid(PortTailT< ElementType >) );
template < typename ElementType >
const MarkType PortTailT< ElementType >::markDestroy( "destroy", baseMarkId+2, typeid(PortTailT< ElementType >) );

template < typename ElementType >
const MarkType PortTailT< ElementType >::markPut( "put", baseMarkId+3, typeid(PortTailT< ElementType >) );
template < typename ElementType >
const MarkType PortTailT< ElementType >::markCut( "cut", baseMarkId+4, typeid(PortTailT< ElementType >) );
template < typename ElementType >
const MarkType PortTailT< ElementType >::markSplice( "splice", baseMarkId+5, typeid(PortTailT< ElementType >) );
template < typename ElementType >
const MarkType PortTailT< ElementType >::markAlert( "alert", baseMarkId+6, typeid(PortTailT< ElementType >) );

template < typename ElementType >
const MarkType PortTailT< ElementType >::markChangeMode( "changeMode", baseMarkId+7, typeid(PortTailT< ElementType >) );
template < typename ElementType >
const MarkType PortTailT< ElementType >::markChangeMaxCapacity( "changeMaxCapacity", baseMarkId+8, typeid(PortTailT< ElementType >) );

