//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004, 2007 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Timer.h

	\author Ronald Kluth
	<!-- [\author <author>]* -->

	\date created at 2007/04/04

	\brief Declaration of odemx::Timer and odemx::TimerObserver

	\sa Timer.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 2.0
*/


#ifndef ODEMX_TIMER_INCLUDED
#define ODEMX_TIMER_INCLUDED

#include <odemx/base/Sched.h>
#include <odemx/base/Event.h>
#include <odemx/base/Process.h>
#include <odemx/base/Simulation.h>
#include <odemx/util/SimTime.h>
#include <odemx/util/LabeledObject.h>

namespace odemx {
	
	// forward declaration
	class TimerObserver;
		
	/// MAXINT as defined in ODEM
	const int MAXPRIO = 0x8fff; 
	
	/** \class Timer

		\ingroup synch

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief %Timers trigger an event to wake up suspended processes in a model.

		\note %Timer supports Observation.
		\note %Timer supports Trace.

		\sa Process, Event, Simulation

		%Timer is a class provided for convenience. When processes are suspended
		using the \p Process::wait() funtion, a Timer can be used to wake up
		processes if no other Memo object becomes available within a certain
		amount of time. Timers should not be used to implement arbitrary events,
		instead different kinds of events should be derived from the 
		base class Event. By default, Timers have the highest priority of all
		schedulable objects and as such they will be triggered first when
		simulation time reeaches their execution time. 

		(par Example:)
		(include Base1.cpp)

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	
	// inherits scheduling functions and must identify as SchedType TIMER 
	class Timer: public Event,
				 public Memo,
				 public Observable<TimerObserver> {

	public:
		/**
			\brief Construction
			\param sim
				pointer to the Simulation object
			\param l
				label of this object
			\param to
				initial timer observer
		*/
		Timer( Simulation* sim, Label l, TimerObserver* to = 0 );
		
		/// Destruction
		virtual ~Timer();

		/**
			\name Timer Scheduling

			These functions are used to schedule a timeout event
			in simulation time. Timers are kept in the same 
			schedule as processes and other events and they follow the 
			same scheduling rules. However, their default priority is 
			set to maximum.

			@{
		*/
		/**
			\brief Trigger the timeout at simulation time \p now + \p t

			This function schedules the timer at time \p now + \p t before
			all other processes and events at that point in simulation time.
			Scheduling a 
			%Timer at SimTime \p now will not suspend the execution of 
			the current process . The calling process, however, should suspend 
			its execution via Process::wait(), which will automatically 
			register that process with the Memo objects given as parameters. 
		*/
		void setIn( SimTime t );

		/**
			\brief Trigger the timeout at absolute simulation time \p t

			This function schedules the timer at the absolute simulation 
			time \p t before all other processes and events with the same 
			execution time.
			Scheduling a %Timer at SimTime \p now will not suspend the execution of 
			the current process. The calling process, however, should suspend 
			its execution via Process::wait(), which will automatically 
			register that process with the Memo objects given as parameters. 
		*/
		void setAt( SimTime t );
		
		/**
			\brief Reschedule the timeout to simulation time \p now + \p t.

			This function reschedules the timer at time \p now + \p t before
			all other processes with the same exection time 
		*/
		void reset( SimTime t );
		
		/**
			\brief Remove the %Timer from the schedule

			This function removes the timer from the execution list
		*/		
		void stop();

		/**
			\brief Remove the %Timer from the schedule
			\return
				true when listed in schedule

			This function reports whether the time is scheduled or not.
		*/		
		bool isSet();
		//@}
		
		/**
			\brief Register a process to wake with this timer
			\return
				true if sucessfully added \p p to timer memory
			\param p
				process to be remembered

			This function stores process \p p in the timer's memory.
			All the stored processes are rescheduled when the
			timeout is triggered.
			
			\sa removeProcess() 
		*/
		bool registerProcess( Process* p );

		/**
			\brief Remove a registered process from timer's memory
			\return
				true if sucessfully remove \p p
			\param p
				process to be removed

			This function removes process \p p from the timer's memory.

			\sa registerProcess()
		*/
		bool removeProcess( Process* p );
		
		/**
			\brief Reimplemented from Event, schedules all remembered Processes

			This function alerts all processes in the timer's memory.
			These processes are then rescheduled at time \p now. 
		*/
		virtual void eventAction();

		/**
			\brief Check availability of the %Timer object
			\return
				true if this timer is not scheduled

			This function checks the timer's availability.
		*/		
		virtual bool available();	// reimplements Memo::available()
		
		/**
			\brief Get pointer to Trace
			\return
				pointer to Trace object
		*/
		virtual Trace* getTrace() const { return Event::getTrace(); }
		
		
		/**
			\brief get Label of object

			\return Label of object
		*/
		virtual Label getLabel() const { return Event::getLabel(); }
		
		
		// user-defined action(s) ???
		// virtual void handle() {}

	// ========================= Trace Marks ===============================
	
	public:
		/**
			\name Trace MarkTypes

			These MarkTypes and Tags are used to trace Timer calls.
			A TraceConsumer can use these constants to identify additional 
			trace events (besides those from base class Event) sent by Timer.

			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markSet;
		static const MarkType markReset;
		static const MarkType markStop;
		static const MarkType markTimeout;
		//@}

	};



	/** \interface TimerObserver

		\author Ronald Kluth
		<!-- [\author <author>]* -->

		\brief Observer for Timer-specific calls.

		<!-- [\note {notes}]* -->

		\sa Timer, Event

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 2.0
	*/
	
	class TimerObserver: public EventObserver, public MemoObserver {
	public:
		virtual ~TimerObserver() {}
	
		virtual void onCreate(Timer* sender) {}; ///< Construction
		virtual void onDestroy(Timer* sender) {}; ///< Destruction

		/// Activation
		virtual void onSet(Timer* sender, SimTime t) {}; ///< Timeout in \p t 
		virtual void onReset(Timer* sender, SimTime oldTime, SimTime newTime) {}; ///< Trigger at
		virtual void onStop(Timer* sender) {}; ///< Trigger in

		/// Process interaction
		virtual void onRegisterProcess(Timer* sender, Process* p) {}; ///< Process registered
		virtual void onRemoveProcess(Timer* sender, Process* p) {}; ///< Process removed
		virtual void onTimeout(Timer* sender) {}; ///< Timer executed 

	};

}


#endif /*ODEMX_TIMER_INCLUDED*/

