//----------------------------------------------------------------------------
//	Copyright (C) 2002, 2004 Humboldt-Universitaet zu Berlin
//
//	This library is free software; you can redistribute it and/or
//	modify it under the terms of the GNU Lesser General Public
//	License as published by the Free Software Foundation; either
//	version 2.1 of the License, or (at your option) any later version.
//
//	This library is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//	Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public
//	License along with this library; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//----------------------------------------------------------------------------
/**	\file Res.h

	\author Ralf Gerstenberger
	<!-- [\author <author>]* -->

	\date created at 2002/03/21

	\brief Declaration of odemx::Res 

	\sa Res.cpp

	<!-- [detailed description] -->

	<!-- [\todo {todos for this file}]* -->

	\since 1.0
*/

#ifndef ODEMX_RES_INCLUDED
#define ODEMX_RES_INCLUDED

#include <odemx/base/Process.h>
#include <odemx/base/Simulation.h>
#include <odemx/util/Report.h>
#include <odemx/util/Observable.h>
#include <odemx/util/StatisticObject.h>
#include <odemx/statistics/Statistics.h>
#include <odemx/synchronization/Queue.h>

namespace odemx {
	class ResObserver;

	/** \class Res

		\ingroup synch

		\author Ralf Gerstenberger
		<!-- [\author <author>]* -->

		\brief %Res is a double (n>=0 && n<=max number of token) bounded
		token abstract resource.

		\note Res supports Observation
		\note Res supports Trace
		\note Res supports Report

		<!-- [\sa {references to other classes}]* -->

		Res is a double bounded (n>=0 && n<=max number of token) token abstract
		resource for synchronising Process objects. A process is blocked if
		it tries to acquire more token than are left or if it tries to
		release more token than there is room left. It is reactivated when
		enough token are released or acquired. If multiple processes are waiting
		for reactivation process-priority and FIFO-strategy are used to 
		choose the process.
		Res is generally used for strong limited resource modelling or producer
		consumer synchronisation with limited storage capacity.

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class Res : public Observable<ResObserver>,
				public DefLabeledObject,
				public StatisticObject,
				public virtual TraceProducer,
				public virtual ReportProducer
	{
	public:
		/**
			\brief Construction
			\param s
				pointer to Simulation object
			\param l
				label of this object
			\param startTokenNumber
				initial number of token in Res
			\param maxTokenNumber
				max number of token in Res
			\param o
				initial observer
		*/
		Res(Simulation* s, Label l, 
			unsigned int startTokenNumber,
			unsigned int maxTokenNumber,
			ResObserver* o = 0);
		/// Destruction
		~Res();

		/**
			\name Token management
			@{
		*/
		/**
			\brief Acquire \p n token

			If there aren't enough token in Res the current
			process is blocked. If a blocked process is interrupted
			it is reactivated and acquire() returns 0. Otherwise the
			function returns \p n.
		*/
		unsigned int acquire(unsigned int n);

		/**
			\brief Release \p n token

			Returns \p n token to resource. If there is not enough
			room in this resource left an error message is created. 
		*/
		void release(unsigned int n);


		/**
			\brief Add token to resource
			
			Add token to the managed token set.
		*/
		void control(unsigned int n);

		/**
			\brief Remove \p n token from resource

			Remove token from the managed token set. If there are not enough
			token left in the resource, the current process is blocked. When
			a blocked process is interrupted, the attempt to take token is 
			cancelled and the function returns 0.
		*/
		unsigned int unControl(unsigned int n);

		/// Current number of token availiable
		unsigned int getTokenNumber() const {return tokenNumber;}

		/// Maximum number of token
		unsigned int getTokenLimit() const {return maxToken;}
		//@}

		/// Get list of blocked processes
		const std::list<Process*>& getWaitingForAcquire() const {return acquireWait.getList();}

		/**
			\name Statistics

			Statistics

			@{
		*/
		/// Reset statistics
		virtual void reset();
		
		/// Number of takes
		unsigned int getAcquires() const {return users;}

		/// Number of gives
		unsigned int getReleases() const {return providers;}

		/// Init number of token
		unsigned int getInitial() const {return initToken;}
		
		/// Min number of token
		unsigned int getMin() const {return (unsigned int)tokenStatistics.getMin();}

		/// Max number of token
		unsigned int getMax() const {return (unsigned int)tokenStatistics.getMax();}

		/// Average free token
		double getAVFreeToken() const {return tokenStatistics.getMean();}

		/// Average waiting time
		double getAVWaitTime() const;

		//@}

		/// Pointer to trace
		virtual Trace* getTrace() const {return env;}

		/**
			\name Res specific marks
			@{
		*/
		static const MarkTypeId baseMarkId;

		static const MarkType markCreate;
		static const MarkType markDestroy;

		static const MarkType markAcquireFail;
		static const MarkType markAcquireSucceed;
		static const MarkType markReleaseFail;
		static const MarkType markReleaseSucceed;

		static const MarkType markChangeTokenNumber;

		static const TagId baseTagId;

		static const Tag tagCurrent;
		static const Tag tagTokenNumber;
		//@}

		/// Generate report
		virtual void report(Report* r);

		// Implementation
	private:
		/// simulation
		Simulation* env;
		/// current number of tokens
		unsigned int tokenNumber;
		/// max number of tokens
		unsigned int maxToken;

		// Statistics		
		unsigned int initToken; /// init token number		
		Accum tokenStatistics; /// token statistics

		unsigned int users;
		unsigned int providers;
		double sumWaitTime;

		/// process management
		Queue acquireWait;

		// help methods
		/// current process
		Process* getCurrentProcess() {return env->getCurrentProcess();}
		/// trace marks
		void traceARSF(MarkType m, unsigned int n);
	};

	/** \interface ResObserver

		\author Ralf Gerstenberger

		\brief Observer for Res specific events

		<!-- [\note {notes}]* -->

		\sa Res

		<!-- [detailed description] -->

		<!-- [\warning {warnings}]* -->

		<!-- [\todo {todos for this file}]* -->

		<!-- [\bug {bug description}]* -->

		<!-- [\test {testcase description}]* -->

		\since 1.0
	*/
	class ResObserver {
	public:
		virtual ~ResObserver() { }
		/// Construction
		virtual void onCreate(Res* sender) {}
		/// Destruction
		virtual void onDestroy(Res* sender) {}

		/// Blocking acquire of n token
		virtual void onAcquireFail(Res* sender, unsigned int n) {}
		/// Successful acquire of n token
		virtual void onAcquireSucceed(Res* sender, unsigned int n) {}
		/// Blocking release of n token
		virtual void onReleaseFail(Res* sender, unsigned int n) {}
		/// Successful release of n token
		virtual void onReleaseSucceed(Res* sender, unsigned int n) {}

		/// Change of token number
		virtual void onChangeTokenNumber(Res* sender, unsigned int oldTokenNumber,
			unsigned int newTokenNumber) {}
	};
}

#endif

